/*******************************************************************************
* nativeadmin.cpp:
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: nativeadmin.cpp,v 1.4 2002/07/16 22:12:36 sam Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/



//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

//#include <ctype.h>

#include "config.h"
#include "../core/core.h"
#include "../core/network.h"
#include "request.h"
#include "admin.h"
#include "nativeadmin.h"


#include "../core/network.cpp"


/*******************************************************************************
* E_NativeAdmin exception
********************************************************************************
*
*******************************************************************************/
E_NativeAdmin::E_NativeAdmin(const C_String& strMsg) :
                      E_Exception(GEN_ERR, strMsg)
{
}

E_NativeAdmin::E_NativeAdmin(const C_String& strMsg, const E_Exception& e) :
                      E_Exception(GEN_ERR, strMsg, e)
{
}


/*******************************************************************************
* C_NativeAdminSession class
********************************************************************************
* 
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_NativeAdminSession::C_NativeAdminSession(C_Socket* pConnection,
                                           void* pAdmin) :
                            C_AdminSession((C_Admin*)pAdmin),
                            m_strPeerName("[Unknown]"), m_cStream(pConnection)
{
  ASSERT(pConnection);

  m_pConnection = pConnection;

  // Starting mode
  m_iPhase = INIT_PHASE;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NativeAdminSession::OnInit()
{
  m_iPhase = LOGIN_PHASE;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NativeAdminSession::OnClose()
{
  try
  {
    m_pConnection->Close();
  }
  catch(E_Exception e)
  {
    throw E_NativeAdmin("Unable to close connection with " + m_strPeerName, e);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NativeAdminSession::ProcessData()
{
  try
  {
    // First get the data from the network
    C_Buffer<byte> cData(256);
    m_cStream >> cData;
    if(cData.GetSize() <= 0)
      throw E_NativeAdmin("Connection reset by peer");

    unsigned int iPos = 0;
    unsigned int iSize = cData.GetSize();
  
    while(iPos < iSize)
    {
      switch(cData[iPos])
      {
        case '\0':
        {
          HandleMessage();
          break;
        }
        default:
        {
          m_strMessage += (char)cData[iPos];
          break;
        }
      }

      // Go to next char in input
      iPos++;
    }
  }
  catch(E_Stream<C_Socket> e)
  {
    throw E_NativeAdmin("Connection error", e);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NativeAdminSession::SendEvent(const C_Event& cEvent)
{
  try
  {
    C_Message cMessage(cEvent);
    C_String strEvent = cMessage.GetString();

    if(m_cStream.GetState() == STREAM_READY)
    m_cStream << strEvent;
  }
  catch(E_Exception e)
  {
    C_String strPeer = m_pConnection->GetPeerName();
    throw E_NativeAdmin("Unable to send data to "+strPeer, e);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NativeAdminSession::HandleMessage()
{
  printf("Handling message: \"%s\"\n", m_strMessage.GetString());

  C_Message cMessage(m_strMessage);
  s32 iType = cMessage.GetType();
  if(iType != INVALID_TYPE)
  {
    if(iType == REQUEST_TYPE)
    {
      C_Request cRequest = cMessage.GetRequest();
      if(m_iPhase == LOGIN_PHASE)
      {
        C_Answer cAnswer("NativeAdmin");

        if(cRequest.GetCmd() == "login")
        {
          C_String strLogin = cRequest.GetArg("login");
          C_String strPasswd = cRequest.GetArg("password");

          int iRc = Authenticate(strLogin, strPasswd);

          if(!iRc)
          {
            cAnswer.SetStatus(NO_ERR);
            cAnswer.AddMessage("Authentication succeeded");
            m_iPhase = COMMAND_PHASE;
          }
          else
          {
            cAnswer.AddMessage("Authentication failed");
            m_iPhase = LOGIN_PHASE;
          }
        }
        else
        {
          cAnswer.AddMessage("Not authenticated");
        }

        C_Message cMessageAnswer(cAnswer);
        m_cStream << cMessageAnswer.GetString();
      }
      else if(m_iPhase == COMMAND_PHASE)
      {
        C_Answer cAdminAnswer = m_pAdmin->ValidateRequest(this, cRequest);
        switch(cAdminAnswer.GetStatus())
        {
        case ADMIN_WELLFORMED_COMMAND:
          if(cRequest.GetCmd() == "logout")
          {
            throw E_NativeAdmin("logout requested");
          }
          else
          {
            C_Answer cAnswer = m_pAdmin->HandleRequest(cRequest);
            C_Message cMessageAnswer(cAnswer);
            m_cStream << cMessageAnswer.GetString();
          }
          break;
        case ADMIN_COMMAND_NOT_VALID:
        case ADMIN_UNKNOWN_COMMAND:
        case ADMIN_COMMAND_DENIED:
          {
            C_Message cMessageAnswer(cAdminAnswer);
            m_cStream << cMessageAnswer.GetString();
          }
          break;
        case ADMIN_EMPTY_COMMAND:
          {
            C_Answer cAnswer("");
            C_Message cMessageAnswer(cAnswer);
            m_cStream << cMessageAnswer.GetString();
          }
          break;
        default:
          ASSERT(false);
          break;
        }
      }
      else
      {
        ASSERT(false);
      }
    }
    else if(iType == ANSWER_TYPE)
      printf("shouldn't receive any answer => trash\n");
    else /*if(iType == EVENT_TYPE)*/
      printf("shouldn't receive any event => trash\n");
  }
  else
    printf("Invalid message => trash\n");

  m_strMessage = "";
}


/*******************************************************************************
* C_NativeAdmin class
********************************************************************************
* 
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_NativeAdmin::C_NativeAdmin(handle hLog, C_Admin *pAdmin) :
                     C_ConnectionsHandler<C_NativeAdminSession>(hLog, pAdmin)
{
  ASSERT(hLog);
  ASSERT(pAdmin);

  m_hLog = hLog;
  m_pAdmin = pAdmin;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NativeAdmin::PropagateEvent(const C_Event& cEvent)
{
//  m_cEventsLock.Lock();

  // Push the event in the fifo
//  C_Event* pEvent = new C_Event(cEvent);
//  m_cEvents.Add(pEvent);

  // Purge the event fifo
//  while(m_cEvents.Size() >= 1)
//  {
    // get the event
//    C_Event* pEvent = m_cEvents.Remove(0);
//    ASSERT(pEvent);

    // send the event to all the sessions
//    C_HashTable<handle, Session> m_cSessions;

    C_HashTableIterator<handle, C_NativeAdminSession> cIterator =
                                            m_cSessions.CreateIterator();
    while(cIterator.HasNext())
    {
      C_HashTableNode<handle, C_NativeAdminSession>* pNode =
                                            cIterator.GetNext();
      C_NativeAdminSession* pSession = pNode->GetValue();
      pSession->SendEvent(cEvent);
    }

//    delete pEvent;
//  }

//  m_cEventsLock.UnLock();

//  m_cEventsLock.UnLock();
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
int C_NativeAdmin::Init()
{
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);

  int iRc = NO_ERR;

  // Get config
  C_String strDomain = pApp->GetSetting("Telnet.Domain", "Inet4").ToLower();

  int iDomain;
  C_String strDefaultHost;

  if(strDomain == "inet4")
  {
    iDomain = AF_INET;
    strDefaultHost = "0.0.0.0";
  }
#ifdef HAVE_IPV6
  else if(strDomain == "inet6")
  {
    iDomain = AF_INET6;
    strDefaultHost = "0::0";
  }
#endif
  else
  {
    Log(m_hLog, LOG_ERROR, "Unknown domain:\n" + strDomain);
    iRc = GEN_ERR;
  }

  if(!iRc)
  {
    C_String strAddr = pApp->GetSetting("NativeAdmin.LocalAddress",
                                        strDefaultHost);
    C_String strPort = pApp->GetSetting("NativeAdmin.LocalPort", "9998");

    try
    {
      C_ConnectionsHandler<C_NativeAdminSession>::Init(iDomain,
                                                       strAddr, strPort);
    }
    catch(E_Exception e)
    {
      Log(m_hLog, LOG_ERROR,
          "Native administrator initialisation failed:\n" + e.Dump());
      iRc = GEN_ERR;
    }
  }

  if(!iRc)
    Log(m_hLog, LOG_NOTE, "Native administrator initialised");

  return iRc;
}


//------------------------------------------------------------------------------
// Execution
//------------------------------------------------------------------------------
int C_NativeAdmin::Run()
{
  int iRc = NO_ERR;

  try
  {
    C_Thread::Create();
  }
  catch(E_Exception e)
  {
    iRc = GEN_ERR;
  }

  return iRc;
}


//------------------------------------------------------------------------------
// Stop
//------------------------------------------------------------------------------
int C_NativeAdmin::Stop()
{
  int iRc = NO_ERR;
  try
  {
    C_Thread::Stop();
  }
  catch(E_Exception e)
  {
    iRc = GEN_ERR;
  }

  return iRc;
}


//------------------------------------------------------------------------------
// Cleaning
//------------------------------------------------------------------------------
int C_NativeAdmin::Destroy()
{
  int iRc = NO_ERR;

  try
  {
    C_ConnectionsHandler<C_NativeAdminSession>::Destroy();
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR,
        "Error during native administrator destruction:\n"+e.Dump());
    iRc = GEN_ERR;
  }

  if(!iRc)
    Log(m_hLog, LOG_NOTE, "Native administrator destroyed");

  return iRc;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_NativeAdmin::InitWork()
{
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_NativeAdmin::DoWork()
{
  try
  {
    C_ConnectionsHandler<C_NativeAdminSession>::Run();
    Log(m_hLog, LOG_NOTE, "Native administrator started");
  }
  catch(E_Exception e)
  {
    throw E_NativeAdmin("Native administrator startup failed", e);
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_NativeAdmin::StopWork()
{
  try
  {
    C_ConnectionsHandler<C_NativeAdminSession>::Stop();
    Log(m_hLog, LOG_NOTE, "Native administrator stopped");
  }
  catch(E_Exception e)
  { 
    throw E_NativeAdmin("Could not stop native administrator", e);
  }
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_NativeAdmin::CleanWork()
{  
}

