/*
 * *************************************************************************
 *  FileUtils.java
 * **************************************************************************
 *  Copyright © 2015 VLC authors and VideoLAN
 *  Author: Geoffrey Métais
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *  ***************************************************************************
 */

package org.videolan.vlc.util;

import android.annotation.TargetApi;
import android.content.ContentResolver;
import android.content.Context;
import android.content.res.AssetManager;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.ParcelFileDescriptor;
import android.provider.MediaStore;
import android.text.TextUtils;
import android.util.Log;

import org.videolan.libvlc.util.AndroidUtil;
import org.videolan.medialibrary.media.MediaWrapper;
import org.videolan.vlc.VLCApplication;
import org.videolan.vlc.media.MediaUtils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.LongBuffer;
import java.nio.channels.FileChannel;

public class FileUtils {

    public final static String TAG = "VLC/FileUtils";

    /**
     * Size of the chunks that will be hashed in bytes (64 KB)
     */
    private static final int HASH_CHUNK_SIZE = 64 * 1024;

    public interface Callback {
        void onResult(boolean success);
    }

    public static String getFileNameFromPath(String path){
        if (path == null)
            return "";
        int index = path.lastIndexOf('/');
        if (index == path.length()-1) {
            path = path.substring(0, index);
            index = path.lastIndexOf('/');
        }
        if (index > -1)
            return path.substring(index+1);
        else
            return path;
    }

    public static String getParent(String path){
        if (path == null || TextUtils.equals("/", path))
            return path;
        String parentPath = path;
        if (parentPath.endsWith("/"))
            parentPath = parentPath.substring(0, parentPath.length()-1);
        int index = parentPath.lastIndexOf('/');
        if (index > 0){
            parentPath = parentPath.substring(0, index);
        } else if (index == 0)
            parentPath = "/";
        return parentPath;
    }

    /*
     * Convert file:// uri from real path to emulated FS path.
     */
    public static Uri convertLocalUri(Uri uri) {
        if (!TextUtils.equals(uri.getScheme(), "file") || !uri.getPath().startsWith("/sdcard"))
            return uri;
        return Uri.parse(uri.toString().replace("/sdcard", AndroidDevices.EXTERNAL_PUBLIC_DIRECTORY));
    }

    public static String getPathFromURI(Uri contentUri) {
        Cursor cursor = null;
        try {
            final String[] proj = {MediaStore.Images.Media.DATA};
            cursor = VLCApplication.getAppContext().getContentResolver().query(contentUri, proj, null, null, null);
            if (cursor == null || cursor.getCount() == 0)
                return "";
            final int column_index = cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DATA);
            cursor.moveToFirst();
            return cursor.getString(column_index);
        } catch (IllegalArgumentException e) {
                return "";
        } finally {
            if (cursor != null && !cursor.isClosed())
                cursor.close();
        }
    }

    static boolean copyAssetFolder(AssetManager assetManager, String fromAssetPath, String toPath) {
        try {
            final String[] files = assetManager.list(fromAssetPath);
            if (files.length == 0)
                return false;
            new File(toPath).mkdirs();
            boolean res = true;
            for (String file : files)
                if (file.contains("."))
                    res &= copyAsset(assetManager,
                            fromAssetPath + "/" + file,
                            toPath + "/" + file);
                else
                    res &= copyAssetFolder(assetManager,
                            fromAssetPath + "/" + file,
                            toPath + "/" + file);
            return res;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    private static boolean copyAsset(AssetManager assetManager,
                                     String fromAssetPath, String toPath) {
        InputStream in = null;
        OutputStream out = null;
        try {
            in = assetManager.open(fromAssetPath);
            new File(toPath).createNewFile();
            out = new FileOutputStream(toPath);
            copyFile(in, out);
            out.flush();
            return true;
        } catch(Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            Util.close(in);
            Util.close(out);
        }
    }

    private static void copyFile(InputStream in, OutputStream out) throws IOException {
        byte[] buffer = new byte[1024];
        int read;
        while((read = in.read(buffer)) != -1){
          out.write(buffer, 0, read);
        }
    }

    private static boolean copyFile(File src, File dst){
        boolean ret = true;
        if (src.isDirectory()) {
            File[] filesList = src.listFiles();
            dst.mkdirs();
            for (File file : filesList)
                ret &= copyFile(file, new File(dst, file.getName()));
        } else if (src.isFile()) {
            InputStream in = null;
            OutputStream out = null;
            try {
                in = new BufferedInputStream(new FileInputStream(src));
                out = new BufferedOutputStream(new FileOutputStream(dst));

                // Transfer bytes from in to out
                byte[] buf = new byte[1024];
                int len;
                while ((len = in.read(buf)) > 0) {
                    out.write(buf, 0, len);
                }
                return true;
            } catch (IOException ignored) {
            } finally {
                Util.close(in);
                Util.close(out);
            }
            return false;
        }
        return ret;
    }

    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    public static boolean deleteFile (String path){
        boolean deleted = false;
        path = Uri.decode(Strings.removeFileProtocole(path));
        //Delete from Android Medialib, for consistency with device MTP storing and other apps listing content:// media
        if (AndroidUtil.isHoneycombOrLater){
            ContentResolver cr = VLCApplication.getAppContext().getContentResolver();
            try {
                deleted = cr.delete(MediaStore.Files.getContentUri("external"),
                        MediaStore.Files.FileColumns.DATA + "=?", new String[]{path}) > 0;
            } catch (IllegalArgumentException ignored) {} // Can happen on some devices...
        }
        final File file = new File(path);
        if (file.exists())
            deleted |= file.delete();
        return deleted;
    }

    private static void asyncRecursiveDelete(String path, Callback callback) {
        asyncRecursiveDelete(new File(path), callback);
    }

    public static void asyncRecursiveDelete(String path) {
        asyncRecursiveDelete(path, null);
    }

    private static void asyncRecursiveDelete(final File fileOrDirectory, final Callback callback) {
        VLCApplication.runBackground(new Runnable() {
            public void run() {
                if (!fileOrDirectory.exists() || !fileOrDirectory.canWrite())
                    return;
                boolean success = true;
                if (fileOrDirectory.isDirectory()) {
                    for (File child : fileOrDirectory.listFiles())
                        asyncRecursiveDelete(child, null);
                    success = fileOrDirectory.delete();
                } else {
                    success = deleteFile(fileOrDirectory.getPath());
                }
                if (callback != null)
                    callback.onResult(success);
            }
        });
    }

    public static boolean canSave(MediaWrapper mw){
        if (mw == null || mw.getUri() == null)
            return false;
        String scheme = mw.getUri().getScheme();
        if (TextUtils.equals(scheme, "file"))
            return false;
        return TextUtils.equals(scheme, "smb")   ||
                TextUtils.equals(scheme, "nfs")  ||
                TextUtils.equals(scheme, "ftp")  ||
                TextUtils.equals(scheme, "ftps") ||
                TextUtils.equals(scheme, "sftp");
    }

    public static boolean canWrite(Uri uri) {
        if (uri == null)
            return false;
        if (TextUtils.equals("file", uri.getScheme()))
            return canWrite(uri.toString());
        return TextUtils.equals("content", uri.getScheme()) && canWrite(getPathFromURI(uri));

    }

    public static boolean canWrite(String path){
        if (AndroidUtil.isOOrLater || TextUtils.isEmpty(path))
            return false;
        if (path.startsWith("file://"))
            path = path.substring(7);
        if (!path.startsWith("/"))
            return false;
        if (path.startsWith(AndroidDevices.EXTERNAL_PUBLIC_DIRECTORY))
            return true;
        if (AndroidUtil.isKitKatOrLater)
            return false;
        File file = new File(path);
        return (file.exists() && file.canWrite());
    }

    static String computeHash(File file) {
        long size = file.length();
        long chunkSizeForFile = Math.min(HASH_CHUNK_SIZE, size);
        long head, tail;
        FileInputStream fis = null;
        FileChannel fileChannel = null;
        try {
            fis = new FileInputStream(file);
            fileChannel = fis.getChannel();
            head = computeHashForChunk(fileChannel.map(FileChannel.MapMode.READ_ONLY, 0, chunkSizeForFile));

            //Alternate way to calculate tail hash for files over 4GB.
            final ByteBuffer bb = ByteBuffer.allocateDirect((int)chunkSizeForFile);
            int read;
            long position = Math.max(size - HASH_CHUNK_SIZE, 0);
            while ((read = fileChannel.read(bb, position)) > 0) {
                position += read;
            }
            bb.flip();
            tail = computeHashForChunk(bb);
            return String.format("%016x", size + head + tail);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            Util.close(fileChannel);
            Util.close(fis);
        }
    }

    private static long computeHashForChunk(ByteBuffer buffer) {
        final LongBuffer longBuffer = buffer.order(ByteOrder.LITTLE_ENDIAN).asLongBuffer();
        long hash = 0;
        while (longBuffer.hasRemaining())
            hash += longBuffer.get();
        return hash;
    }


    public static Uri getUri(Uri data) {
        Uri uri = data;
        final Context ctx = VLCApplication.getAppContext();
        if (data != null && ctx != null && TextUtils.equals(data.getScheme(), "content")) {
            // Mail-based apps - download the stream to a temporary file and play it
            if ("com.fsck.k9.attachmentprovider".equals(data.getHost()) || "gmail-ls".equals(data.getHost())) {
                InputStream is = null;
                OutputStream os = null;
                Cursor cursor = null;
                try {
                    cursor = ctx.getContentResolver().query(data,
                            new String[]{MediaStore.MediaColumns.DISPLAY_NAME}, null, null, null);
                    if (cursor != null && cursor.moveToFirst()) {
                        String filename = cursor.getString(cursor.getColumnIndex(MediaStore.MediaColumns.DISPLAY_NAME));
                        Log.i(TAG, "Getting file " + filename + " from content:// URI");
                        is = ctx.getContentResolver().openInputStream(data);
                        if (is == null)
                            return data;
                        os = new FileOutputStream(AndroidDevices.EXTERNAL_PUBLIC_DIRECTORY + "/Download/" + filename);
                        final byte[] buffer = new byte[1024];
                        int bytesRead;
                        while ((bytesRead = is.read(buffer)) >= 0) {
                            os.write(buffer, 0, bytesRead);
                        }
                        uri = AndroidUtil.PathToUri(AndroidDevices.EXTERNAL_PUBLIC_DIRECTORY + "/Download/" + filename);
                    }
                } catch (Exception e) {
                    Log.e(TAG, "Couldn't download file from mail URI");
                    return data;
                } finally {
                    Util.close(is);
                    Util.close(os);
                    Util.close(cursor);
                }
            }
            // Media or MMS URI
            else if (TextUtils.equals(data.getAuthority(), "media")){
                uri = MediaUtils.getContentMediaUri(data);
            } else {
                ParcelFileDescriptor inputPFD;
                try {
                    inputPFD = ctx.getContentResolver().openFileDescriptor(data, "r");
                    if (inputPFD == null) return data;
                    if (AndroidUtil.isHoneycombMr1OrLater)
                        uri = AndroidUtil.LocationToUri("fd://" + inputPFD.getFd());
                    else {
                        String fdString = inputPFD.getFileDescriptor().toString();
                        uri = AndroidUtil.LocationToUri("fd://" + fdString.substring(15, fdString.length() - 1));
                    }
//                    Cursor returnCursor =
//                            getContentResolver().query(data, null, null, null, null);
//                    if (returnCursor != null) {
//                        if (returnCursor.getCount() > 0) {
//                            int nameIndex = returnCursor.getColumnIndex(OpenableColumns.DISPLAY_NAME);
//                            if (nameIndex > -1) {
//                                returnCursor.moveToFirst();
//                                title = returnCursor.getString(nameIndex);
//                            }
//                        }
//                        returnCursor.close();
//                    }
                } catch (FileNotFoundException e) {
                    Log.e(TAG, "Couldn't understand the intent");
                    return data;
                } catch (SecurityException e) {
                    Log.e(TAG, "Permission is no longer valid");
                    return data;
                }
            }
        }
        return uri;
    }
}
