#!/usr/bin/env python3
"""
Make backend documentation
"""

import os
import io
import subprocess

marker = "{{< rem autogenerated options"
start = marker + " start"
stop = marker + " stop"
end = ">}}"

def find_backends():
    """Return a list of all backends"""
    return [ x for x in os.listdir("backend") if x not in ("all",) ]

def output_docs(backend, out):
    """Output documentation for backend options to out"""
    out.flush()
    subprocess.check_call(["rclone", "help", "backend", backend], stdout=out)

def output_backend_tool_docs(backend, out):
    """Output documentation for backend tool to out"""
    out.flush()
    subprocess.call(["rclone", "backend", "help", backend], stdout=out, stderr=subprocess.DEVNULL)
    
def alter_doc(backend):
    """Alter the documentation for backend"""
    doc_file = "docs/content/"+backend+".md"
    if not os.path.exists(doc_file):
        raise ValueError("Didn't find doc file %s" % (doc_file,))
    new_file = doc_file+"~new~"
    altered = False
    with open(doc_file, "r") as in_file, open(new_file, "w") as out_file:
        in_docs = False
        for line in in_file:
            if not in_docs:
                if start in line:
                    in_docs = True
                    start_full = (start + "\" - DO NOT EDIT - instead edit fs.RegInfo in backend/%s/%s.go then run make backenddocs\" " + end + "\n") % (backend, backend)
                    out_file.write(start_full)
                    output_docs(backend, out_file)
                    output_backend_tool_docs(backend, out_file)
                    out_file.write(stop+" "+end+"\n")
                    altered = True
            if not in_docs:
                out_file.write(line)
            if in_docs:
                if stop in line:
                    in_docs = False
    os.rename(doc_file, doc_file+"~")
    os.rename(new_file, doc_file)
    if not altered:
        raise ValueError("Didn't find '%s' markers for in %s" % (start, doc_file))

if __name__ == "__main__":
    failed, success = 0, 0
    for backend in find_backends():
        try:
            alter_doc(backend)
        except Exception as e:
            print("Failed adding docs for %s backend: %s" % (backend, e))
            failed += 1
        else:
            success += 1
    print("Added docs for %d backends with %d failures" % (success, failed))
