# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/

# Embedded icon is designed by Igor Kekeljevic (http://www.admiror-ns.co.yu).

# Follows PEP8

from core import models
from lib.reverse_translation import _t


def init():
    global Image
    import Image


def preserve_proportions(x0, y0, x1, y1):
    scaleX = float(x1) / x0
    scaleY = float(y1) / y0
    if scaleX < scaleY:
        y1 = int(round(y0 * scaleX))
    else:
        x1 = int(round(x0 * scaleY))
    return x1, y1


class Action(models.Action):
    """Resize an image"""

    label = _t('Scale')
    author = 'Stani'
    email = 'spe.stani.be@gmail.com'
    init = staticmethod(init)
    version = '0.1'
    tags = [_t('default'), _t('transform'), _t('size')]
    __doc__ = _t('Make the image smaller or bigger')

    def interface(self, fields):
        fields[_t('Canvas Width')] = self.PixelField('800px',
            choices=self.PIXELS_X)
        fields[_t('Canvas Height')] = self.PixelField('800px',
            choices=self.PIXELS_Y)
        fields[_t('Resolution')] = self.DpiField(choices=self.DPIS)
        fields[_t('Constrain Proportions')] = self.BooleanField(True)
        fields[_t('Resample Image')] = self.ImageResampleAutoField('automatic')
        fields[_t('Scale Down Only')] = self.BooleanField(False)

    def apply(self, photo, setting, cache):
        #get info
        info = photo.info
        #dpi
        dpi = info['dpi'] = self.get_field('Resolution', info)
        #size
        x0, y0 = info['size']
        x1 = self.get_field_size('Canvas Width', info, x0, dpi)
        y1 = self.get_field_size('Canvas Height', info, y0, dpi)
        #check if image size has changed
        scaleDownOnly = self.get_field('Scale Down Only', info)
        # check if image size has changed
        if (x0 != x1 or y0 != y1) and \
            (not scaleDownOnly or (scaleDownOnly and (x1 < x0 or y1 < y0))):
            proportions = self.get_field('Constrain Proportions', info)
            method = self.get_field('Resample Image', info)
            #calculate proportions
            if proportions:
                x1, y1 = preserve_proportions(x0, y0, x1, y1)
            #in case of automatic choose antialias for smaller
            #and bicubic for bigger
            if method == 'AUTOMATIC':
                if x1 < x0 and y1 < y0:
                    method = 'ANTIALIAS'
                else:
                    method = 'BICUBIC'
            method = getattr(Image, method)
            #resize image
            photo.resize((x1, y1), method)
        return photo

    icon = \
'x\xda\x01(\x0e\xd7\xf1\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x000\x00\
\x00\x000\x08\x06\x00\x00\x00W\x02\xf9\x87\x00\x00\x00\x04sBIT\x08\x08\x08\
\x08|\x08d\x88\x00\x00\r\xdfIDATh\x81\xed\x9ayT\x94\xe7\xbd\xc7?\xcf\xbb\xcc\
\xc0\xb0\xce\x0c\xc8\x8e"\xc1\xa0\x104A\\\n\x86\xc4#j\xcdUc\xe25&\xa6\x92\
\xe5f\xf76I\x9b\xb4\xe7f\xb3i\x9a\xa5MmN\x1aO\xd6\xf66"\xb8\x15cP\xdc\x80\
\xd4\x18\x03\x91\xb8a\x926\xa1F\x04e\x1d6ad\x9d\x99\xf7\xbd\x7f0\x83\x03\xd8\
\xe85\xd8{z\xce}\xce\xe1\xf0\xcf\xfb>\xbf\xef\xe7y\xbe\xef\xef\xf7{\x9fw\x84\
\xae\xeb\xfc+\x0f\xe9\xffZ\xc0\xf7\x1d\xff\x0fp\xb1\xf1\x1f\xad\x89\x01Wr\
\xfe+\x06\xb0\xe0\x8d\t\xc6G\x1b\xa7\xbc\x1d,\x02\x1b\x1f\xaa\x9dx\x9f\x10B\
\\\x898\xe2J<\xc4\x0fW^w\x95\xd5b\xddn4\x18\'\xf6\xeb=\xd8\xf56Zz\xea\xd7\
\x9d,\xb3=T\xb6T\xef\x19\xcdX\xa3\n \x84\x10?\xab\xb9\xe1\x8e\xf0\xa0\x98weI\
6\xf5\xea\xe7\xe8\xd0\x9ai\xd7\xeai\xd1j\xe9\xed\xeb\xf9\xab\xd6\xce\x92\xc2\
d\xfd\xc4\xa8\xc5\x1c-\x80\x07\xde\x9d\xaa\xc6-\x19\xff\x87\x08\xdf\xb8\x1f\
\xe9B\x13\xddZ\x07\xedZ#-\xda\x19\x9a]\xa79\xa7\xb7\x01\xa0\xb9\xe8vvq\xcf\
\xee\xab\xd8\xa2\x8fB\xf0Q\x01x\xe2\xd4\xec\xf8\x04s\xf2\xeep\xe3\xd8\x04Mw\
\xd1\xa17\xd3\xea\xaa\xc5\xa6\xd5p\xba\xfb$=\xaam\xc4=Z/o\xda\xab\xf8\xe9\
\xbeL\xbd\xf7\xfb\xc4\xfe^\x00B\x08\xf1\xcb\x86\x9b\x97\'\xfa\xa7\xfd1T\x89\
\xf1u\xd0G\xabV\x87\xcdUM\x83\xf3$\x85\xfb+8\xfbU\x00\xd6\x8cz\xc6%\x8d\xbc\
\xdf\xd5\xcfq\xcer\xcb\x8e$\xbd\xea\x9f\x0e\xf0\xfc\xc77*\x81\xd7\x8c}o\x92i\
\xe6]\xa1\xf2X\xba\xf5\x0e\x1a]U4\xb9\xaa\xa8\xec\xfc\x8a\x0fw\xfc\x8d[\xfa\
\xef\xe6\xba\xe0\xeb\xd8\xd6\xb8\x8d\xa31\xdbI\xb9ACV\x86\xce\xa3\xbb8\xe7\
\xeagea\xac\xbe\xed\x9f\x06\xf0\xb3\xca\x8c\xd8\xf8\xf0){\xe2\x8d\xd7M\x0c\
\x93\xe3\xe8\xd4\x9aip}K\xa3\xab\x8a\xb2S\xe5\x1c\xd9\xd3\xc5#\xa1\x8fqu\xe4\
\xd5\x18\x8dF\xecv;\x07\xaa\x0f\x90\xab\xbfM\xe2<;\x81\x96\x91s\xba\xfax\xbd\
\xc1\xc6\x93\x87\xaf\xd3\x1dW\x14\xe0\xa9\xc6\xd9\x8b\xc7\x9b&\xe7F\xab\x13\
\xfc\x03$+=\xba\x9df\xd7i\x1a\x9cU\xe4\x97\xfd\x05\xdf\x8aD\xee\x8d\xbb\x97\
\xf01\xe1\x98L&dY\xc6\xe1p\xd0\xd1\xd1\xc1\x89\xba\x13\xbc\xd5\xbc\x16\xdf\
\x8c\x13\xc4&^\x00\xc2\xc1a\xdd\xc1\xd2\xc2\xb1z\xcd\xa8\x03\x88\xe7\x85\xf4\
\xd4\xc3\xb3\xd7\xc4\xf9$=j\x91\xa3\x84\x8f\xf0\xa7_\xef\xa1]k\xe0d\xd77l\
\xdcq\x909\x1dK\xc9\x1a\x9b\x85\xd5j\xc5h4"I\x03uR\x08\x81\xa6i\xd8\xedvl\
\xcd6\xf2N\xe7\xf1\xf5\xf8\xbd\\\x93\xa9#\xcbC\xe3h\x1a\x9d\xf4\xb1b{\xac^8j\
\x00w\x7f\x1d\x17\x1e\x11\x1aW\x10i\x8c\x9b\x16(\x85b\xc0\x87~z\xe9\xd4\x9a9\
t\xba\x82O\x0b\x9b\xb9\xc7\xff\x01\x92\xa2\x92\x08\x0e\x0eFQ\x06\x8c^SSCcc\
\xa3#55U5\x1a\x8dh\x9aF__\x1f---\x1c8s\x80|\xc3\x7f\x934\xaf\x1b\x7f\xf3\x88\
\x90\xba\xe6\xe0\xb7\xf6\x13<\xb5/Sw~/\x80\xbbj\xa3o\x0c\xf5\x89\xc9\x0f1DY\
\xfcD0\x8a0\xe0\xd4\xfb\xb0\xbb\xda(,\xff\x1c\xfd\xf3\xb1dGd\x13\x19\x16\x89\
\xbf\xbf?\xb2{I+**\x9ceeeo\x7f\xf1\xc5\x17\x15\x8b\x17/\xfeyzzz\x82\xd9<\xa0\
\xd4c\xa9\xca\xbaJ\xde=\xfb&\xc1\x995DO\x18\x19[sr\xd0\xd9\xcd\xbf\xef\x8a\
\xd7k/\x0b`EC\xc83!\xc6\x88\xe7\x83\xa41\x92\xaf\x14\x88\x8c\x82C\xef\xc3\
\xd6e\xa3`\xf7\xd7\xcc\xb2-bN\xe4\x1cBBB\xf0\xf1\xf1A\x92$\x1c\x0e\x07\xa5\
\xa5\xa5\x8d{\xf7\xee}\xa1\xbc\xbc\xfc\x1b@3\x99L\xa6\xec\xec\xec\x07\xa7O\
\x9f\xbe0>>~\xd0R\x9d\x9d\x9d4\xda\x1a\xc9\xab\xcb\xe3T\xe2>\xae\xc9\x00i\
\xa4\xa5\xcej\xfd,/\x8c\xd1\xf7^2\xc0\xa2\xa3\xc2\xa4F\x18\xb7\x9a\rc\xe6\
\x07HV|\x84\x1f\x122N\xfa\xa9jh\xa4|{\x0f+\xa4\xbbI\x8eL\xc6l6\xa3\xaa*B\x08\
\xda\xda\xda(++\xfbt\xd3\xa6M\xaf655\xd9\x80N\xc0\t\xf8\x01AK\x96,\xc9JOO\
\xffIjj\xaa\x8f\xaa\xaa\x00tww\xd3\xdc\xdc\xcc\xc7\xb5\x1f\xb3\xddo=\x93\x17\
\xf4\xe2\xeb?T\x8f\xae\xa3\xe3\xe2\x97\x05\x11\xfa/.\t@d\tk\xe2\xc3\x86\xe3\
\xd33B\xa3L\xf2y\xf1\xbd\xfa9:z:\x98T\xb6\x82\x14=\x8b\x80\x80\x00$IBUU*++]e\
ee\xef\xe5\xe5\xe5m\x01:\x80V7\x80\x0b0\x01f\xc02e\xca\x94\x84\x05\x0b\x16<\
\x97\x91\x91\x11\x17\x1c\x1c\x8c\xa6i8\x1c\x0eZ\xdbZ\xf9K\xe0;\xd4^S\x8c\x18\
\xd6#\xf7\xd8\xe1\x8b\x8f(l\xbc\x9f[t}h\x9a\xbd0\x80\x10cX\xc8J\xcb\x1c\xf9\
\x17\xb3n\n\xf03\x05i\xf4\xe8\xe7\xd0\xd1\x06\xaf\x99R\xbf\x8ci\r+\x91\x85By\
yysII\xc9\x0b\x07\x0e\x1c\xa8\x00\xda\xdc\x7f\x9d@\x1f\xa0\x03\x8a{\x17\xcc\
\x80500\xd0z\xe7\x9dw\xfex\xc6\x8c\x19\xf3\xe3\xe3\xe3\xe9S\xec\x14\x84\xac\
\xa6:\xa0|\x84\x96\xa6\x1a8\xb6\x93\x9a\xbe\xbd\xac\xe23>\xd6u\xfd\xdc\xa5\
\x00\x04\x8c\x1d;v\xe2\xcd\xd97\xaf\xffP\xcd\x9f\x10:\xbb\x8e\xa8\x84\x11\
\x97\x11\xda>\x11\x9f\xbc\x1f\x1c\xdd\xfa\xde\xae_5444x\x89?\x07\xf4\xeb\xba\
\xae\xb9\xe7\x13\x80\xec\xde\x89 \xc0\n\x04\xdfz\xeb\xad\x0b\x92\x17E=\xf6\
\xf7\xd9yj\xb7\xb1u\xc8\xdc\xba\x06_\x97CP\xc5TG\xcc71\xab\xb7m\xdc\xb6\t\
\xa8\xd7u\xbd\xef\xa2\x00\x07\x0f\x1e\x9cf\xb7\xdbw644\x84\xd45\xd4Q`/\xc06\
\xa5\x9c\xe4Y \r\xdb^W\x1f\x9d\xf5;x\xf2\xe8\xc3\x14\x01g\x81.\xc0y\xa1NS\
\x08!\x03F \x00\xb0f\xec\xe0\x01K\x1a\xab\x844\xf4\xc5\xaa\xb7\x0b\x8e\x17\
\xa9\xdc\xd8\xb2\x94\xb9\x91s\x89\x89\x89q\x06\x05\x05\xbd\x9c\x96\x96\xb6z\
\xf8\xbc#\x00\x8e\x1d;\xf6DWW\xd7K.\x97K\xed\xed\xed\xa5\xbd\xbd\x9d\x86\x86\
\x06>k\xfd\x8c\x8f#\n\x98<\xbf\x1f\xbf\xc0a\xcat4G7/\xeeZ\xca\x0b\x1c\xb9\
\xb0x/\x08\xe9\xdauX#3y_1\xb1@FA\x15F$\x14\x04\x82\xc6:\x07\'\xf6\xf8q\x87t7\
\x93\xa3\'\x13\x15\x15Epp0\xaa\xaa\xa2(JQOO\xcf\xf2\x94\x94\x94\xf6\x11\x00\
\xd5\xd5\xd5>mmm[{zz\x16\xe8\xba\x8e\xae\xeb\x83\x0fXgg\'MMMT\x9c\xaa`\x8b\
\x9a\xd77aQ\x971b\xfcHq\xbaF\x89\xaeq\xfb\xf6\x08\xbd\xe5\x1f\x01,n\x16\xd7\
\xaa\x9ao\x81\xbf\x12\x18c\x14~\x18\xf0A\x16*\xe8\x82C\xc7\x1ah\xdci\xd2\xee\
\x8b\xba_J\x8cO$,,\x0c\x7f\x7f\x7f\x14EA\x08\x81\x10\x02I\x92\xce\x00K\x13\
\x12\x12>\x07\xf7;qee\xe5\xb8\x96\x96\x96\xca\xfe\xfe\xfe\x05\x9e\x0b\xdd\
\x17c4\x1a\xb1X,\xf4\xf6\xf6jg\xbe:S\xd8\xb1\xa6\xfb\x95\xcf\xdf\xa0\xfc\xab\
R\xd0\xb4\xa1\xe2\x84\xc4\x1c\x04_\xdel\x13?\xb8\x90\xf8\x95\xad!\xab"\xa5\
\x84\xcf\xc7\x19\x92bb\xe4ID\xcb\x89D*\t\x048#\xd9\xb3\xb3\xd6Y\xf6J\xd3vG\
\xbe\xf3\x8d\xdaS\xb5M\xaa\xaa\x0e\x11/I\x12\x92$!\x84\x88QU\xf5\xd3\xd3\xa7\
O\xff\x18\x06\xb2\x03999]\xb3f\xcdR\xcdf\xf3`\xff\xa2\xb9\xd5\xe9\xba\x8e\
\x10\x82\xf8\xf8\xf8\xaf\x9ey\xe6\x99\xb7\xf4>]\x90G\xe5\xc9&f\xb5\xd5sO\xda\
|\x8c\xbe^\xe7\x0e\x92L8:\x9f\xdc\xdc$\xfe\xeb\xc30\xfdU\x80e\xcdc\xfcC\xa5\
\x90\xdc\x189y\xb1\xbfd\xc6$\x021\n?T\x0cT\xb7\xd4\xf1\xd6\xa6\x92\xe6\xd6\
\xdc\xae\xb5\xfc\x95o\xcep\xa6K\x92$[xx\xf8s\xaa\xaa\x1a\xbd\x17TQ\x14\x14E\
\xa1\xbb\xbb[\xae\xa8\xa8Pccc\x11\x80HMM\xf5\x9d;w\xeeo\xe2\xe3\xe3\x1fNJJ\
\x12\x92$\xe1\xb1\x91\x07\xc0m\xa9\xeam\xdb\xb6\xdd\xb1f\xcd\x1a\x1b\x10\xcd\
\xd5L6\xde\xc6\x93S\xfe\x8d\xe8\xf0q#W<@Xv\x07\x88\x905A"\xfc}\xab\x12\x11\
\x1d(\x85\xe0\'\x82\xf1\x15\x01(\xc2\xc0\xae\xbf\x95\xf0\xc7?\xed=\xe8\xdc\
\xe0\xfa=\xed|\x0b\xb4\xec\xd9\xb3\xe7\x81\x98\x98\x98\x9fJ\x92\xa4\xb8W|\
\xb0\xd6(\x8aBMM\r%%%U6\x9b\xed\xf6\xb5k\xd7\x1e\x16n\x1b\x99\x801\xcb\x96-\
\xbbe\xea\xd4\xa9\xcf\xcf\x981\xc3\xe4\xe3\xe3\x83\xe7\xf9\xf0@\x00\x08!\xce\
\xf5\xf7\xf7?\x98\x91\x91Q\x04\xc4\xe2O\x02\xcb\xb9?a!7N\x9c\x0eB\x02\x83\
\xf0\xc1,Eb\x91"1K\xe1\x98\xa5p\x82\xa41\x04\x8a\x10\xfc\xa4`\xfa\x9d\x0e^.~\
\xdd\xf5Q\xce\xd1\x1c\xb6\xf1g\xa0\xea\xbe\xfb\xee\xb3\xafZ\xb5\xea]\x83\xc1\
p\x93\xc7\x05\x1e\xe1\x9e\xaa]^^\xce\'\x9f|\xf2Q~~\xfe\xd3\xb5\xb5\xb5U\xc0Y\
\x01\x087@\x08\x10\x96\x9c\x9c|\xf5\xc2\x85\x0b\x9f\x999s\xe6\x84\xc8\xc8\
\xc8A\x00\xef\xa1\xeb\xba\xae(\xca;\x99\x99\x99O\xdb\xed\xf6p \x8e\x1fr\x93u\
\x11\xf7\xde\xb0\xc0l\x88\x0e\x88\xc1"Eb\x91\xa2\xdc\xff#\x08\x92\xc2\x08\
\x92B\xa9>\xf7-\xff\xb9\xfe\xe9\xb6\xaau\r\xbf\xe5(\x9f\x01U\xc5\xc5\xc5\xd1\
QQQy\xb2,\x8f\xf3\xb6\x8bg\xd5\xbb\xbb\xbb\xd9\xbd{\xb7\xb3\xb4\xb4\xf4\x9d\
\r\x1b6\xfc\x19h\x06\x9a\x80N\xe1^]\x95\x81Ji\x01\xac~~~\xd6\x95+W>\x94\x96\
\x96\xb6(%%e\xb0\xc3\xf4\x06q\xef\xc8\xe1\xb2\xb2\xb2e\x8f<\xf2\x08\xc0x&q\
\xad\xffm\xca\xe3w\xdc6-25\xeaZ\xacR4V)\n\x8b\x1c\x85E\x8a`{\xcd\x16\x1e_\
\xfb\xe2\xd1\xfe\x1c\xc7\xeb\xb4\xf2\rPs\xec\xd8\xb1\xdbL&\xd3\xaf\x85\x10>\
\x1e\xf1\x06\x83aP|]]\x1d\x05\x05\x05MEEE/\x94\x96\x96~\xc1@\xa1le\xa0]\xe9\
\xf3\x00x*\xa5/^\x95r\xc9\x92%sg\xce\x9c\xf9xzz\xba\xaf\xbf\xff\xb0\x0e\xeb<\
Hkgg\xe7]\x19\x19\x19G\x80q\xf8\x11/\xad\x14\x0f\xfc({vF\xf6\xe4lB\xe4h\x8c\
\xba?\xbf\xd8\xff\x9c\xb6\xf1\x9d=\x1b\xd8\xca\x16\xe0\xd4\xfc\xf9\xf3\xdb^{\
\xed\xb5\xdf\xc9\xb2|\x9b\xc7\xeb\xb2,c0\x18\x063\xcf\xa1C\x87(***\xdd\xb8q\
\xe3\xabMMMMn\xe1\xed\x80\x1dp\xe8\xba\xae\x89a\xab\xea])-\x9co\xbe\x9e\xcd\
\xc8\xc8\x18\x1f\x1b\x1b{!\x00\x00\x97\xcb\xe5z%%%\xe5w@\x0c\x10\xc7B\x16^\
\x9f\x9d\x92}\x7f\xfa\x9d\xf2\xf3\xf9\xaf\x9f=\x91S\xf7\x1a\x87\x06,\xb3a\
\xc3\x06KZZ\xdazI\x92\xae\xf6\xcc\xe1\xbd\xea\xbd\xbd\xbd\xec\xda\xb5\xcb\
\xb5\x7f\xff\xfe?\xe4\xe6\xe6n\xe6|sx\x16\xe8\xc6\xab\xd2\x8f\xa8\xc4B\x08\t\
\xf0XjH\xf35m\xda\xb4\xf9\xa9\xa9\xa9\xc8\xb2<\xf8P{A\xa0\xebz\xc9\x96-[\xee\
]\xbdz\xb5\x0f0\x9ed\xaee:Yl\'\x97f\xfe\x0e\x9c:r\xe4\xc8\xfc\xa0\xa0\xa0\
\xdf\xcb\xb2l\x02\x90ey\xf0AUU\x95\xfa\xfaz\xf2\xf3\xf3\x9bKJJ~\xb5\x7f\xff\
\xfec\x0cm\x0e{\x01\xcd\xbb\xd2\xff\xa3f\xee\x82\xcd\xd7\xd2\xa5K\x7f8c\xc6\
\x8cGg\xcd\x9ae\xf4\xbc]]\xe0\xcc\xf6Luu\xf5\x9dYYY\xd5@\xac{!liiiMyyy/\x18\
\x0c\x86{<\xf7xD+\x8a\x82$I\x1c>|\x98\x1d;v\x94o\xde\xbc\xf9\xd7\xdf\xd5\x1c\
\x0e\xd1\xfa]od\xc3\x9b/\xc0\x9c\x9a\x9a:\xd1m\xa9\xd8\xab\xae\xbaj\xc4.\xb8\
\xff\xf7\xf7\xf4\xf4\xfc|\xd2\xa4I\xeb\x00CNNNXzz\xfazUUS<\x0f\xaa\xb7\xf8\
\xfe\xfe~\n\x0b\x0b\xb5}\xfb\xf6\xfdi\xdd\xbau\x1b\x18\xb0J\x1b\x17i\x0e/\n\
\xe0\x16\xe4m)\x0b`1\x9b\xcd\xa1+V\xacxl\xea\xd4\xa9s\xd2\xd3\xd31\x18\x0c\
\xc3\xef\xf1\x14\xbfMN\xa7\xb3\xd0`0\xbc)\x84\x08\x04\x86\x14%EQ\xb0\xd9ll\
\xda\xb4\xa9\xad\xb8\xb8\xf8\xc5}\xfb\xf6\x1dr\x8b\x1e\xcc2\x80\xeb;\x9b\xc3\
K9\x95\x18f\xa9`\x06v#h\xd9\xb2e\x8b\xa6M\x9b\xf6HVV\x96\xc1j\xb5\x0e\x8a\
\xf7\x06\xf1\x1e\x1e\xd1\x1e\xcb\x1c?~\x9c\xad[\xb7\x1e\xde\xbau\xeb+555\xf5\
\x0c\xcd2\x17\xb4\xcce\x01x\t\xf2X*\xd0\ra\x9e>}zRVV\xd6\xb3\x99\x99\x99QII\
\x178\x00u\x83x\x8bw:\x9d\xec\xdc\xb9S+))Y\x9f\x9b\x9b\xbb\xde\xe9tzV\xdd\
\x93e\x1c\x97zr\xad\\\xfc\x92\xf3C\xd7u\x97\x10\xa2\x97\x81\xf7\\\x07\xd0W^^\
\xee8y\xf2\xe4C\x1d\x1d\x1d?ill\xbc\xe1\xfa\xeb\xaf\x1f,\xfd0`\x19o\xf1\xad\
\xad\xadl\xde\xbc\xf9lQQ\xd1K%%%\xe5\x0c\xac\xb8\xe7\xfd\xb9\x97\x8bXf\xf8\
\xb8\xac\xb3Q\xb7\xa5\x14\x86e\xa9\xdbo\xbf}IZZ\xda\x83s\xe7\xceU-\x16\x0b\
\xb2,\x0f\n\x97e\x99/\xbf\xfc\x92\xfc\xfc\xfc\x8a\x0f>\xf8\xe0\xa5S\xa7N\xd5\
r>\xcb\\\xb2eF\x05\xc0\x0bD\x06|\x18\xb0\x94\x050\xcf\x9c93%++\xeb\xd9y\xf3\
\xe6\x85O\x9c8q\xd02{\xf7\xee\xd5\x8b\x8b\x8b\xf3rrr\xdew:\x9d\x1d@\x0b\x97a\
\x99Q\x05pCx\xb2T\x00\xee\xa3\x93\xb0\xb0\xb0\xf0\xe5\xcb\x97?1o\xde\xbc\x8c\
\xc4\xc4D\xf2\xf3\xf3;JJJ^.**\xfa\x8c\x01\xcb\xb41\x90e\xfe\xd7\x96\x19u\x00\
7\x84\xb7\xa5\x06\xb3\xd4\x8a\x15+\x96\x85\x85\x85M+((\xf8\xcd\xc9\x93\'O3\
\xb40\xf5]\x8eeF\xc4\x1e\xadodn\x08\x89\xa1\x96\nb \xfdv3\xb40]\xb6eF\xc4\
\x1d\xed\xcf\xacnK\x19\x18(|\x01\x0c\x00\xf4p\x99Y\xe6\xa2\xf1\xae\xc4wb/K\
\xa9\x0c\xec\x8a\x03w\xfb;\xea\xb1\xae\xe4\xafU\xbc\xbf\xce\x8f\xe6\xaa\x0f\
\x89\xf1\xaf\xfes\x9b\xff\x01  4SXO{W\x00\x00\x00\x00IEND\xaeB`\x82\xc6\xde\
\x1f\x84'
