///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/undo/UndoManager.h>

#if !defined(Q_WS_MAC)
	#include <GL/glu.h>
#endif

#include "BondsDataChannel.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(BondsDataChannel, DataChannel)
DEFINE_FLAGS_REFERENCE_FIELD(BondsDataChannel, FloatController, "BondsWidth", PROPERTY_FIELD_ALWAYS_DEEP_COPY, _bondWidth)
DEFINE_PROPERTY_FIELD(BondsDataChannel, "FlatBonds", _flatBonds)
SET_PROPERTY_FIELD_LABEL(BondsDataChannel, _bondWidth, "Bonds display width")
SET_PROPERTY_FIELD_LABEL(BondsDataChannel, _flatBonds, "Render flat bonds")
SET_PROPERTY_FIELD_UNITS(BondsDataChannel, _bondWidth, WorldParameterUnit)

/******************************************************************************
* Serialization constructor.
******************************************************************************/
BondsDataChannel::BondsDataChannel(bool isLoading) : DataChannel(isLoading)
{
	init(isLoading);
}

/******************************************************************************
* Special constructor to create a standard data channel.
******************************************************************************/
BondsDataChannel::BondsDataChannel(DataChannelIdentifier which, size_t maxBonds) : DataChannel(which, maxBonds)
{
	init(false);
}

/******************************************************************************
* Initializes the object. This is called from the constructors.
******************************************************************************/
void BondsDataChannel::init(bool isLoading)
{
	INIT_PROPERTY_FIELD(BondsDataChannel, _bondWidth);
	INIT_PROPERTY_FIELD(BondsDataChannel, _flatBonds);
	if(!isLoading) {
		_flatBonds = false;
		_bondWidth = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_bondWidth->setCurrentValue(0.3);
		clearBonds();
	}
}

/******************************************************************************
* Copies the contents from the given source channel into this channel.
* Atoms for which the bit in the given mask is set are skipped.
******************************************************************************/
void BondsDataChannel::filterCopy(DataChannel* source, const dynamic_bitset<>& mask)
{
	OVITO_ASSERT(source->size() == mask.size());
	OVITO_ASSERT(perAtomSize() == source->perAtomSize());
	size_t oldAtomsCount = source->size();

	// This is used to remap atom indices.
	QVector<int> remap(source->size());
	int counter = 0;
	for(size_t i = 0; i < oldAtomsCount; i++) {
		if(!mask.test(i))
			remap[i] = counter++;
		else
			remap[i] = -1;
	}
	OVITO_ASSERT(counter == size());

	const int* src = source->constDataInt();
	int* dst = dataInt();
	for(size_t i = 0; i < oldAtomsCount; i++, src += _componentCount) {
		if(!mask.test(i)) {
			for(size_t c = 0; c < _componentCount; ++c) {
				if(src[c] >= 0)
					dst[c] = remap[src[c]];
				else
					dst[c] = -1;
			}
			dst += _componentCount;
		}
	}
}

/******************************************************************************
* Render the displacement vectors into the viewport.
******************************************************************************/
void BondsDataChannel::render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode)
{
	AffineTransformation inverseTM = vp->inverseWorldMatrix() * vp->inverseViewMatrix();
	Vector3 viewDir = Normalize(inverseTM * Vector3(0,0,-1));
	Point3 viewPoint = inverseTM * Point3(0,0,0);
	OVITO_ASSERT(viewDir != NULL_VECTOR);

	if(_bondWidth && _bondWidth->getValueAtTime(time) > 0.0) {
		if(_flatBonds)
			renderBondsFlat(time, atoms, vp->isPerspectiveProjection(), viewDir, viewPoint, vp);
		else
			renderBondsShaded(time, atoms, vp->isPerspectiveProjection(), viewDir, viewPoint, vp);
	}
	else
		renderBondsLines(time, atoms, vp->isPerspectiveProjection(), viewDir, viewPoint, vp);
}

/******************************************************************************
* Renders the channel's contents in high-quality mode to an offscreen buffer.
******************************************************************************/
void BondsDataChannel::renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
	Matrix4 tm;
	glGetFloatTypev(GL_MODELVIEW_MATRIX, tm.data());
	Matrix4 inverseTM = tm.inverse();
	Vector3 viewDir = Normalize(inverseTM * Vector3(0,0,-1));
	Point3 viewPoint = inverseTM * Point3(0,0,0);

	if(_bondWidth && _bondWidth->getValueAtTime(time) > 0.0) {
		if(_flatBonds)
			renderBondsFlat(time, atoms, view.isPerspective, viewDir, viewPoint, NULL);
		else
			renderBondsShaded(time, atoms, view.isPerspective, viewDir, viewPoint, NULL);
	}
	else
		renderBondsLines(time, atoms, view.isPerspective, viewDir, viewPoint, NULL);
}

/******************************************************************************
* Renders the atomic bonds as thin lines.
******************************************************************************/
void BondsDataChannel::renderBondsLines(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp)
{
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return;

	const Point3* p = posChannel->constDataPoint3();
	const Point3* pend = p + posChannel->size();
	const int* b = this->constDataInt();

	TimeInterval interval;
	const QVector<Color> atomColors = atoms->getAtomColors(time, interval);
	QVector<Color>::const_iterator color1 = atomColors.constBegin();

	glPushAttrib(GL_LIGHTING_BIT);
	glDisable(GL_LIGHTING);

	const AffineTransformation simCell = atoms->simulationCell()->cellMatrix();
	const AffineTransformation simCellInverse = simCell.inverse();
	const array<bool,3> pbc = atoms->simulationCell()->periodicity();

	glBegin(GL_LINES);
	for(; p != pend; ++p, ++color1) {
		for(size_t j = componentCount(); j != 0; --j, ++b) {
			if(*b < 0 || *b > atoms->atomsCount()) continue;
			const Point3& p2 = posChannel->getPoint3(*b);
			Vector3 delta = p2 - *p;

			// Use minimum image convention to handle periodic boundary conditions.
			Vector3 reduced = simCellInverse * delta;
			for(size_t k=0; k<3; k++) {
				if(pbc[k]) {
					while(reduced[k] < -0.5) {
						reduced[k] += 1.0;
						delta += simCell.column(k);
					}
					while(reduced[k] > 0.5) {
						reduced[k] -= 1.0;
						delta -= simCell.column(k);
					}
				}
			}

			glColor3v(color1->constData());
			glVertex3v(p->constData());
			glVertex3(p->X + delta.X*0.5, p->Y + delta.Y*0.5, p->Z + delta.Z*0.5);
		}
	}
	glEnd();

	glPopAttrib();
}

/******************************************************************************
* Renders the atomic bonds as shaded cylinders.
******************************************************************************/
void BondsDataChannel::renderBondsShaded(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp)
{
	// Get the position channel.
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return;

	// Setup material
	glPushAttrib(GL_LIGHTING_BIT);
	Color color(1,1,1);
	Window3DMaterial material;
	if(vp) {
		material.diffuse = color;
		vp->setMaterialCount(1);
		vp->setMaterial(0, &material);
		glEnable(GL_LIGHTING);
		vp->realizeMaterial(0);
	}
	else {
		glEnable(GL_LIGHTING);
		float ambient[4] = { 0.1f, 0.1f, 0.1f, 1 };
		glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);
		float diffuse[4] = { color.r, color.g, color.b, 1 };
		glMaterialfv(GL_FRONT, GL_DIFFUSE, diffuse);
		float specular[4] = { 0.1f, 0.1f, 0.1f, 1 };
		glMaterialfv(GL_FRONT, GL_SPECULAR, specular);
		float emission[4] = { 0, 0, 0, 1 };
		glMaterialfv(GL_FRONT, GL_EMISSION, emission);
		glMaterialf(GL_FRONT, GL_SHININESS, 0);
		glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, 0);
	}

	// Enable color tracking.
	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT, GL_DIFFUSE);

	const AffineTransformation simCell = atoms->simulationCell()->cellMatrix();
	const AffineTransformation simCellInverse = simCell.inverse();
	const array<bool,3> pbc = atoms->simulationCell()->periodicity();

	TimeInterval interval;
	const QVector<Color> atomColors = atoms->getAtomColors(time, interval);
	QVector<Color>::const_iterator color1 = atomColors.constBegin();

	const Point3* p = posChannel->constDataPoint3();
	const int* b = this->constDataInt();
	FloatType bondWidth = 0.2f;
	if(_bondWidth)
		bondWidth = _bondWidth->getValueAtTime(time) * 0.5;
	GLUquadricObj* cylinderQuadric = gluNewQuadric();
	GLUquadricObj* diskQuadric = gluNewQuadric();
	gluQuadricNormals(cylinderQuadric, GLU_SMOOTH);
	for(size_t i = size(); i != 0; --i, ++p, ++color1) {
		for(size_t j = componentCount(); j != 0; --j, ++b) {
			if(*b < 0 || *b >= atoms->atomsCount()) continue;
			AffineTransformation tm;
			const Point3& p2 = posChannel->getPoint3(*b);
			Vector3 delta = (p2 - *p);

			// Use minimum image convention to handle periodic boundary conditions.
			bool renderCap = false;
			Vector3 reduced = simCellInverse * delta;
			for(size_t k=0; k<3; k++) {
				if(pbc[k]) {
					while(reduced[k] < -0.5) {
						reduced[k] += 1.0;
						delta += simCell.column(k);
						renderCap = true;
					}
					while(reduced[k] > 0.5) {
						reduced[k] -= 1.0;
						delta -= simCell.column(k);
						renderCap = true;
					}
				}
			}

			// Construct cylinder orientation matrix
			tm.setTranslation(*p - ORIGIN);
			tm.column(2) = delta;
			tm.column(1) = CrossProduct(tm.column(2), Vector3(0,0,1));
			if(tm.column(1) == NULL_VECTOR)
				tm.column(1) = CrossProduct(tm.column(2), Vector3(1,0,0));
			tm.column(0) = Normalize(CrossProduct(tm.column(2), tm.column(1)));
			tm.column(1) = Normalize(tm.column(1));
			FloatType length = Length(delta);
			tm.column(2) /= length;

			glColor3v(color1->constData());
			glPushMatrix();
			glMultMatrix(Matrix4(tm).constData());
			glScalef(-1, 1, 1);
			gluCylinder(cylinderQuadric, bondWidth, bondWidth, length*0.5, 16, 1);
			if(renderCap) {
				glTranslatef(0,0,length*0.5);
				gluDisk(diskQuadric, 0, bondWidth, 16, 1);
			}
			glPopMatrix();
		}
	}
	gluDeleteQuadric(cylinderQuadric);
	gluDeleteQuadric(diskQuadric);
	glPopAttrib();
	if(vp)
		vp->setMaterialCount(0);

	// Disable color tracking.
	glDisable(GL_COLOR_MATERIAL);
}

/******************************************************************************
* Renders the atomic bonds as flat rectangles.
******************************************************************************/
void BondsDataChannel::renderBondsFlat(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp)
{
	// Get the position channel.
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return;

	// Setup material
	glPushAttrib(GL_LIGHTING_BIT);
	Color color(1,1,1);
	Window3DMaterial material;
	if(vp) {
		material.diffuse = color;
		vp->setMaterialCount(1);
		vp->setMaterial(0, &material);
		vp->setRenderingColor(color);
	}
	else {
		glColor3v(color.constData());
	}
	glDisable(GL_LIGHTING);

	const AffineTransformation simCell = atoms->simulationCell()->cellMatrix();
	const AffineTransformation simCellInverse = simCell.inverse();
	const array<bool,3> pbc = atoms->simulationCell()->periodicity();

	TimeInterval interval;
	const QVector<Color> atomColors = atoms->getAtomColors(time, interval);
	QVector<Color>::const_iterator color1 = atomColors.constBegin();

	const Point3* p = posChannel->constDataPoint3();
	const int* b = this->constDataInt();
	FloatType bondWidth = 0.2f;
	if(_bondWidth)
		bondWidth = _bondWidth->getValueAtTime(time) * 0.5;
	glBegin(GL_QUADS);
	for(size_t i = size(); i != 0; --i, ++p, ++color1) {
		for(size_t j = componentCount(); j != 0; --j, ++b) {
			if(*b < 0 || *b > atoms->atomsCount()) continue;
			AffineTransformation tm;

			const Point3& p2 = posChannel->getPoint3(*b);
			Vector3 delta = (p2 - *p);

			// Use minimum image convention to handle periodic boundary conditions.
			Vector3 reduced = simCellInverse * delta;
			for(size_t k=0; k<3; k++) {
				if(pbc[k]) {
					while(reduced[k] < -0.5) {
						reduced[k] += 1.0;
						delta += simCell.column(k);
					}
					while(reduced[k] > 0.5) {
						reduced[k] -= 1.0;
						delta -= simCell.column(k);
					}
				}
			}

			if(isPerspective)
				viewDir = *p - viewPoint;
			tm.column(2) = delta;
			tm.setTranslation(*p - ORIGIN);
			tm.column(1) = CrossProduct(tm.column(2), viewDir);
			FloatType length = Length(delta);
			if(tm.column(1) != NULL_VECTOR) {
				tm.column(0) = CrossProduct(tm.column(2), tm.column(1));
				tm.column(0) = Normalize(tm.column(0));
				tm.column(1) = Normalize(tm.column(1));
				tm.column(2) = Normalize(tm.column(2));
				glColor3v(color1->constData());
				glVertex3v((tm * Point3(0,  bondWidth, 0)).constData());
				glVertex3v((tm * Point3(0, -bondWidth, 0)).constData());
				glVertex3v((tm * Point3(0, -bondWidth, length*0.5)).constData());
				glVertex3v((tm * Point3(0,  bondWidth, length*0.5)).constData());
			}
		}
	}
	glEnd();
	glPopAttrib();
	if(vp)
		vp->setMaterialCount(0);
}


/******************************************************************************
* Lets the channel clear all its internal caches.
******************************************************************************/
void BondsDataChannel::clearCaches()
{
	DataChannel::clearCaches();
}

/******************************************************************************
* Removes all bonds stored in this channel.
******************************************************************************/
void BondsDataChannel::clearBonds()
{
	fill(dataInt(), dataInt() + (componentCount() * size()), -1);
}

IMPLEMENT_PLUGIN_CLASS(BondsDataChannelEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void BondsDataChannelEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Bonds"), rolloutParams);

	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
	layout->setSpacing(0);
#endif
	layout->setColumnStretch(1, 1);

	BooleanPropertyUI* showBondsUI = new BooleanPropertyUI(this, "isVisible", tr("Show bonds"));
	layout->addWidget(showBondsUI->checkBox(), 0, 0, 1, 2);

	BooleanPropertyUI* flatBondsUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(BondsDataChannel, _flatBonds));
	layout->addWidget(flatBondsUI->checkBox(), 1, 0, 1, 2);

	FloatControllerUI* bondsWidthUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(BondsDataChannel, _bondWidth));
	layout->addWidget(bondsWidthUI->label(), 2, 0);
	layout->addLayout(bondsWidthUI->createFieldLayout(), 2, 1);
	bondsWidthUI->setMinValue(0);
}

};	// End of namespace AtomViz
