/*jshint node:true*/
module.exports = function( grunt ) {

"use strict";

var banner,
	umdStart,
	umdMiddle,
	umdEnd,
	umdStandardDefine,
	umdAdditionalDefine,
	umdLocalizationDefine;

banner = "/*!\n" +
	" * jQuery Validation Plugin v<%= pkg.version %>\n" +
	" *\n" +
	" * <%= pkg.homepage %>\n" +
	" *\n" +
	" * Copyright (c) <%= grunt.template.today('yyyy') %> <%= pkg.author.name %>\n" +
	" * Released under the <%= _.map(pkg.licenses, 'type').join(', ') %> license\n" +
	" */\n";

// Define UMD wrapper variables

umdStart = "(function( factory ) {\n" +
	"\tif ( typeof define === \"function\" && define.amd ) {\n";

umdMiddle = "\t} else if (typeof module === \"object\" && module.exports) {\n" +
	"\t\tmodule.exports = factory( require( \"jquery\" ) );\n" +
	"\t} else {\n" +
	"\t\tfactory( jQuery );\n" +
	"\t}\n" +
	"}(function( $ ) {\n\n";

umdEnd = "return $;" +
	"\n}));";

umdStandardDefine = "\t\tdefine( [\"jquery\"], factory );\n";
umdAdditionalDefine = "\t\tdefine( [\"jquery\", \"./jquery.validate\"], factory );\n";
umdLocalizationDefine = "\t\tdefine( [\"jquery\", \"../jquery.validate\"], factory );\n";

grunt.initConfig( {
	pkg: grunt.file.readJSON( "package.json" ),
	concat: {

		// Used to copy to dist folder
		dist: {
			options: {
				banner: banner +
					umdStart +
					umdStandardDefine +
					umdMiddle,
				footer: umdEnd
			},
			files: {
				"dist/jquery.validate.js": [ "src/core.js", "src/*.js" ]
			}
		},
		extra: {
			options: {
				banner: banner +
					umdStart +
					umdAdditionalDefine +
					umdMiddle,
				footer: umdEnd
			},
			files: {
				"dist/additional-methods.js": [ "src/additional/additional.js", "src/additional/*.js" ]
			}
		}
	},
	uglify: {
		options: {
			preserveComments: false,
			banner: "/*! <%= pkg.title || pkg.name %> - v<%= pkg.version %> - " +
				"<%= grunt.template.today('m/d/yyyy') %>\n" +
				" * <%= pkg.homepage %>\n" +
				" * Copyright (c) <%= grunt.template.today('yyyy') %> <%= pkg.author.name %>;" +
				" Licensed <%= _.map(pkg.licenses, 'type').join(', ') %> */\n"
		},
		dist: {
			files: {
				"dist/additional-methods.min.js": "dist/additional-methods.js",
				"dist/jquery.validate.min.js": "dist/jquery.validate.js"
			}
		},
		all: {
			expand: true,
			cwd: "dist/localization",
			src: "**/*.js",
			dest: "dist/localization",
			ext: ".min.js"
		}
	},
	compress: {
		dist: {
			options: {
				mode: "zip",
				level: 1,
				archive: "dist/<%= pkg.name %>-<%= pkg.version %>.zip",
				pretty: true
			},
			src: [
				"changelog.txt",
				"demo/**/*.*",
				"dist/**/*.js",
				"Gruntfile.js",
				"lib/**/*.*",
				"package.json",
				"LICENSE.md",
				"README.md",
				"src/**/*.*",
				"test/**/*.*"
			]
		}
	},
	qunit: {
		files: "test/index.html",
		options: {
			puppeteer: {
				args: [
					"--headless",
					"--disable-web-security",
					"--allow-file-access-from-files"
				]
			},
			timeout: 10000
		}
	},
	jshint: {
		options: {
			jshintrc: true
		},
		core: {
			src: "src/**/*.js"
		},
		test: {
			src: [ "test/*.js", "test/additional/*.js" ]
		},
		grunt: {
			src: "Gruntfile.js"
		}
	},
	watch: {
		options: {
			atBegin: true
		},
		src: {
			files: "<%= jshint.core.src %>",
			tasks: [
				"concat"
			]
		}
	},
	jscs: {
		all: [ "<%= jshint.core.src %>", "<%= jshint.test.src %>", "<%= jshint.grunt.src %>" ]
	},
	copy: {
		dist: {
			options: {

				// Append UMD wrapper
				process: function( content ) {
					return umdStart + umdLocalizationDefine + umdMiddle + content + umdEnd;
				}
			},
			src: "src/localization/*",
			dest: "dist/localization",
			expand: true,
			flatten: true,
			filter: "isFile"
		}
	},
	replace: {
		dist: {
			src: "dist/**/*.min.js",
			overwrite: true,
			replacements: [
				{
					from: "./jquery.validate",
					to: "./jquery.validate.min"
				}
			]
		}
	},

	// Generate the sub-resource integrity hashes of the distribution files
	sri: {
		options: {
			algorithms: [ "sha256", "sha384", "sha512" ],

			// The target json file
			dest: "dist/jquery-validation-sri.json",

			// Stringify the JSON output in a pretty format
			pretty: true
		},

		all: {
			src: [
				"dist/jquery.validate.{min.js,js}",
				"dist/additional-methods.{min.js,js}",
				"dist/localization/*.js"
			]
		}
	}
} );

grunt.loadNpmTasks( "grunt-contrib-jshint" );
grunt.loadNpmTasks( "grunt-contrib-qunit" );
grunt.loadNpmTasks( "grunt-contrib-uglify" );
grunt.loadNpmTasks( "grunt-contrib-concat" );
grunt.loadNpmTasks( "grunt-contrib-compress" );
grunt.loadNpmTasks( "grunt-contrib-watch" );
grunt.loadNpmTasks( "grunt-jscs" );
grunt.loadNpmTasks( "grunt-contrib-copy" );
grunt.loadNpmTasks( "grunt-text-replace" );
grunt.loadNpmTasks( "grunt-sri" );

grunt.registerTask( "default", [ "concat", "copy", "jscs", "jshint", "qunit" ] );
grunt.registerTask( "release", [ "default", "uglify", "replace", "compress", "sri" ] );
grunt.registerTask( "start", [ "concat", "watch" ] );

};
