/* nexp_iface.c
  
   Tcl commands that deal with network interfaces are
   implemented here. The iface() array is also created here.

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "util-tcl.h"

/*
 * FIXME: add support fo iface(<name>,ip6).
 */
static int
create_iface_callback(const struct intf_entry *entry, void *arg)
{
    Tcl_Interp *interp = arg;
    Tcl_Obj *obj, *flag;
    char buffer[256], addr[20];
    char *iftype, *s;
    uint32_t mask;
    struct addr a;

    /*
     * Create iface(<ifname>,flags).
     */
    obj = Tcl_NewListObj(0, NULL);

    if (entry->intf_flags & INTF_FLAG_UP) {
	flag = Tcl_NewStringObj("UP", -1);
	Tcl_ListObjAppendElement(NULL, obj, flag);
    }

    if (entry->intf_flags & INTF_FLAG_LOOPBACK) {
	flag = Tcl_NewStringObj("LOOPBACK", -1);
	Tcl_ListObjAppendElement(NULL, obj, flag);
    }

    if (entry->intf_flags & INTF_FLAG_POINTOPOINT) {
	flag = Tcl_NewStringObj("POINTTOPOINT", -1);
	Tcl_ListObjAppendElement(NULL, obj, flag);
    }

    if (entry->intf_flags & INTF_FLAG_NOARP) {
	flag = Tcl_NewStringObj("NOARP", -1);
	Tcl_ListObjAppendElement(NULL, obj, flag);
    }

    if (entry->intf_flags & INTF_FLAG_BROADCAST) {
	flag = Tcl_NewStringObj("BROADCAST", -1);
	Tcl_ListObjAppendElement(NULL, obj, flag);
    }

    if (entry->intf_flags & INTF_FLAG_MULTICAST) {
	flag = Tcl_NewStringObj("MULTICAST", -1);
	Tcl_ListObjAppendElement(NULL, obj, flag);
    }

    snprintf(buffer, sizeof(buffer), "%s,flags", entry->intf_name);
    Tcl_SetVar2Ex(interp, "iface", buffer, obj, 0);

    /*
     * Create iface(<ifname>,mtu).
     */
    obj = Tcl_NewIntObj(entry->intf_mtu);
    snprintf(buffer, sizeof(buffer), "%s,mtu", entry->intf_name);
    Tcl_SetVar2Ex(interp, "iface", buffer, obj, 0);

    /*
     * Create iface(<ifname>,type).
     */
    if (entry->intf_type == INTF_TYPE_OTHER)
	iftype = "Other";
    else if (entry->intf_type == INTF_TYPE_ETH)
	iftype = "Ethernet";
    else if (entry->intf_type == INTF_TYPE_LOOPBACK)
	iftype = "Loopback";
    else if (entry->intf_type == INTF_TYPE_TUN)
	iftype = "Virtual/Internal";
    else
	/* Shouldn't happen */
	iftype = "Other";

    obj = Tcl_NewStringObj(iftype, strlen(iftype) );
    snprintf(buffer, sizeof(buffer), "%s,type", entry->intf_name);
    Tcl_SetVar2Ex(interp, "iface", buffer, obj, 0);

    /*
     * Create iface(<ifname>,hw_addr) (only if type is Ethernet)
     */
    if (entry->intf_type == INTF_TYPE_ETH) {
	addr_ntop(&entry->intf_link_addr, addr, sizeof(addr) );
	obj = Tcl_NewStringObj(addr, strlen(addr) );
	snprintf(buffer, sizeof(buffer), "%s,hw_addr", entry->intf_name);
	Tcl_SetVar2Ex(interp, "iface", buffer, obj, 0);
    }

    /*
     * Create iface(<ifname>,ip) and iface(<ifname>,netmask), but only if the
     * address type is IPv4. Don't worry about aliases for now.
     *
     * FIXME - worry about aliases.
     */
    if (entry->intf_addr.addr_type == ADDR_TYPE_IP) {
	addr_ntop(&entry->intf_addr, addr, sizeof(addr) );
	/*
	 * This test is necessary because addr_ntop() will return a string
	 * with a "/nn" suffix if the number of bits in the netmask is
	 * not 32. In this case we need to remove the trailing "/nn".
	 */
	if ( (s = strchr(addr, '/') ) )
	    *s = '\0';
	obj = Tcl_NewStringObj(addr, strlen(addr) );
	snprintf(buffer, sizeof(buffer), "%s,ip", entry->intf_name);
	Tcl_SetVar2Ex(interp, "iface", buffer, obj, 0);

	addr_btom(entry->intf_addr.addr_bits, &mask, IP_ADDR_LEN);
	a.addr_type = ADDR_TYPE_IP;
	a.addr_bits = IP_ADDR_BITS;
	a.addr_ip = mask;

	addr_ntop(&a, addr, sizeof(addr) );
	obj = Tcl_NewStringObj(addr, strlen(addr) );
	snprintf(buffer, sizeof(buffer), "%s,netmask", entry->intf_name);
	Tcl_SetVar2Ex(interp, "iface", buffer, obj, 0);
    }

    return 0;
}

static void
create_iface_array(Tcl_Interp *interp)
{
    intf_t *i;

    i = intf_open();
    if (!i)
	warn("can't obtain handle to network interface");
    else {
	intf_loop(i, create_iface_callback, interp);
	intf_close(i);
    }
}

/********************************************************************
 *				iflist                              *
 ********************************************************************/

static int
iface_list_callback(const struct intf_entry *entry, void *arg)
{
    Tcl_Obj *obj = arg, *ifname;

    ifname = Tcl_NewStringObj(entry->intf_name, -1);
    Tcl_ListObjAppendElement(NULL, obj, ifname);

    return 0;
}

static int
NExp_IfListCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	       const char **argv)
{
    intf_t *i;
    Tcl_Obj *obj;

    if (argc > 2) {
	nexp_error(interp, "usage: %s [-refresh]", argv[0]);
	return TCL_ERROR;
    }

    if (argc == 2) {
	if (strcmp(argv[1], "-refresh") && strcmp(argv[1], "refresh") ) {
	    nexp_error(interp, "usage: iflist [-refresh]");
	    return TCL_ERROR;
	} else {
	    Tcl_UnsetVar(interp, "iface", TCL_LEAVE_ERR_MSG);
	    create_iface_array(interp);
	}
    }

    obj = Tcl_NewListObj(0, NULL);

    i = intf_open();
    if (!i)
	warn("can't obtain handle to network interface");
    else {
	intf_loop(i, iface_list_callback, obj);
	intf_close(i);
    }

    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *                               outif                              *
 ********************************************************************/

static int
NExp_OutIfCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	      const char **argv)
{
    intf_t *i;
    struct intf_entry iface;
    struct addr dest;

    if (argc != 2) {
	nexp_error(interp, "must have one arg: dest. hostname or IP addr");
	return TCL_ERROR;
    }

    if (addr_pton(argv[1], &dest) == -1) {
	nexp_error(interp, "can't resolve '%s'", argv[1]);
	return TCL_ERROR;
    }

    i = intf_open();
    if (i == NULL) {
	nexp_error(interp, "intf_open(): can't get interface handle");
	return TCL_ERROR;
    }

    /*
     * Don't forget to do this or bad things will happen (dumbnet(3)
     * says that this is required for all intf_get_*() functions.)
     */
    iface.intf_len = sizeof(struct intf_entry);

    if (intf_get_dst(i, &iface, &dest) == 0)
	Tcl_SetResult(interp, iface.intf_name, TCL_VOLATILE);
    else {
	nexp_error(interp, "intf_get_dst() failed: %s", strerror(errno) );
	intf_close(i);
	return TCL_ERROR;
    }

    intf_close(i);

    return TCL_OK;
}

static struct nexp_cmd_data cmd_data[] = {
    {"iflist", NULL, NExp_IfListCmd, 0, 0},
    {"outif", NULL, NExp_OutIfCmd, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_iflist_cmd(Tcl_Interp *interp)
{
    create_iface_array(interp);

    nexp_create_commands(interp, cmd_data);
}
