/*
 For more information, please see: http://software.sci.utah.edu

 The MIT License

 Copyright (c) 2012 Scientific Computing and Imaging Institute,
 University of Utah.


 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included
 in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 DEALINGS IN THE SOFTWARE.
 */

/**
 \brief A Lua class proxy for IO's dataset class.
 */

#include "Controller/Controller.h"
#include "3rdParty/LUA/lua.hpp"
#include "IO/IOManager.h"
#include "IO/FileBackedDataset.h"
#include "IO/uvfDataset.h"

#include <vector>

#include "../LuaScripting.h"
#include "../LuaClassRegistration.h"

#include "LuaTuvokTypes.h"

#include "LuaDatasetProxy.h"

using namespace std;

namespace tuvok
{

LuaDatasetProxy::LuaDatasetProxy()
    : mDS(NULL)
    , mDatasetType(Unknown)
{
  mReg = NULL;
}

LuaDatasetProxy::~LuaDatasetProxy()
{
  if (mReg != NULL)
    delete mReg;
  mDS = NULL;
}

void LuaDatasetProxy::bind(Dataset* ds, shared_ptr<LuaScripting> ss)
{
  if (mReg == NULL)
    throw LuaError("Unable to bind dataset, no class registration available.");

  mReg->clearProxyFunctions();

  mDS = ds;
  if (ds != NULL)
  {
    // Register dataset functions using ds.
    string id;

    id = mReg->functionProxy(ds, &Dataset::GetDomainSize,
                             "getDomainSize", "", false);
    id = mReg->functionProxy(ds, &Dataset::GetRange,
                             "getRange", "", false);
    id = mReg->functionProxy(ds, &Dataset::GetLODLevelCount,
                             "getLODLevelCount", "", false);
    id = mReg->functionProxy(ds, &Dataset::GetNumberOfTimesteps,
                             "getNumberOfTimesteps", "", false);
    id = mReg->functionProxy(ds, &Dataset::GetMeshes,
                             "getMeshes", "", false);
    id = mReg->functionProxy(ds, &Dataset::GetBitWidth,
                             "getBitWidth", "", false);
    id = mReg->functionProxy(ds, &Dataset::Get1DHistogram,
                             "get1DHistogram", "", false);
    id = mReg->functionProxy(ds, &Dataset::Get2DHistogram,
                             "get2DHistogram", "", false);
    id = mReg->functionProxy(ds, &Dataset::Name,
                             "name", "Dataset descriptive name.", false);
    id = mReg->functionProxy(ds, &Dataset::SaveRescaleFactors,
                             "saveRescaleFactors", "", false);
    id = mReg->functionProxy(ds, &Dataset::GetRescaleFactors,
                             "getRescaleFactors", "", false);
    // We do NOT want the return values from GetMeshes stuck in the provenance
    // system (Okay, so the provenance system doesn't store return values, just
    // function parameters. But it's best to be safe).
    ss->setProvenanceExempt(id);

    // Attempt to cast the dataset to a file backed dataset.
    FileBackedDataset* fileDataset = dynamic_cast<FileBackedDataset*>(ds);
    if (fileDataset != NULL)
    {
      id = mReg->functionProxy(fileDataset, &FileBackedDataset::Filename,
                               "fullpath", "Full path to the dataset.", false);
    }

    UVFDataset* uvfDataset = dynamic_cast<UVFDataset*>(ds);
    if (uvfDataset != NULL)
    {
      mDatasetType = UVF;

      id = mReg->functionProxy(uvfDataset, &UVFDataset::RemoveMesh,
                               "removeMesh", "", true);
      id = mReg->functionProxy(uvfDataset, &UVFDataset::AppendMesh,
                               "appendMesh", "", false);
      id = mReg->functionProxy(uvfDataset, &UVFDataset::GeometryTransformToFile,
                               "geomTransformToFile", "", false);
      ss->setProvenanceExempt(id);
    }

    /// @todo Expose 1D/2D histogram? Currently, it is being transfered
    ///       via shared_ptr. If lua wants to interpret this, the histogram
    ///       will need to be placed in terms that lua can understand.
    ///       Two approaches:
    ///       1) Add Grid1D to the LuaStrictStack.
    ///       2) Create a Histogram1D and Histogram2D proxy.
    ///       
    ///       The second solution would be more efficient, since there wouldn't
    ///       be any time spent converting datatypes to and from Lua (and with
    ///       histograms, that time wouldn't be negligible).
  }

}

void LuaDatasetProxy::defineLuaInterface(
    LuaClassRegistration<LuaDatasetProxy>& reg,
    LuaDatasetProxy* me,
    LuaScripting*)
{
  me->mReg = new LuaClassRegistration<LuaDatasetProxy>(reg);

  string id;

  // Register our functions
  id = reg.function(&LuaDatasetProxy::getDatasetType, "getDSType", "", false);
  id = reg.function(&LuaDatasetProxy::proxyGetMetadata, "getMetadata", "",
                    false);
}


std::vector<std::pair<std::string, std::string>>
LuaDatasetProxy::proxyGetMetadata()
{
  return mDS->GetMetadata();
}

} /* namespace tuvok */
