/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "main.h"
#include "cdw_processwin.h"
#include "gettext.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_regex_dispatch.h"
#include "cdw_cdrecord_regex.h"
#include "cdw_disc.h"
#include "cdw_task.h"
#include "cdw_debug.h"
#include "cdw_config.h"
#include "cdw_drive.h"
#include "cdw_utils.h"




/*

Output of "wodim -toc -vv dev=/dev/scd0
Profile: 0x0015 (DVD-R/DL sequential recording)
Profile: 0x0016 (DVD-R/DL layer jump recording)
Profile: 0x002B (DVD+R/DL)
Profile: 0x001B (DVD+R)
Profile: 0x001A (DVD+RW)
Profile: 0x0014 (DVD-RW sequential recording)
Profile: 0x0013 (DVD-RW restricted overwrite)
Profile: 0x0012 (DVD-RAM)
Profile: 0x0011 (DVD-R sequential recording)
Profile: 0x0010 (DVD-ROM)
Profile: 0x000A (CD-RW) (current)
Profile: 0x0009 (CD-R)
Profile: 0x0008 (CD-ROM)
Profile: 0x0002 (Removable disk)
*/


struct {
	int wodim_id;
	cdw_disc_type_t cdw_id; } wodim_profiles[] = {

	{ 0x00,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x01,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x02,   CDW_DISC_TYPE_UNKNOWN }, /* Removable disk */
	{ 0x03,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x04,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x05,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x06,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x07,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x08,   CDW_CD_ROM },
	{ 0x09,   CDW_CD_R },
	{ 0x0A,   CDW_CD_RW },
	{ 0x0B,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x0C,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x0D,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x0E,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x0F,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x10,   CDW_DVD_ROM },
	{ 0x11,   CDW_DVD_R },             /* DVD+R Seq */
	{ 0x12,   CDW_DISC_TYPE_UNKNOWN }, /* DVD-RAM */
	{ 0x13,   CDW_DVD_RW_RES },
	{ 0x14,   CDW_DVD_RW_SEQ },
	{ 0x15,   CDW_DISC_TYPE_UNKNOWN }, /* DVD-R/DL sequential recording */
	{ 0x16,   CDW_DISC_TYPE_UNKNOWN }, /* DVD-R/DL layer jump recording */
	{ 0x17,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x18,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x19,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x1A,   CDW_DVD_RWP },
	{ 0x1B,   CDW_DVD_RP },
	{ 0x1C,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x1D,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x1E,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x1F,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x20,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x21,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x22,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x23,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x24,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x25,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x26,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x27,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x28,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x29,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x2A,   CDW_DISC_TYPE_UNKNOWN },
	{ 0x2B,   CDW_DVD_RP_DL }};






/*
 cdrecord can recognize following disc types (cdrtools-2.01.01, cdrecord/scsi_mmc.c):
"Reserved"
"Non -removable Disk"
"Removable Disk"
"MO Erasable"
"MO Write Once"
"AS-MO"
"CD-ROM"
"CD-R"
"CD-RW"
"DVD-ROM"
"DVD-R sequential recording"
"DVD-RAM"
"DVD-RW restricted overwrite"
"DVD-RW sequential recording"
"DVD-R/DL sequential recording"
"DVD-R/DL layer jump recording"
"DVD-RW/DL"
"DVD+RW"
"DVD+R"
"DDCD-ROM"
"DDCD-R"
"DDCD-RW"
"DVD+RW/DL"
"DVD+R/DL"
"BD-ROM"
"BD-R sequential recording"
"BD-R random recording"
"BD-RE"
"HD DVD-ROM"
"HD DVD-R"
"HD DVD-RAM"
"HD DVD-RW"
"HD DVD-R/DL"
"HD DVD-RW/DL"
"No standard Profile"

some of them are supported by cdw as well. Labels of supported discs are
captured in cdw_cdrecord_handle_disc_type_{new|old}_style() */



extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];

/* time captured at the beginning of process */
extern time_t time0;

extern cdw_config_t global_config;
extern cdw_task_t *thread_task;


static int previous_percent = 0;
static bool cdw_cdrecord_burning_performed = false;

/* functions used for stdout output handling */
static int cdw_cdrecord_handle_blanking_time(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_image_writing(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_direct_writing(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_fixating(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_blanking_not_supported(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_drive_max_speed(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_drive_default_speed(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_current_write_speed(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_msinfo_session_sectors(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_disc_type_old_style(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_disc_type_new_style(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_prcap_write_speed_table(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_reload_disk_and_hit_cr(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_toc_beginning(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_write_error_after(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_supported_disc_modes(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_phys_size(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_atip_start_of_lead_out(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_rzone_size(regex_t *regex, regmatch_t *matches);

/* functions used for stderr output handling */
static int cdw_cdrecord_handle_msinfo_cannot_get_next_writable(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_msinfo_cannot_read_session_offset(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_device_or_resource_busy(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_cannot_read_toc(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_bad_option(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_cannot_open_device(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_no_such_file_or_dir(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_badly_placed_option(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_no_disk(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_cannot_send_cue_sheet(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_non_ricoh_based_drive(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_cannot_open_next_track(regex_t *regex, regmatch_t *matches);
static int cdw_cdrecord_handle_data_may_not_fit(regex_t *regex, regmatch_t *matches);


static cdw_regex_t stdout_regex[] = {
	{ "direct writing", 1001,
	  /* "Track 01:  190 of  257 MB written (fifo 100%) [buf 100%]  10.3x." */
	  "Track ([0-9]+): ([ ]*)([0-9]+) of ([ ]*)([0-9]+) MB written \\(fifo([ ]*)([0-9]+)%\\)([ ]+)\\[buf([ ]*)([0-9]+)%\\]([ ]+)([0-9]+)\\.([0-9]+)x",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_direct_writing },

	{ "image writing", 1002,
	  /* "Track 02: 304 MB written (fifo 100%) [buf 100%]   4.2x." */
	  "Track ([0-9]+): ([ ]*)([0-9]+) MB written \\(fifo([ ]*)([0-9]+)\\)([ ]+)\\[buf([ ]*)([0-9]+)%\\][([ ]+)([0-9]+)\\.([0-9]+)x",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_image_writing },

	{ "blanking time", 1003,
	  /* "Blanking time:   27.819s" - printed after successful blanking of CD (and DVD?) */
	  "Blanking time:([ ]+)([0-9]*).([0-9]*)s",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_blanking_time },

	{ "fixating", 1004,
	  /* "Fixating..." */
	  "Fixating...",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_fixating },

	{ "disc type, old style", 1005,
	  /* "Current: 0x000A (CD-RW)" - wodim reports type of disc currently
	     in drive (for both CD and DVD); older versions of wodim (like
	     1.1.0) don't print anything after hex number, so we need to rely
	     only on hex number; 0x00 is explicitly put into regex to make
	     sure that format of information (new/old) is correctly recognized */
	  "Current:[ ]+0x00([0-9a-fA-F]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_disc_type_old_style },

	{ "disc type, new style", 1006,
	  /* "Current: CD-RW" - cdrecord reports type of disc currently in drive (for both CD and DVD) */
	  /* "[a-z ]*" is for purposes of DVD-RW discs, like "Current: DVD-RW restricted overwrite" */
	  /* ([CDVRWOMDL+/\-]+) on purpose don't include '0x' to distinguish long format from short */
	  "Current:[ ]+([CDVRWOMDL+/-]+[a-z ]*)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_disc_type_new_style },

	{ "write speeds", 1007,
	  /* "Write speed # 0:  1764 kB/s CLV/PCAV (CD  10x, DVD  1x, BD  0x)" */
	  "Write speed # ([0-9]+):([ a-zA-Z0-9/]+)([(])CD([ ]+)([0-9]*)x,[ ]+DVD([ ]+)([0-9]*)x",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_prcap_write_speed_table },

	{ "session info", 1008,
	  /* "122176,129604" - information about last_sess_start and
	    next_sess_start sectors, comma is significant; this is result
	    of running cdrecord -msinfo */
	  "^([0-9]+),([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_msinfo_session_sectors },

	{ "drive default speed", 1009,
	  /* "Drive default speed: xy" */
	  "Drive default speed([ ]*):([ ]*)([0-9]*)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_drive_default_speed },

	{ "current write speed", 1010,
	  /* "Current write speed: 22160 kB/s (CD 125x, DVD 16x, BD  4x)" */
	  "Current write speed:([ ]+)([0-9]+)([ ]+)kB/s([ ]+)\\(CD([ ]+)([0-9]+)x,([ ]+)DVD([ ]+)([0-9]+)x,([ ]+)BD([ ]+)([0-9]+)x\\)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_current_write_speed },

	{ "drive max speed", 1011,
	  /* "Drive max speed    : xy" */
	  "Drive max speed([ ]*):([ ]*)([0-9]*)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_drive_max_speed },

	{ "does not support blanking", 1012,
	  /* "this media does not support blanking" */
	  "this media does not support blanking",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_blanking_not_supported },

	{ "write error after X bytes", 1013,
	  /* "write track data: error after X bytes" */
	  /* I've seen it when attempting to burn data to non-empty DVD+R */
	  "write track data: error after",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_write_error_after },

	{ "TOC first, last", 1014,
	  /* "first: 1 last 3" */
	  /* part of information about TOC: TOC exists, the numbers are numbers of tracks */
	  "first: ([0-9]+) last ([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_toc_beginning },

	{ "reload disc and hit Enter", 1015,
	  /* "Re-load disk and hit <CR>" */
	  /* may happen when cdrecord reloads tray, but the tray can't be closed,
	     e.g. because this is a drive in notebook */
	  "load disk and hit",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_reload_disk_and_hit_cr },

	{ "supported modes", 1016,
	  /* "Supported modes: TAO PACKET SAO SAO/R96P SAO/R96R RAW/R16 RAW/R96P RAW/R96R" */
	  /* information about disc modes supported by given drive;
	     may be useful when selecting drive mode for burning */
	  "Supported modes: (.+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_supported_disc_modes },

	{ "phys size", 1017,
	  /* "phys size:...    2295104"
	     Only for DVDs;
	     Capacity of a disc in sectors - total numbers of sectors that can
	     be stored on a disc; parameter not displayed by wodim 1.1.11 */
	  "phys size:... +([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_phys_size },

	{ "atip start of lead out", 1018,
	  /* "  ATIP start of lead out: 359849 (79:59/74)"
	     let's consider start of lead out to be total disc capacity */
	  "ATIP start of lead out: ([0-9]+) ",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_atip_start_of_lead_out },

	{ "rzone size", 1019,
	  /* "rzone size:...    2295104"

	     Only for DVDs;
	     RZone is something similar to a track on a CD. I didn't
	     investigate it too much, so we are in a gray area here. */
	  "rzone size: +([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_rzone_size },

	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};




static cdw_regex_t stderr_regex[] = {
	{ "device or resource busy", 2001,
	  /* "Device or resource busy" - system message when cdrecord tries to access busy CD. */
	  "Device or resource busy",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_NOSUB,
	  cdw_cdrecord_handle_device_or_resource_busy },

	{ "incompatible format", 2002,
	  /* cannot write medium - incompatible format */
	  "cannot write medium - incompatible format",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_blanking_not_supported },

	{ "cannot blank disc", 2003,
	  /* Cannot blank disk */
	  "Cannot blank disk",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_blanking_not_supported },

	{ "cannot get next writable address", 2004,
	  /* "Cannot get next writable address" - printed for 'complete' discs */
	  /* "read" and "first" were added during tests with dvd+r dl */
	  "(Cannot (get|read) (next|first) writable address)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_msinfo_cannot_get_next_writable },

	{ "cannot read session offset", 2005,
	  /* "Cannot read session offset" - printed for blank discs */
	  "(Cannot read session offset)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_msinfo_cannot_read_session_offset },

	{ "permission denied", 2006,
	  /* "cdrecord: Permission denied. Cannot open '/dev/sg0'. Cannot open or use SCSI driver." */
	  /* TODO: test this for wodim too */
	  "(cdrecord|wodim): Permission denied. Cannot open",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_cannot_open_device },

	{ "no such file or directory", 2007,
	  /* "cdrecord: No such file or directory. Cannot open 'some'." */
	  /* this is similar to "No such file or directory. Invalid node"
	     from mkisofs - probably incorrect entry in "other cdrecord options" */
	  /* TODO: test this for wodim too */
	  "(cdrecord|wodim): No such file or directory. Cannot open",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_no_such_file_or_dir },

	{ "cannot send CUE sheet", 2008,
	  /* /home/acerion/bin/cdrecord: Cannot send CUE sheet. */
	  "annot send CUE sheet",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_cannot_send_cue_sheet },

	{ "non Ricoh based drive", 2009,
	  /* "Cannot blank DVD+RW media with non Ricoh based drive" */
	  "Cannot blank DVD[+]RW media with non Ricoh based drive",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_non_ricoh_based_drive },

	{ "cannot open next track", 2010,
	  /* "wodim: Cannot open new session"
	     "cdrecord: Cannot open next track" */
	  /* printed e.g. when cdrecord is called for non-appendable medium */
	  "annot open (new|next) (session|track)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_cannot_open_next_track },

	{ "cannot read TOC header", 2011,
	  /* Cannot read TOC header */
	  /* printed for empty discs; may be useful when wodim can't recognize
	     non-empty DVD disc */
	  "annot read TOC header",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_cannot_read_toc },

	{ "data may not fit", 2012,
	  /* "WARNING: Data may not fit on current disk" */
	  /* Printed when size of selected files or iso image exceeds
	     size space available on optical disc */
	  "WARNING: Data may not fit on current disk",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_data_may_not_fit },

	{ "bad option", 2013,
	  /* "cdrecord: Bad Option: <option name>. */
	  /* printed when there is incorrect option in command line */
	  "Bad Option: (.+).",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_bad_option },

	{ "badly placed option", 2014,
	  /* "wodim: Badly placed option. Global options must be before any track." */
	  /* printed when there are some incorrect options forwarded to wodim,
	     most probably some incorrect string in "other cdrecord options" field;
	     this may also mean that some error was made when creating command string
	     in cdrecord_interface.c */
	  "Badly placed option",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_badly_placed_option },

	{ "no disk / wrong disk", 2015,
	  /* "wodim: No disk / Wrong disk!" */
	  /* to invoke this message do following:
	     - put closed CD-RW disc in drive
	     - issue command "wodim -msinfo dev=<dev>
	     - wodim will open and will try to close drive tray, but you have to hold the tray open
	     - wodim will prompt you with "close tray and press enter" message
	     - press enter, but keep tray open
	     - wodim will try to read disc, will fail to do it, and will print "no disk / wrong disk" message" */
	  "No disk / Wrong disk",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_cdrecord_handle_no_disk },

	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};





/**
   \brief Do some initialization and call cdw_regex_prepare_regexes_table() for stdout regexes table
*/
void cdrecord_stdout_regexp_prepare(void)
{
	cdw_cdrecord_burning_performed = false;
	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stdout_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stdout\n");
	}
	return;
}





/**
   \brief Do some initialization and call cdw_regex_execute_regexes_table()
   for stdout regexes table and stdout pipe buffer.
*/
void cdrecord_stdout_regexp_execute(void)
{
	stdout_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stdout_regex, stdout_pipe_buffer);

	return;
}





/**
   \brief Call cdw_regex_clean_up_regexes_table() for stdout regexes table
*/
void cdrecord_stdout_regexp_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stdout_regex);
	return;
}





/**
   \brief Call cdw_regex_prepare_regexes_table() for stderr regexes table
*/
void cdrecord_stderr_regexp_prepare(void)
{
	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stderr_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stderr\n");
	}
	return;
}





/**
   \brief Do some initialization and call cdw_regex_execute_regexes_table()
   for stderr regexes table and stderr pipe buffer.
*/
void cdrecord_stderr_regexp_execute(void)
{
	stderr_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stderr_regex, stderr_pipe_buffer);

	return;
}





/**
   \brief Call cdw_regex_clean_up_regexes_table() for stderr regexes table
*/
void cdrecord_stderr_regexp_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stderr_regex);
	return;
}





int cdw_cdrecord_handle_image_writing(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information about writing image:
	             1        2      3                  4        5       6        7           8      9           10    11         12
	   "Track ([0-9]+): ([ ]*)([0-9]+) MB written ([(])fifo([ ]*)([0-9]+)\\)([ ]+)\\[buf([ ]*)([0-9]+)%\\)[([ ]+)([0-9]+)\\.([0-9]+)x */
	cdw_regex_assert_subex_number(regex->re_nsub, 12);

	int fifo = 0;
	int done_size = 0;
	int tracknum = 0;
	int speed_decimal = 0, speed_fract = 0;

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		if (i != 1 && i != 3 && i != 6 && i != 11 && i != 12) {
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE + 1];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

 		if (i == 1) {
			tracknum = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, tracknum = \"%s\" -> %d\n", i, submatch, tracknum);
		} else if (i == 3) {
			done_size = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, done size = \"%s\" -> %d\n", i, submatch, done_size);
		} else if (i == 6) {
			fifo = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, fifo = \"%s\" -> %d\n", i, submatch, fifo);
		} else if (i == 11) {
			speed_decimal = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed decimal = \"%s\" -> %d\n", i, submatch, speed_decimal);
		} else if (i == 12) {
			speed_fract = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed fractional = \"%s\" -> %d\n", i, submatch, speed_fract);
		} else {
			;
		}
	}

	if (tracknum > 1) {
		/* sometimes cdrecord prints tracknum = 02 when burning iso
		   image to non-virgin DVD+RW */
		if (thread_task->disc->type == CDW_DVD_RWP) {
			cdw_sdm ("WARNING: cdrecord reports tracknum = %d during burning image to DVD+RW, resetting to 1\n", tracknum);
		} else {
			cdw_sdm ("WARNING: cdrecord reports tracknum = %d during burning image to !DVD+RW (%d), resetting to 1\n", tracknum, thread_task->disc->type);
		}
		tracknum = 1;
	}

	cdw_processwin_display_fifo_and_speed(fifo, -1, -1);

	char text_info2_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window,
	   Track is cd data track, %d is its number */
	snprintf(text_info2_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("Track: %d"), tracknum);
	cdw_processwin_display_sub_info(text_info2_string);

	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window,
	   %d is amount of data already written to CD */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1,_("%d MB"), done_size);
	cdw_processwin_display_progress_conditional(done_size, 0, current_value_string);

	cdw_processwin_wrefresh();

	return 0;
}





int cdw_cdrecord_handle_direct_writing(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information on direct writing:
	             1        2      3          4      5                         6       7         8           9      10         11    12         13
	   "Track ([0-9]+): ([ ]*)([0-9]+) of ([ ]*)([0-9]+) MB written \\(fifo([ ]*)([0-9]+)%\\)([ ]+)\\[buf([ ]*)([0-9]+)%\\]([ ]+)([0-9]+)\\.([0-9]+)x" */
	cdw_regex_assert_subex_number(regex->re_nsub, 13);

	int fifo = 0;
	int total_size = 0, done_size = 0;
	int speed_decimal = 0, speed_fract = 0;
	int tracknum = 0;

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		if (i != 1 && i != 3 && i != 5 && i != 7 && i != 12 && i != 13) {
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE + 1];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}
		if (i == 1) {
			tracknum = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, tracknum = \"%s\" -> %d\n", i, submatch, tracknum);
		} else if (i == 3) {
			done_size = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, done size = \"%s\" -> %d\n", i, submatch, done_size);
		} else if (i == 5) {
			total_size = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, total size = \"%s\" -> %d\n", i, submatch, total_size);
		} else if (i == 7) {
			fifo = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, fifo = \"%s\" -> %d\n", i, submatch, fifo);
		} else if (i == 12) {
			speed_decimal = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed_decimal = \"%s\" -> %d\n", i, submatch, speed_decimal);
		} else if (i == 13) {
			speed_fract = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed_fract = \"%s\" -> %d\n", i, submatch, speed_fract);
		} else {
			cdw_vdm ("WARNING: unexpected submatch %d = %s\n", i, submatch);
		}
	}

	cdw_processwin_display_fifo_and_speed(fifo, speed_decimal, speed_fract);

	int percent = (done_size * 100) / total_size;
	if ((percent >= 0) && (percent <= 100)) {
		if (previous_percent != percent) {
			cdw_sdm ("INFO: previous_percent = %d, percent = %d\n", previous_percent, percent);

			double time_from_start = difftime(time(NULL), time0);
			int todo_size = total_size - done_size;
			int eta = (int) (((1.0 * todo_size) / done_size) * time_from_start);

			char eta_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
			cdw_utils_eta_calculations(eta_string, eta, PROCESSWIN_MAX_RTEXT_LEN);
			previous_percent = percent;
			cdw_processwin_display_eta(eta_string);

			cdw_sdm ("INFO: eta = %d, eta_string = %s\n", eta, eta_string);
		}
	}

	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window,
	   first %d is amount of data already written to CD,
	   second %d is total amount of data to write */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("%d/%d MB"), done_size, total_size);
	cdw_processwin_display_progress_conditional(done_size, total_size, current_value_string);

	if (tracknum > 1) {
		/* sometimes cdrecord prints tracknum = 02 when burning
		   files to non-virgin DVD+RW; since (so far) cdrecord
		   can't properly get multisession info for DVD+RW disc,
		   it is impossible to burn second (or any higher) session
		   to DVD+RW */
		if (thread_task->disc->type == CDW_DVD_RWP) {
			cdw_vdm ("WARNING: cdrecord reports tracknum = %d during burning files to DVD+RW, resetting to 1\n", tracknum);
			tracknum = 1;
		}
	}

	char text_info2_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window,
	   Track is cd data track, %d is its number */
	snprintf(text_info2_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("Track: %d"), tracknum);
	cdw_processwin_display_sub_info(text_info2_string);

	cdw_processwin_wrefresh();

	cdw_cdrecord_burning_performed = true;

	return 0;
}





int cdw_cdrecord_handle_fixating(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* 2TRANS: this is message displayed in process window,
	   meaning that closing session is in progress */
	cdw_processwin_display_sub_info(_("Fixating, please wait..."));

	/* this is unnecessary - writing process always (?) displays 100%,
	so it will be displayed in processwin as well */
	/* processwin_display_progress(100, 100, ""); */

	cdw_processwin_wrefresh();

	return 0;
}





/**
   When attempting to blank media cdrecord may say this:

   wodim: Cannot blank disk, aborting.
   wodim: Some drives do not support all blank types.
   wodim: Try again with wodim blank=all.

   This function is called when cdw catches "Cannot blank disk". The function
   sets thread_task->disc->type_erasable to CDW_FALSE, so that higher level
   code will know that blanking failed and perhaps blanking with mode 'all' is
   required.
*/
int cdw_cdrecord_handle_blanking_not_supported(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->disc->type_erasable = CDW_FALSE;

	return 0;
}





int cdw_cdrecord_handle_device_or_resource_busy(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* 2TRANS: this is message in progress window, meaning that
	   subprocess cannot access CD drive, because the drive is busy */
	cdw_processwin_display_main_info(_("Device or resource busy."));
	/* 2TRANS: this is message in progress window,
	   possible explanation for previous message */
	cdw_processwin_display_sub_info(_("Make sure that device is unmounted."));
	cdw_processwin_wrefresh();

	/* FIXME: cdrecord will try for a few seconds to access drive and
	   then exits. During the tries cdw will display this information, and
	   then will try to read cd info once again, ant then will (or will
	   not) display log with unsuccessful tries. Display a message
	   instead of a log */

	return 0;
}




/* this works rather for wodim */
int cdw_cdrecord_handle_drive_default_speed(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing drive max speed:
                                 1      2      3
           "Drive default speed([ ]*):([ ]*)([0-9]*)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 3);

	const unsigned int sub = 3;
	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		thread_task->disc->write_speeds.drive_default_speed = atoi(submatch);
		cdw_vdm ("INFO: submatch %d, drive default speed = \"%s\" -> %d\n",
			 sub, submatch, thread_task->disc->write_speeds.drive_default_speed);
		return 0;
	}
}




/* this works rather for cdrecord */
int cdw_cdrecord_handle_current_write_speed(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing write speeds:
	                        1      2      3         4          5      6        7        8      9        10      11     12
	 "Current write speed:([ ]+)([0-9]+)([ ]+)kB/s([ ]+)\\(CD([ ]+)([0-9]+)x,([ ]+)DVD([ ]+)([0-9]+)x,([ ]+)BD([ ]+)([0-9]+)x\\)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 12);

	int cd_speed = 0;
	int dvd_speed = 0;

	char submatch[PIPE_BUFFER_SIZE + 1];
	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		if (i != 6 && i != 9) {
			continue;
		}
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 6) { /* speed value for CD */
			cd_speed = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, CD current write speed = \"%s\" -> %d\n", i, submatch, cd_speed);

		} else if (i == 9) { /* speed value for DVD */
			dvd_speed = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, DVD current write speed = \"%s\" -> %d\n", i, submatch, dvd_speed);
		} else {
			;
		}
	}
	/* at this point simple data type should be known,
	   so this should be safe */
	cdw_assert (thread_task->disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD
		    || thread_task->disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_DVD,

		    "ERROR: simple type is not set to any correct value\n");

	if (thread_task->disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		cdw_vdm ("INFO: current write speed for CD = %d\n", cd_speed);
		thread_task->disc->write_speeds.drive_default_speed = cd_speed;
	} else {
		cdw_vdm ("INFO: current write speed for DVD = %d\n", dvd_speed);
		thread_task->disc->write_speeds.drive_default_speed = dvd_speed;
	}

	return 0;
}





int cdw_cdrecord_handle_drive_max_speed(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing drive max speed:
                             1      2      3
	   "Drive max speed([ ]*):([ ]*)([0-9]*)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 3);

	const unsigned int sub = 3;
	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		thread_task->disc->write_speeds.drive_max_speed = atoi(submatch);
		cdw_vdm ("INFO: submatch %d, drive max speed = \"%s\" -> %d\n",
			 sub, submatch, thread_task->disc->write_speeds.drive_max_speed);
		return 0;
	}
}





int cdw_cdrecord_handle_msinfo_cannot_get_next_writable(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* can't get multisession info for next
	   session - to be interpreted by higher-level code */
	thread_task->disc->cdrecord_info.last_sess_start = -1;
	thread_task->disc->cdrecord_info.next_sess_start = -1;

	cdw_vdm ("INFO: last session start=%ld, next session start=%ld\n",
		 thread_task->disc->cdrecord_info.last_sess_start, thread_task->disc->cdrecord_info.next_sess_start);

	return 0;

}





int cdw_cdrecord_handle_msinfo_cannot_read_session_offset(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* can't get multisession info about any existing session (disc
	   is blank?) - to be interpreted by higher-level code */
	thread_task->disc->cdrecord_info.last_sess_start = 0;
	thread_task->disc->cdrecord_info.next_sess_start = 0;

	cdw_vdm ("INFO: last session start=%ld, next session start=%ld\n",
		 thread_task->disc->cdrecord_info.last_sess_start, thread_task->disc->cdrecord_info.next_sess_start);

	return 0;
}





/**
   \brief Read current disc type (version capturing short and long format of information)

   'cdrecord -atip -v' prints information about disc type. Depending on
   version of cdrecord, the format can be:
   new: "Current: CD-RW"
   old: "Current: 0x000A (CD-RW)"   (in old format human-readable part may be omitted)

   This function handles new format of information.

   'CD-RW' can be replaced by other disc types. List of disc types
   recognized by cdrecord is on top of this file. Some of the types are
   also supported by cdw. For discs supported by cdw proper fields in
   thread_task->disc will be set to corresponding values. Otherwise these
   fields will be set to 'unknown' values.
*/
int cdw_cdrecord_handle_disc_type_new_style(regex_t *regex, regmatch_t *matches)
{
	/* format:                  1
	   "Current:[ ]+([CDVRWOMDL+/-]+[a-z ]*)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	unsigned int sub = 1;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}
	cdw_vdm ("INFO: format: new / cdrecord, submatch = \"%s\"\n", submatch);

	if (!strcmp(submatch, "CD-RW")) {
		thread_task->disc->type = CDW_CD_RW;
	} else if (!strcmp(submatch, "CD-R")) {
		thread_task->disc->type = CDW_CD_R;
	} else if (!strcmp(submatch, "CD-ROM")) {
		/* TODO: checking of cdio_disc_mode should be
		   moved to higher-level code */
		if (thread_task->disc->cdio->mode == CDIO_DISC_MODE_CD_DA) {
			thread_task->disc->type = CDW_CD_AUDIO;
		} else {
			thread_task->disc->type = CDW_CD_ROM;
		}
	} else if (!strcmp(submatch, "DVD-R")) {
		thread_task->disc->type = CDW_DVD_R;
	} else if (!strcmp(submatch, "DVD-R sequential recording")) {
		thread_task->disc->type = CDW_DVD_R_SEQ;
	} else if (!strcmp(submatch, "DVD+R")) {
		thread_task->disc->type = CDW_DVD_RP;
	} else if (!strcmp(submatch, "DVD+RW")) {
		thread_task->disc->type = CDW_DVD_RWP;
#if 0 /* cdw 0.6.0 will not support cdrecord + DVD-RW */
	} else if (!strcmp(submatch, "DVD-RW sequential recording")) {
		thread_task->disc->type = CDW_DVD_RW_SEQ;
	} else if (!strcmp(submatch, "DVD-RW restricted overwrite")) {
		thread_task->disc->type = CDW_DVD_RW_RES;
#endif
#if 0 /* no support for DVD+R DL with cdrecord/wodim; disabling this
	 piece of code disables the support in all higher level code,
	 since no value is assigned to "disc->type" */
	} else if (!strcmp(submatch, "DVD+R/DL")
		   && global_config.support_dvd_rp_dl) {

		cdw_vdm ("INFO: disc type is DVD+R/DL\n");
		thread_task->disc->type = CDW_DVD_RP_DL;
#endif
	} else if (!strcmp(submatch, "DVD-ROM")) {
		thread_task->disc->type = CDW_DVD_ROM;
	} else {
		/* thread_task->disc->type = CDW_DISC_TYPE_UNKNOWN */
		cdw_vdm ("WARNING: submatch = \"%s\" => disc type = unsupported\n", submatch);
	}

	return 0;
}





int cdw_cdrecord_handle_disc_type_old_style(regex_t *regex, regmatch_t *matches)
{
	/*                      1
	  Current:[ ]+0x00([0-9a-fA-F]+) */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	unsigned int sub = 1;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr #%d is negative: %d\n", sub, len);
		return -1;
	}
	cdw_vdm ("INFO: format: old / wodim, submatch = 0x00%s\"\n", submatch);

	size_t l = strlen(submatch);
	cdw_assert (l == 2, "ERROR: len is %zd, expected 2\n", l);
	long long ind = strtol(submatch, (char **) NULL, 16);

	thread_task->disc->type = wodim_profiles[ind].cdw_id;
	/* the line above sets correct type of disc_type in most cases,
	   but there are few special cases, handled below: */
	if (thread_task->disc->type == CDW_CD_ROM) {
		/* TODO: checking of cdio_disc_mode should be
		   moved to higher-level code */
		if (thread_task->disc->cdio->mode == CDIO_DISC_MODE_CD_DA) {
			thread_task->disc->type = CDW_CD_AUDIO;
		} else {
			thread_task->disc->type = CDW_CD_ROM;
		}
	} else if (thread_task->disc->type == CDW_DVD_RW_SEQ
		   || thread_task->disc->type == CDW_DVD_RW_RES) {

		/* cdw 0.6.0 will not support cdrecord + DVD-RW */
		thread_task->disc->type = CDW_DISC_TYPE_UNKNOWN;

	} else if (thread_task->disc->type == CDW_DVD_RP_DL) {
		/* no support for DVD+R DL with cdrecord/wodim;
		   this line of code disables the support in all
		   higher level code, since in no other place a
		   value is assigned to "type" */
		thread_task->disc->type = CDW_DISC_TYPE_UNKNOWN;
	} else {
		; /* thread_task->disc->type = some value from wodim_profiles[] */
	}

	if (thread_task->disc->type == CDW_DISC_TYPE_UNKNOWN) {
		cdw_vdm ("WARNING: submatch = \"%s\" => disc type = unknown\n", submatch);
	}

	return 0;
}





/**
   \brief Get multisession info printed by cdrecord when called with -msinfo option

   This function is called if cdrecord prints addresses of:
   \li beginning of last written session
   \li beginning of next session

   that are written in following format: X,Y
   Capturing such information means that this disc is appendable (because
   the second address is valid). Therefore this is the only reasonable place
   to set thread_task->disc->type_writable = CDW_TRUE in this file.
   This assignment can be made here or in higher level code that examines
   values of thread_task->disc->cdrecord_info.last_sess_start and
   thread_task->disc->cdrecord_info.next_sess_start closer.

   If disc was blank then we would have captured "Cannot read session offset";
   if the disc was closed, we would capture "Cannot get next writable address".
*/
int cdw_cdrecord_handle_msinfo_session_sectors(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information about sectors:
                1        2
	   "^([0-9]+),([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE + 1];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 1) {
			thread_task->disc->cdrecord_info.last_sess_start = atol(submatch);
			cdw_vdm ("INFO: submatch %d, last session start = \"%s\" ->  %ld\n",
				 i, submatch, thread_task->disc->cdrecord_info.last_sess_start);
		} else if (i == 2) {
			thread_task->disc->cdrecord_info.next_sess_start = atol(submatch);
			cdw_vdm ("INFO: submatch %d, next session start = \"%s\" ->  %ld\n",
				 i, submatch, thread_task->disc->cdrecord_info.next_sess_start);
		} else {
			;
		}
	}

	/* This function is called when two numbers in format XXX,YYY are
	   captured.
	   If one of the numbers or both are positive then disc is non-empty
	   and appendable.
	   If both numbers are "0" then disc is empty and appendable.
	   The numbers may be not printed at all, then there are
	   two possibilities:
	   1. disc is non-appendable and cdrecord/wodim prints
	      "Cannot read first writable address"
	   2. disc is empty, and cdrecord may print
	      "Cannot read session offset" */
	cdw_vdm ("INFO: last session start=%ld, next session start=%ld\n",
		 thread_task->disc->cdrecord_info.last_sess_start, thread_task->disc->cdrecord_info.next_sess_start);

	return 0;
}





/**
   \brief Get information about duration of blanking process

   Format of such information is following:
   "Blanking time:   27.819s"

  If cdrecord/wodim prints this information then blanking was successful
*/
int cdw_cdrecord_handle_blanking_time(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing blanking time:
	                    1      2        3
	   "Blanking time:([ ]+)([0-9]*).([0-9]*)s" */
	cdw_regex_assert_subex_number(regex->re_nsub, 3);

	size_t seconds = 0;
	size_t miliseconds = 0;
	char submatch[PIPE_BUFFER_SIZE + 1];

	unsigned int sub = 2;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}
	seconds = (size_t) atoi(submatch);
	cdw_vdm ("INFO: submatch %d, seconds: \"%s\" -> %zd\n", sub, submatch, seconds);

	sub = 3;
	len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}
	miliseconds = (size_t) atoi(submatch);
	cdw_vdm ("INFO: submatch %d, miliseconds: \"%s\" -> %zd\n", sub, submatch, miliseconds);

	/* in case of correct blanking time of blanking will always be
	   more than 0 seconds, so I don't check 'if (seconds == 0)' */
	if (miliseconds > 0) {
		thread_task->erase.erase_time = seconds + 1;
	} else {
		thread_task->erase.erase_time = seconds;
	}

	return 0;
}





/**
   \brief Parse table of writing speeds printed by "cdrecord -prcap" (CD speeds only)

   Function parses "Write speed..." lines from example output
   below to extract speed number and value and puts them in
   thread_task->disc->cdrecord_info.write_speeds.speeds[] table.
   Currently only CD speeds are parsed and stored in write_speeds[].
   "Number of supported..." line is not parsed.

  Number of supported write speeds: 9
  Write speed # 0:  1764 kB/s CLV/PCAV (CD  10x, DVD  1x, BD  0x)
  Write speed # 1:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
  Write speed # 2:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
  Write speed # 3:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
  Write speed # 4:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
  Write speed # 5:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
  Write speed # 6:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
  Write speed # 7:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
  Write speed # 8:   705 kB/s CLV/PCAV (CD   4x, DVD  0x, BD  0x)
*/
int cdw_cdrecord_handle_prcap_write_speed_table(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing write speeds:
	                   1           2           3      4      5               6      7
	 "Write speed # ([0-9]+):([ a-zA-Z0-9/]+)([(])CD([ ]+)([0-9]*)x,[ ]+DVD([ ]+)([0-9]*)x" */
	cdw_regex_assert_subex_number(regex->re_nsub, 7);

	int speed_index = 0;
	int cd_speed = 0;
	int dvd_speed = 0;

	char submatch[PIPE_BUFFER_SIZE + 1];
	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		if (i != 1 && i != 5 && i != 7) {
			continue;
		}
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 1) { /* write speed index */
			speed_index = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, write speed index = \"%s\" -> %d\n", i, submatch, speed_index);

		} else if (i == 5) { /* speed value for CD */
			cd_speed = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, CD write speed = \"%s\" -> %d\n", i, submatch, cd_speed);

		} else if (i == 7) { /* speed value for DVD */
			dvd_speed = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, DVD write speed = \"%s\" -> %d\n", i, submatch, dvd_speed);
		} else {
			;
		}
	}
	/* at this point simple data type should be known,
	   so this should be safe */
	cdw_assert (thread_task->disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD
		    || thread_task->disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_DVD,

		    "ERROR: simple type is not set to any correct value\n");

	if (speed_index < CDW_DISC_N_SPEEDS_MAX) {
		if (thread_task->disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			cdw_vdm ("INFO: adding speed %d for CD\n", cd_speed);
			thread_task->disc->write_speeds.speeds[speed_index] = cd_speed;
		} else {
			cdw_vdm ("INFO: adding speed %d for DVD\n", dvd_speed);
			thread_task->disc->write_speeds.speeds[speed_index] = dvd_speed;
		}
	} else {
		cdw_vdm ("WARNING: discarding speed #%d, index larger than write speeds table\n", speed_index);
	}

	cdw_vdm ("INFO: write speed #%2d = %2d\n", speed_index, thread_task->disc->write_speeds.speeds[speed_index]);

	return 0;
}





int cdw_cdrecord_handle_cannot_read_toc(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->disc->cdrecord_info.has_toc = false;
	return 0;
}



int cdw_cdrecord_handle_bad_option(regex_t *regex, regmatch_t *matches)
{
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_BAD_OPTION\n");
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_BAD_OPTION;

	/* regex for capturing bad option:
	                 1
	   "Bad Option: (.+)." */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, 1, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", 1, len);
		return -1;
	} else {
		cdw_vdm ("ERROR: bad cdrecord option is \"%s\"\n", submatch);
		return 0;
	}
}





int cdw_cdrecord_handle_toc_beginning(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing track numbers printed before proper TOC is printed:
	             1             2
	   first: ([0-9]+) last ([0-9]+) */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	int first_track = 0;
	int last_track = 0;

	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, 1, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", 1, len);
		return -1;
	} else {
		first_track = atoi(submatch);
	}
	len = cdw_regex_get_submatch(matches, 2, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", 2, len);
		return -1;
	} else {
		last_track = atoi(submatch);

	}
	cdw_vdm ("INFO: first/last track on disc: %d/%d\n", first_track, last_track);

	/* actually this is the most interesting part */
	thread_task->disc->cdrecord_info.has_toc = true;

	return 0;
}





int cdw_cdrecord_handle_reload_disk_and_hit_cr(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* 2TRANS: this is message in progress window */
	cdw_processwin_display_main_info(_("You need to close drive tray and press ENTER"));
	cdw_processwin_wgetch();

	cdw_rv_t crv = cdw_thread_send_key_to_child_process(CDW_KEY_ENTER);
	if (crv == CDW_OK) {
		;
	} else {
		cdw_vdm ("ERROR: sending key to child process failed\n");
	}
	return 0;
}





int cdw_cdrecord_handle_badly_placed_option(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_BAD_OPTION (2)\n");
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_BAD_OPTION;
	return 0;
}

int cdw_cdrecord_handle_write_error_after(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER\n");
	return 0;
}

int cdw_cdrecord_handle_cannot_open_device(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE\n");
	return 0;
}

int cdw_cdrecord_handle_no_such_file_or_dir(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR\n");
	return 0;
}

int cdw_cdrecord_handle_no_disk(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK\n");
	return 0;
}

int cdw_cdrecord_handle_cannot_send_cue_sheet(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET\n");
	return 0;
}

int cdw_cdrecord_handle_non_ricoh_based_drive(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH\n");
	return 0;
}

int cdw_cdrecord_handle_cannot_open_next_track(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT\n");
	return 0;
}

int cdw_cdrecord_handle_data_may_not_fit(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT;
	cdw_vdm ("ERROR: tool_status.cdrecord |= CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT\n");
	return 0;
}



int cdw_cdrecord_handle_supported_disc_modes(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing bad option:
	                       1
	   "Supported modes: (.+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	unsigned int sub = 1;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	cdw_vdm ("INFO: supported modes: \"%s\"\n", submatch);
	const char *delim = " ";
	char *mode = strtok(submatch, delim);
	while (mode != (char *) NULL) {
		if (!strcmp(mode, "TAO")) {
			cdw_vdm ("INFO: drive supports TAO\n");
			thread_task->disc->accepted_disc_modes |= CDW_DISC_MODE_TAO;
		} else if (!strcmp(mode, "DAO")) {
			cdw_vdm ("INFO: drive supports DAO\n");
			thread_task->disc->accepted_disc_modes |= CDW_DISC_MODE_DAO;
		} else if (!strcmp(mode, "SAO")) {
			cdw_vdm ("INFO: drive supports SAO\n");
			thread_task->disc->accepted_disc_modes |= CDW_DISC_MODE_SAO;
		} else {
			;
		}
		mode = strtok((char *) NULL, delim);
	}

	return 0;
}




int cdw_cdrecord_handle_phys_size(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing physical size:
	                      1
	   "phys size:... +([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	unsigned int sub = 1;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	thread_task->disc->cdrecord_info.phys_size = strtoll(submatch, (char **) NULL, 10);
	cdw_vdm ("INFO: physical size: %lld\n", thread_task->disc->cdrecord_info.phys_size);

	return 0;
}





int cdw_cdrecord_handle_atip_start_of_lead_out(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing start of lead out:
	                              1
	  "ATIP start of lead out: ([0-9]+) " */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	unsigned int sub = 1;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	thread_task->disc->cdrecord_info.start_of_lead_out = strtoll(submatch, (char **) NULL, 10);
	cdw_vdm ("INFO: start of lead out: %lld\n", thread_task->disc->cdrecord_info.start_of_lead_out);

	return 0;
}





int cdw_cdrecord_handle_rzone_size(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing physical size:
	                      1
	   "rzone size:... +([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE + 1];
	unsigned int sub = 1;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	long long int rzone_size = strtoll(submatch, (char **) NULL, 10);
	if (rzone_size > thread_task->disc->cdrecord_info.rzone_size) {
		thread_task->disc->cdrecord_info.rzone_size = rzone_size;
	}
	cdw_vdm ("INFO: rzone size: %lld\n", thread_task->disc->cdrecord_info.rzone_size);

	return 0;
}
