(* :Name: Component *)

(* :Title: Component *)

(* :Author: Tom Wickham-Jones*)

 (* :Package Version: 1.0 *)

(* :Mathematica Version: 2.2 *)


(* :History:
	Created winter 1993 by Tom Wickham-Jones.
	
	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)



BeginPackage[ "ExtendGraphics`ComponentPlot`", "Utilities`FilterOptions`"]

ComponentGraphics::usage = 
	"ComponentGraphics is a graphics object that 
	represents a ternary component plot."


ComponentListPlot::usage = 
	"ComponentListPlot[ data] plots a three-dimensional data set
	in a ternary component plot."


GridStyle::usage = "GridStyle is an option of ComponentGraphics
	that specifies the style in which the triangular grid will
	be drawn."

Grid::usage = "Grid is an option of ComponentGraphics that specifies
	whether or not the grid should be drawn."


Begin["`Private`"]

If[ Not[ $VersionNumber > 2.2],
		Offset[ off_, pos_] := Scaled[ off/100, pos]]

General::baddata = 
	"The data point `1` does not consist of three components
	which sum to 1 and each lie between 0 and 1 inclusive."


Options[ ComponentGraphics] = 
	Join[ Options[ Graphics], 
			{
			Grid -> False,
			GridStyle -> {AbsoluteThickness[ 0.1], AbsoluteDashing[{2}]}
			}]

SetOptions[ ComponentGraphics, AspectRatio -> Automatic];
SetOptions[ ComponentGraphics, PlotRange -> All];
SetOptions[ ComponentGraphics, AxesStyle -> {}];

Options[ ComponentListPlot] = 
	Join[ Options[ ComponentGraphics],
			{
			PlotJoined -> False,
			PlotStyle -> AbsolutePointSize[ 3]
			}]

SetOptions[ ComponentListPlot, AspectRatio -> Automatic];
SetOptions[ ComponentListPlot, PlotRange -> All];
SetOptions[ ComponentListPlot, AxesStyle -> {}];


SizeQ[ d_] :=
	If[ Chop[ d] >= 0. && Chop[ d -1] <= 0., True, False]


ConvertCoordinates[ {h_, k_, l_}] :=
		If[ N[ h+k+l] == 1. && SizeQ[ h] && SizeQ[ k] && SizeQ[ l],
			{h/2 + k, h Tan[Pi/3]/2},
			Message[ ComponentGraphics::baddata, {h,k,l}];
			{0,0}]

ConvertCoordinates[ Offset[ off_, pt_]] :=
	Offset[ off, ConvertCoordinates[ pt]]

ConvertCoordinates[ pt_] :=
	(Message[ ComponentListPlot::baddata, pt]; {0,0})

BuildPrimitives[ Point[ pt_]] :=
	Point[ ConvertCoordinates[ pt]]

BuildPrimitives[ Text[ t_, pt_, stuff___]] :=
	Text[ t, ConvertCoordinates[ pt], stuff]

BuildPrimitives[ Line[ l_List]] :=
	Line[ Map[ ConvertCoordinates, l]]

BuildPrimitives[ Polygon[ l_List]] :=
	Polygon[ Map[ ConvertCoordinates, l]]

BuildPrimitives[ l_List] :=
	Map[ BuildPrimitives, l]

BuildPrimitives[ obj_] :=
	obj

JoinStyle[ style_, prims_] :=
	If[ style === Automatic, prims, Join[ Flatten[ {style}], prims]]

ComponentListPlot[ data_List, opts___] :=
	Block[{style, opt, join, prims},
		opt = Join[ {opts}, Options[ ComponentListPlot]] ;
		style = PlotStyle /. opt ;
		join = PlotJoined /. opt ;
		prims = If[ join === True, Line[ data], Map[ Point, data]] ;
		Show[ ComponentGraphics[ {style, prims}, 
				FilterOptions[ ComponentGraphics, Sequence @@ opt]]]
		]

ComponentGraphics /:
Graphics[ ComponentGraphics[ prims_, opts___]] :=
	Block[{ndata, axes, grid, gstyle, 
	       axstyle, opt, nprims, prol,epil},
		opt = Join[ Flatten[{opts}], Options[ ComponentGraphics]] ;
		axstyle = AxesStyle /. opt ;
		grid    = Grid /. opt ;
		gstyle  = GridStyle /. opt ;
		prol    = BuildPrimitives[ Prolog /. opt] ;
		epil    = BuildPrimitives[ Epilog /. opt] ;
		nprims  = BuildPrimitives[ prims] ;
		axes = JoinStyle[ axstyle, TriangularAxes[]] ;		
		grid = If[ grid === Automatic,
					JoinStyle[ gstyle, Map[ Grid, Range[0.2,0.8,0.2]]], {}] ;		
		Graphics[ {axes, nprims, grid}, 
			Prolog -> prol, Epilog -> epil,
			FilterOptions[ Graphics, Sequence @@ opt]]
		]


ComponentPlot[ fun_, {t_, t1_, t2_}] :=
	ParametricPlot[ ConvertCoordinates[ fun], {t,t1,t2}]


TriangularAxes[ ] :=
	Block[
			{ 
			a = {1,0,0}, 
			b = {0,1,0},
			c = {0,0,1},
			text
			},
		text = {
				Text[ "A", ConvertCoordinates[a], {0, -1.5}],
				Text[ "B", ConvertCoordinates[b], {-1.5,0}],
				Text[ "C", ConvertCoordinates[c], {1.5,0}]
				} ;
		{ text, Line[ Map[ ConvertCoordinates, {a,b,c,a}]]}
		]

	
Grid[ a_] :=
	Block[{b, pos},
		b = 1-a ;
		pos = {
				{{a,b,0}, {0,b,a}},
				{{a,b,0}, {a,0,b}},
				{{a,0,b}, {0,a,b}}
				} ;
		Map[ Line, Map[ ConvertCoordinates, pos, {2}]]
		]

ComponentGraphics /:
Show[ ComponentGraphics[ p_, opts___], nopts___] := 
	(Show[ Graphics[ ComponentGraphics[ p, nopts, opts]]];
	 ComponentGraphics[ p, nopts, opts])


ComponentGraphics /:
Display[ stm_, c:ComponentGraphics[ stuff___]] :=
	(Display[ stm, Graphics[ c]]; c) 


Format[ ComponentGraphics[ ___]] = "-ComponentGraphics-"

End[]

EndPackage[]

(*


<<Component.m

ComponentListPlot[ {{0.15,0.24,0.61}, {0.2, 0.6, 0.2}, {0.4, 0.2, 0.4}}]

ComponentListPlot[ 
	{
	Line[{{0.15,0.24,0.61}, {0.2, 0.6, 0.2}, 
	      {0.4, 0.2,0.4},{0.15,0.24,0.61}}]
	 }]

rndpt := {a=Random[], b = Random[Real, {0,1-a}], 1-a-b}

ComponentListPlot[ Table[ rndpt, {10}]]


ComponentListPlot[ 
	{
	Line[{{0.15,0.24,0.61}, {0.2, 0.6, 0.2}, 
	      {0.4, 0.2,0.4},{0.15,0.24,0.61}}],
	GrayLevel[ 0.5], Polygon[ {rndpt, rndpt, rndpt}]
	 }]


*)

