# functions to create, read, write, and compare md5 checksum tables

"createMD5Sums" <- function(files, path, out=NULL, recursive=FALSE, full.names=FALSE, NA.remove=TRUE){
    
    # perform argument checks
    if (missing(files) && missing(path))
        stop("At least one of the files or path argument must be specified")   

    # define local functions
    "removeDirs" <- function(x){
        dirs <- which(file.info(x$files)$isdir)
        if (length(dirs) > 0){
            x$files <- x$files[-dirs]
            x$labels <- x$labels[-dirs]
        }
        
        if (length(x$files) == 0)
            stop("No files specified after removing directory paths")
        x
    }
    
    # obtain list of files for which to create md5 checksums and
    # generate corresponding labels for resulting md5 strings.
    # 
    #    1. Only path specified:  md5files <- list.files(path, full.names=TRUE, recursive=recursive)
    #    Labels are md5files or md5files with path reface removed depending on full.names argument (absolute or relative paths).
    #    
    #    2. files + path specified: md5files <- list.files(file.path(path,files), full.names=TRUE, recursive=recursive)
    #    Labels are files (relative path)
    #    
    #    3. Only files specified: md5files <- files
    #    Labels are files (absolute paths).        
    if (!missing(files) && missing(path)){ # only files specified
 
        if (!all(file.exists(files)))
            stop("Not all specified files exist on the system. If only the files argument is used, the file paths must be absolute.")
        md5 <- removeDirs(list(files=files, labels=files))
    }
    else if (!missing(path) && missing(files)){ # only path specified
        
        if (!file.info(path)$isdir)
            stop("Specified path does not exist")
 
        md5files <- list.files(path, full.names=TRUE, recursive=recursive)
        labels <- if (full.names) md5files else substring(md5files, nchar(path)+2)
        md5 <- removeDirs(list(files=md5files, labels=labels))
    }
    else {  # files + path specified
        if (!file.info(path)$isdir)
            stop("Specified path does not exist")
        md5files <- file.path(path,files)
        if (!all(file.exists(md5files)))
            stop("Not all specified files exist in the ", path, " directory.")
        md5 <- removeDirs(list(files=md5files, labels=files))
    }

    z <- data.frame(tools::md5sum(md5$files), row.names=md5$labels, stringsAsFactors=FALSE)
    names(z) <- "md5sum"
    
    if (NA.remove){
        nastrings <- which(is.na(z))
        if (length(nastrings) > 0){
            warning("The following files could not be processed to produce md5 strings, possibly because R does not have read permission:\n\n\t", 
                    paste(row.names(z)[nastrings],collapse="\n\t"),"\n")
            z <- z[-nastrings,,drop=FALSE]
        }
    }

    if (is.character(out))
        write.table(z, file=out, col.names=FALSE)
    z
}

"readMD5Sums" <- function(file){
   if (!is.character(file) || !file.exists(file))
       stop(paste("File", file, "does not exist"))
   read.table(file, row.names=1, stringsAsFactors=FALSE)
}

"compareMD5Sums" <- function(src, target, target.file=NULL, ignoreAdditionalFiles=FALSE, verbose=FALSE){
    
    # perform argument checks
    if (missing(target)){
        
        if (!is.character(target.file))
            stop("Either a target MD5 data.frame must be supplied or a path to a file that contains the MD5 table must be specified")
        
        target <- readMD5Sums(target.file)
    }
    
    if (!is.data.frame(src) || !is.data.frame(target))
        stop("src and target must be data.frame objects (e.g., as output by the createMD5Sums function)")
    nsrc <- dim(src)[1]
    ntarget <- dim(target)[1]
    if (nsrc > ntarget)
        stop("Incompatible md5 comparison: the source has ", nsrc, " md5 file summations while the target has ", ntarget, ".")
    
    if (!is.logical(verbose))
        stop("The verbose argument must be TRUE or FALSE")
    
    # define local functions
    "status" <- function(x, str)
        cat("\t", x, ": ", str, "\n", sep="")
    
    "failure" <- function(problems, fname, failure, verbose){
        if (verbose)
            status(fname, failure)
        c(problems, failure)
    }
    
    # initialize variables
    problems <- NULL

    # check for mismatch in file names
    target.names <- row.names(target)
    src.names <- row.names(src)
    file.comparison <- target.names %in% src.names
    if (!ignoreAdditionalFiles && !all(file.comparison)){
        extra <- target.names[which(!file.comparison)]
        warning("The following files on the target system were not included in the source md5 checksum table:\n\n\t", paste(extra,collapse="\n\t"),"\n")
        msg <- paste(length(extra), "additional file(s) found on target system not included in source md5 checksum table")
        problems <- failure(problems, "Extra files", msg, verbose)
    }
           
    for (nm in src.names){

        md5src <- src[nm,]
        md5tgt <- target[nm,]
        
        if (is.na(md5tgt)){
            msg <- paste("Target file", nm, "is missing")
            problems <- failure(problems, nm, msg, verbose)
        }
        else if (md5src != md5tgt){
            msg <- paste("File", nm, "md5sum is different on source and target systems")
            problems <- failure(problems, nm, msg, verbose)
        }
        else if (verbose)
            status(nm, "PASS")
    }
    
    if (is.null(problems) && verbose)
        cat("\n\tMD5 summations check passed on all files")
    
    if (!is.null(problems)){
        problems <- as.matrix(problems)
        dimnames(problems) <- list(paste("[", seq(nrow(problems)),"]", sep=""), "MD5 Problem Description(s)")
    }
    
    problems
}

"checkMD5Revo" <- function(verbose=FALSE){
    
    # not currently supported in Windows
    if (.Platform$OS.type == "windows") 
        .NotYetImplemented()

    # define local functions
    "removeMD5" <- function(x, ...){       
        patterns <- unlist(list(...))
        for (regex in patterns){
          labels <- row.names(x)
          files <- grep(regex, labels)
          if (length(files) > 0)
            x <- x[-files,,drop=FALSE]
        }
        x
    }
    
    # initialize variables
    md5R.file <- file.path(R.home(),"R.md5")
    md5Revo.file <- file.path(R.home(),"Revo.md5")
    md5tools.file <- file.path(R.home(),"tools.md5")
    md5parr.file <- file.path(R.home(),"parr.md5")
    Revopath <- file.path(R.home(),".Revohome")
    tools.installed <- file.exists(md5tools.file)
    parr.installed <- file.exists(md5parr.file)
    z <- NULL
    
#This shouldn't be necessary anymore because of Revo.home function...rbc
    # obtain Revo.home
    #if (!file.exists(Revopath))
    #    stop("File ", Revopath, " does not exist")
    #Revo.home <- scan(file=Revopath, what="character", quiet=TRUE)
    #if (!file.info(Revo.home)$isdir)
    #    stop("Directory ", Revo.home, " does not exist")
    
    # R md5 checksums
    if (!file.exists(md5R.file))
        stop("File ", md5R.file, " does not exist")
    src <- removeMD5(readMD5Sums(md5R.file),"R[.]md5")
    if (!tools.installed)
        src <- removeMD5(src,"^tools")
    if (!parr.installed)
        src <- removeMD5(src,"^library/nws/", "^library/bootNWS/", 
            "^library/nwsserver/", "^library/snow/", "^library/sprngNWS/",
            "^library/sleighMan/", "^library/iterators/", "^library/foreach/",
            "^library/randomShrubberyNWS/", "^library/randomForest/",
            "^library/doNWS/", "^library/doSNOW/") 
    target <- removeMD5(createMD5Sums(path=R.home(), recursive=TRUE),"R[.]md5")
    z <- c(z, compareMD5Sums(src=src, target=target, ignoreAdditionalFiles=TRUE, verbose=verbose))

    # Revo md5 checksums
    if (!file.exists(md5Revo.file))
        stop("File ", md5Revo.file, " does not exist")
    src <- readMD5Sums(md5Revo.file)
    src <- removeMD5(src, "^uninstall.*osx-intel$", "^installers.*installbuilder$")
    if (!tools.installed)
        src <- removeMD5(src,"^tools")    
    target <- createMD5Sums(files=row.names(src), path=Revo.home())
    target <- removeMD5(target, "^uninstall.*osx-intel$", "^installers.*installbuilder$")
    z <- c(z, compareMD5Sums(src=src, target=target, ignoreAdditionalFiles=TRUE, verbose=verbose))
    
    # tools md5 checksums (e.g., gfortran, tcltk)
    if (tools.installed){ 
        src <- readMD5Sums(md5tools.file)
        target <- createMD5Sums(files=row.names(src))
        z <- c(z, compareMD5Sums(src=src, target=target, ignoreAdditionalFiles=TRUE, verbose=verbose))
    }

    # parr md5 checksums (e.g., ParallelR doc and README)
    if (parr.installed){ 
        src <- readMD5Sums(md5parr.file)
        target <- createMD5Sums(files=row.names(src), path=Revo.home())
        z <- c(z, compareMD5Sums(src=src, target=target, ignoreAdditionalFiles=TRUE, verbose=verbose))
    }

    if (is.null(z))
        cat("Source and target md5 checksums match")
    else{
        warning("Source and target md5 checksums do not match")
        print(z)
    }
    
    invisible(z)   
}

