/* This converts a PBM to a C header file containing a DGen font made from the
 * PBM. The PBM should be 768x13, with 8x13 cells which from left to right
 * are the glyphs from 0x20 (space) to 0x7f (character after ~).
 * The color 1 in the PBM is opaque, and 0 is transparent. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

int main(int argc, char *argv[])
{
  FILE *pbm, *df;
  unsigned int pbm_contents[768][13];
  char magic[0x20], tmp[0x200];
  int x,y, i,j;

  /* Open the pbm and read it */
  if(argc < 3)
    { printf("Usage: %s pbmfile outfile\n", argv[0]); exit(2); }
  if(!(pbm = fopen(argv[1], "r"))) exit(1);

  fgets(magic, 0x20, pbm);
  /* Throw away possible comments */
  do {
    fgets(tmp, 0x200, pbm);
  } while(*tmp == '#');
  sscanf(tmp, "%d%d", &x, &y);
  /* Verify magic and dimensions */
  if(strcmp(magic, "P1\n")) { printf("Bad magic\n"); return 1; }
  if(x != 768) { printf("Bad X dimension\n"); return 1; }
  if(y != 13) { printf("Bad Y dimension\n"); return 1; }

  for(y = 0; y < 13; ++y)
    for(x = 0; x < 768; ++x)
	fscanf(pbm, "%d", &pbm_contents[x][y]);
  fclose(pbm);

  /* Start the header to our output file */
  if(!(df = fopen(argv[2], "w"))) exit(1);
  fprintf(df, "/* DGen font, generated by pbm2df from \"%s\" */\n\n"
	      "#include <stdio.h> /* for NULL definition */\n\n", 
	      argv[1]);
  /* Now do each cell */
  for(i = 0; i < 96; ++i)
    {
      fprintf(df, "static int _glyph_%02X[] = {", i + 32);
      j = 0;
      for(y = 0; y < 13; ++y)
	for(x = (i << 3); x < ((i << 3) + 8); ++x)
	  {
	    if(pbm_contents[x][y]) { fprintf(df, "%d, ", j); j = 0; }
	    ++j;
	  }
      fprintf(df, "-1};\n");
    } 
  /* Compile it all into one big mess ;) */
  fprintf(df, "int *dgen_font[] = {\n"
	      "  /* Fill in for the control characters */\n"
	      "  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,\n"
	      "  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,\n"
	      "  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,\n"
	      "  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL");
  for(i = 32; i < 128; ++i)
    fprintf(df, ",\n  _glyph_%02X", i);
  
  /* Close off the end of the file */
  fprintf(df, "\n};\n");
  fclose(df);

  printf("Successfully generated dgen font source \"%s\" from \"%s\"\n",
	 argv[2], argv[1]);
  return 0;
}
