/*
  Copyright (C) 2000-2007

  Code contributed by Greg Collecutt, Joseph Hope and the xmds-devel team

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_integrate.h 1706 2008-01-29 04:42:30Z gmcmanus $
*/

/*! @file xmds_integrate.h
  @brief 

  More detailed explanation...
*/

#ifndef XMDS_INTEGRATE_H
#define XMDS_INTEGRATE_H

#include <xmds_segment.h>
#include <xmds_sequence.h>

// *****************************************************************************
// *****************************************************************************
//                              xmdsIntegrate
// *****************************************************************************
// *****************************************************************************

//! Class to organise integration of relevant equations in simulation
class xmdsIntegrate : public xmdsSegment {

  public :

    //! Enumerator containing the half and full step integration parameters
    enum stepCaseEnum {
      FIRST_HALFSTEP,
      SECOND_HALFSTEP,
      FULLSTEP
    };

  //! Constructor for xmdsIntegrate object
  xmdsIntegrate(
                const xmdsSimulation *const yourSimulation,
                const bool& yourVerboseMode, const bool& adaptiveIP, const bool& is9thOrderIP);

  //! Destructor
  ~xmdsIntegrate();

  //! Processes the xmds element
  void processElement(
                      const Element *const yourElement);

  //! Whether or not the segment needs to run when the simulation is in overtime
  bool runsInOvertime() const;

  protected :

    //! Writes the defines to outfile
    void writeDefines(
          FILE *const outfile) const;

  //! Writes global variables to outfile
  void writeGlobals(
                    FILE *const outfile) const;

  //! Writes function prototypes to outfile
  void writePrototypes(
                       FILE *const outfile) const;

  //! Writes the x-space prototypes to outfile
  virtual void writexSpacePrototype(
                                    FILE *const outfile) const;

  //! Writes the simulation routines
  void writeRoutines(
                     FILE *const outfile) const;

  //! Writes the x-space simulation routines
  virtual void writexSpaceRoutine(
                                  FILE *const outfile) const;

  //! Returns the interval (buh?)
  const XMLString* interval() const;

  //! Returns the lattice
  unsigned long lattice() const;

  //! Returns the number of samples
  unsigned long samples(
                        const unsigned long& index) const;

  //! Returns the vector name list in k-space
  const list<XMLString>* KVectorNamesList() const;

  //! Return whether k-space operators are propagation independent
  bool constantK() const;

  //! Returns the number of operators in k-space
  unsigned long nKOperators() const;

  //! Returns the k-space operator as XMLString
  const XMLString* KOperator(
                             const unsigned long& index) const;

  //! Try to store the index of the named k-space operator.
  //! Return whether the k-space operator exists.
  bool getKOperator(
                    const XMLString& operatorName,
                    unsigned long& index) const;

  //! k-space operators code
  const XMLString* KOperatorsCode() const;

  //! Returns the vector names list
  list<XMLString>* vectorNamesList();

  //! Returns the vector names list
  const list<XMLString>* vectorNamesList() const;

  //! Returns the propagation code
  XMLString* propagationCode();

  //! Returns the propagation code
  const XMLString* propagationCode() const;

  //! Returns the cross vector names list
  const list<XMLString>* crossVectorNamesList() const;

  //! Returns the cross dimension number
  unsigned long crossDimNumber() const;

  //! Returns the cross propagation code
  const XMLString* crossPropagationCode() const;

  //! Determines if k-space operators are used
  bool usesKOperators() const;

  //! returns true if the algorithm is ARK45IP
  const bool AdaptiveIP() const;

  //! returns true if the algorithm is RK9 or ARK89
  const bool Is9thOrderIP() const;

  //! returns if k_propagate should always use the time-dependend version
  const bool Smallmemory() const;

  //! Returns the code element list
  list<XMLString>* codeElementList();

  //! Returns the code element list
  const list<XMLString>* codeElementList() const;

  //! Returns number of non-loop propagation elements
  long unsigned int numNonLoopPropagation() const;

  //! Returns the non-loop propagation code list
  const list<XMLString>* nonLoopPropagationCodeList() const;

  //! Returns the non-loop propagation code list
  list<XMLString>* nonLoopPropagationCodeList();

  //! Returns number of integrate moment groups
  long unsigned int numIntegrateMomentGroups() const;

  //! Returns the integrate moment group list
  const list<integrateMomentGroup>* integrateMomentGroupList() const;

  //! Returns the integrate moment group list
  list<integrateMomentGroup>* integrateMomentGroupList();

  //! Try to store the component structure corresponding to the given number.
  //! Return whether such a coStruct exists.
  //! \todo Why are we passing references to constant ints, here and elsewhere?
  //!       Why not just pass in the int?
  bool getcoStruct(
                   const unsigned long& componentNumber,
                   const coStruct*& thecoStruct) const;

  //! Gets the component key
  // Returns true if the component key has been used before.
  bool getcoKey(
                const unsigned long& componentNumber,
                const unsigned long& operatorNumber,
                unsigned long& coKey) const;

  //! Adds a component pair
  unsigned long addcoPair(
                          const unsigned long& componentNumber,
                          const unsigned long& operatorNumber,
						  const unsigned long& componentLength);

  //! Store the next component pair and its start and end position,
  //! starting from the given position.
  //! Return whether a next component pair was found.
  bool findNextcoPair(
                      XMLString& operatorName,
                      XMLString& componentName,
                      unsigned long& start,
                      unsigned long& end) const;

  //! Determines if generation of noises is to be supressed
  bool noNoises() const;

  //! Write code to check if the simulation should halt
  void writeHaltCheckingCode(FILE *outfile) const;

  //! Write code to check if the step-size is too small
  //! (for adapative integration).
  void writeStepCheckingCode(FILE *outfile) const;

  private :

  XMLString                   myInterval;                     //!< The interval over which to integrate
  unsigned long               myLattice;                      //!< The integration lattice
  list<unsigned long>         mySamplesList;                  //!< The list of samples over the lattice
  list<XMLString>             myKVectorNamesList;             //!< The list of the names of the k-space vectors
  bool                        myConstantK;                    //!< Whether or not the k-operators are constant
  list<XMLString>             myKOperatorNamesList;           //!< The list of k-operator names
  XMLString                   myKOperatorsCode;               //!< The k-operators code
  list<XMLString>             myVectorNamesList;              //!< The list of vector names
  XMLString                   myPropagationCode;              //!< The propagation code
  list<XMLString>             myCrossVectorNamesList;         //!< The list of cross-propagation vector names
  unsigned long               myCrossDimNumber;               //!< The cross-propagation dimension number
  list<XMLString>             myCodeElementList;              //!< Names of code elements
  long                        myNumIntegrateMomentGroups;     //!< Number of non-loop propagation elements
  list<integrateMomentGroup>  myIntegrateMomentGroupList;     //!< The list of integrate moment group elements
  long                        myNumNonLoopPropagation;        //!< Number of non-loop propagation elements
  list<XMLString>             myNonLoopPropagationCodeList;   //!< The list of non-loop propagation codes
  XMLString                   myCrossPropagationCode;         //!< The cross-propagation code
  list<coStruct>              mycoStructList;                 //!< The component structure list
  unsigned long               myNextcoKey;                    //!< The key of the next component
  bool                        myNonoises;                     //!< Flag for turning off noise generation
  bool                        myAdaptiveIP;                   //!< Is it adaptive?
  bool                        myIs9thOrderIP;                 //!< Is it 9th order?
  bool                        mySmallmemory;                  //!< can it save six copies of k-op fields
  bool                        myHaltNonFinite;                //!< If true, halt on non-finite numbers?
  double                      myMinStep;                      //!< Minimum allowed step size for adaptive integration

  //! The output sample count
  void outputSampleCount() const;

  //! Write the integrate initialisation calls to file
  void writeInitialisationCalls(FILE *const outfile) const;

  //! Write code to take a last sample,
  //! print a warning message with a title,
  //! and halt the integration pass.
  //! Important: Don't have back-slash escaped characters
  //! in title or msg! Or, escape the _back-slashes_, at your peril.
  void writeHaltCode(FILE *outfile, const char *title, const char *msg) const;

  //! Writes destruction calls to file
  virtual void writeDestructionCalls(FILE *const outfile) const;

};


#endif // XMDS_INTEGRATE_H
