/*
 * Copyright (C) 2014 Canonical Ltd.
 *
 * This file is part of unity-chromium-extension
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

extern "C" {
#include <unity-webapps-permissions.h>
#include <unity-webapps-application-repository.h>
}

#include "webapps-handler.h"

#include <QDebug>
#include <QSignalSpy>
#include <QTest>


// {{{ mock implementations
struct AppInfo {
    bool install;
    bool allowed;
    bool dontask;
    UnityWebappsApplicationStatus status;
    QString name;
    QString domain;
};

QMap<QString, AppInfo> _mock_app_info;

UnityWebappsApplicationRepository *unity_webapps_application_repository_new_default()
{
    return (UnityWebappsApplicationRepository*)g_object_new(G_TYPE_OBJECT, 0);
}

gboolean unity_webapps_application_repository_prepare(UnityWebappsApplicationRepository *repository)
{
    Q_UNUSED(repository);

    return true;
}

GList * unity_webapps_application_repository_resolve_url(UnityWebappsApplicationRepository *repository, const gchar *url)
{
    Q_UNUSED(repository);

    if (!_mock_app_info.contains(QString::fromUtf8(url))) {
        return 0;
    }

    return g_list_append((GList*)0, (gpointer)g_strdup(url));
}

UnityWebappsApplicationStatus
unity_webapps_application_repository_get_resolved_application_status(
    UnityWebappsApplicationRepository *repository,
    const gchar *application
)
{
    Q_UNUSED(repository);

    if (!_mock_app_info.contains(QString::fromUtf8(application))) {
        return UNITY_WEBAPPS_APPLICATION_STATUS_UNRESOLVED;
    }

    return _mock_app_info.value(QString::fromUtf8(application)).status;
}

const gchar *
unity_webapps_application_repository_get_resolved_application_name(UnityWebappsApplicationRepository *repository,
                                                                    const gchar *application)
{
    Q_UNUSED(repository);

    if (!_mock_app_info.contains(QString::fromUtf8(application))) {
        return 0;
    }

    return g_strdup(_mock_app_info.value(QString::fromUtf8(application)).name.toLocal8Bit().constData());
}

const gchar *
unity_webapps_application_repository_get_resolved_application_domain(UnityWebappsApplicationRepository *repository,
																	  const gchar *application)
{
    Q_UNUSED(repository);

    if (!_mock_app_info.contains(QString::fromUtf8(application))) {
        return 0;
    }

    return g_strdup(_mock_app_info.value(QString::fromUtf8(application)).domain.toLocal8Bit().constData());
}


gboolean unity_webapps_permissions_get_domain_allowed(const gchar *domain)
{
    Q_FOREACH(const QString &key, _mock_app_info.keys()) {
        AppInfo appInfo = _mock_app_info.value(key);
        if (appInfo.domain == QLatin1String(domain)) {
            return appInfo.allowed;
        }
    }

    return false;
}

gboolean unity_webapps_permissions_get_domain_dontask(const gchar *domain)
{
    Q_FOREACH(const QString &key, _mock_app_info.keys()) {
        AppInfo appInfo = _mock_app_info.value(key);
        if (appInfo.domain == QLatin1String(domain)) {
            return appInfo.dontask;
        }
    }

    return false;
}

void unity_webapps_permissions_dontask_domain(const gchar *domain)
{
    Q_UNUSED(domain);
}

void
unity_webapps_application_repository_install_application(UnityWebappsApplicationRepository *repository, const gchar *name,
														 UnityWebappsApplicationRepositoryInstallCallback callback, gpointer user_data)
{
    Q_UNUSED(repository);
    Q_UNUSED(name);
    Q_UNUSED(callback);
    Q_UNUSED(user_data);
}

// }}}


class WebappsHandlerTest : public QObject
{
    Q_OBJECT

public:
    WebappsHandlerTest();

private Q_SLOTS:
    void initTestCase();
    void cleanupTestCase();
    void testUrlLoadedMalformedRequest();
    void testUrlLoaded_data();
    void testUrlLoaded();
    void testDontAskMalformedRequest();
    void testDontAsk_data();
    void testDontAsk();
    void testInstallMalformedRequest();
    void testInstall_data();
    void testInstall();
    void testCreateApplicationDesktopName_data();
    void testCreateApplicationDesktopName();
};

WebappsHandlerTest::WebappsHandlerTest():
    QObject(0)
{
}


void WebappsHandlerTest::initTestCase()
{
    _mock_app_info.insert("https://mail.google.com/", { true, false, false, UNITY_WEBAPPS_APPLICATION_STATUS_AVAILABLE, "Gmail", "mail.google.com" });
    _mock_app_info.insert("http://www.tumblr.com/", { false, true, true, UNITY_WEBAPPS_APPLICATION_STATUS_AVAILABLE, "Tumblr", "www.tumblr.com" });
}

void WebappsHandlerTest::cleanupTestCase()
{
}

void WebappsHandlerTest::testUrlLoadedMalformedRequest()
{
    UnityWebapps::WebappsHandler handler;

    QVariantMap message;
    message.insert("method", QString("url_loaded"));

    QVariantMap reply = handler.url_loaded(message);

    QVERIFY(!reply.empty());
    QCOMPARE(reply.value("error").toString(), QStringLiteral("malformed request"));
}

void WebappsHandlerTest::testUrlLoaded_data()
{
    QTest::addColumn<QString>("url");
    QTest::addColumn<bool>("available");
    QTest::addColumn<QString>("appName");
    QTest::addColumn<QString>("appDomain");

    QTest::newRow("available") << "https://mail.google.com/" << true << "Gmail" << "mail.google.com";
    QTest::newRow("not_available") << "http://www.example.com/" << false << "None" << "None";
}

void WebappsHandlerTest::testUrlLoaded()
{
    QFETCH(QString, url);
    QFETCH(bool, available);
    QFETCH(QString, appName);
    QFETCH(QString, appDomain);

    QVariantMap message;
    message.insert("method", QString("url_loaded"));
    message.insert("url", url);

    UnityWebapps::WebappsHandler handler;
    QVariantMap reply = handler.url_loaded(message);

    QVERIFY(!reply.empty());
    QCOMPARE(reply.value("available").toBool(), available);
    if (available) {
        QCOMPARE(appName, reply.value("appName").toString());
        QCOMPARE(appDomain, reply.value("appDomain").toString());
    }
}

void WebappsHandlerTest::testDontAskMalformedRequest()
{
    UnityWebapps::WebappsHandler handler;

    QVariantMap message;
    message.insert("method", QString("dont_ask"));

    QVariantMap reply = handler.url_loaded(message);

    QVERIFY(!reply.empty());
    QCOMPARE(reply.value("error").toString(), QStringLiteral("malformed request"));
}

void WebappsHandlerTest::testDontAsk_data()
{
    QTest::addColumn<QString>("url");
    QTest::addColumn<bool>("available");

    QTest::newRow("ask") << "https://mail.google.com/" << true;
    QTest::newRow("dont_ask") << "http://www.tumblr.com/" << true;
    QTest::newRow("no app") << "http://www.example.com/" << false;
}

void WebappsHandlerTest::testDontAsk()
{
    QFETCH(QString, url);
    QFETCH(bool, available);

    UnityWebapps::WebappsHandler handler;

    QVariantMap message;
    message.insert("method", QString("dont_ask"));
    message.insert("url", url);

    QVariantMap reply = handler.dont_ask(message);

    QVERIFY(!reply.empty());
    if (available) {
        QCOMPARE(reply.value("dont_ask").toBool(), true);
    } else {
        QCOMPARE(reply.value("dont_ask").toBool(), false);
        QCOMPARE(reply.value("available").toBool(), false);
    }
}

void WebappsHandlerTest::testInstallMalformedRequest()
{
    UnityWebapps::WebappsHandler handler;

    QVariantMap message;
    message.insert("method", QString("install"));

    QVariantMap reply = handler.url_loaded(message);

    QVERIFY(!reply.empty());
    QCOMPARE(reply.value("error").toString(), QStringLiteral("malformed request"));
}

void WebappsHandlerTest::testInstall_data()
{
    QTest::addColumn<QString>("url");
    QTest::addColumn<bool>("available");
    QTest::addColumn<bool>("installed");

    QTest::newRow("installs") << "https://mail.google.com/" << true << true;
    QTest::newRow("wont install") << "http://www.tumblr.com/" << true << false;
    QTest::newRow("no app") << "http://www.example.com/" << false << false;
}

void WebappsHandlerTest::testInstall()
{
    QFETCH(QString, url);
    QFETCH(bool, available);
    QFETCH(bool, installed);

    QVariantMap message;
    message.insert("method", QString("install"));
    message.insert("url", url);

    UnityWebapps::WebappsHandler handler;
    QVariantMap reply = handler.install(message);

    QVERIFY(!reply.empty());
    QCOMPARE(reply.value("installed").toBool(), installed);
    if (!available) {
        QCOMPARE(reply.value("available").toBool(), false);
    }
}

void WebappsHandlerTest::testCreateApplicationDesktopName_data()
{
    QTest::addColumn<QString>("appName");
    QTest::addColumn<QString>("appDomain");
    QTest::addColumn<QString>("expected_uri");

    // Behaviour
    QTest::newRow("strip whitespace") << "White Space Name" << "example.com" << "application://WhiteSpaceNameexamplecom.desktop";
    QTest::newRow("strip special chars") << "%$special%^" << "example.com" << "application://specialexamplecom.desktop";
    QTest::newRow("numbers allowed") << "13monkeys" << "example.com" << "application://13monkeysexamplecom.desktop";
    QTest::newRow("combined") << "##12combined test$%67" << "example.com" << "application://12combinedtest67examplecom.desktop";

    // Examples
    QTest::newRow("live example 1") << "FacebookMessanger" << "facebook.com" << "application://FacebookMessangerfacebookcom.desktop";
    QTest::newRow("live example 2") << "Gmail" << "mail.google.com" << "application://Gmailmailgooglecom.desktop";
    QTest::newRow("live example 3") << "GooglePlus" << "plus.google.com" << "application://GooglePlusplusgooglecom.desktop";
}

void WebappsHandlerTest::testCreateApplicationDesktopName()
{
    QFETCH(QString, appName);
    QFETCH(QString, appDomain);
    QFETCH(QString, expected_uri);

    UnityWebapps::WebappsHandler handler;
    QCOMPARE(handler.createApplicationDesktopName(appName, appDomain), expected_uri);
}

QTEST_MAIN(WebappsHandlerTest);

#include "tst_webapps-handler.moc"
