
"""This demo program solves the equations of static linear elasticity
for a gear clamped at two of its ends and twisted 30 degrees."""

# Copyright (C) 2007 Kristian B. Oelgaard
#
# This file is part of SyFi.
#
# SyFi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# SyFi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SyFi. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Anders Logg, 2008.
# Modified by Kent-Andre Mardal, 2010.
#
# First added:  2007-11-14
# Last changed: 2009-02-25

from dolfin import *
parameters["form_compiler"]["name"]= "sfc"

# Load mesh and define function space
mesh = Mesh("../../data/gear.xml.gz")
V = VectorFunctionSpace(mesh, "CG", 1)

# Sub domain for clamp at left end
class Left(SubDomain):
    def inside(self, x, on_boundary):
        return x[0] < 0.5 and on_boundary

# Dirichlet boundary condition for rotation at right end
class Rotation(Expression):
    def value_shape(self):
        return (3,)

    def eval(self, values, x):
        # Center of rotation
        y0 = 0.5
        z0 = 0.219

        # Angle of rotation (30 degrees)
        theta = 0.5236

        # New coordinates
        y = y0 + (x[1] - y0)*cos(theta) - (x[2] - z0)*sin(theta)
        z = z0 + (x[1] - y0)*sin(theta) + (x[2] - z0)*cos(theta)

        # Clamp at right end
        values[0] = 0.0
        values[1] = y - x[1]
        values[2] = z - x[2]

# Sub domain for rotation at right end
class Right(SubDomain):
    def inside(self, x, on_boundary):
      return x[0] > 0.9 and on_boundary

# Define variational problem
v = TestFunction(V)
u = TrialFunction(V)
f = Constant((0, 0, 0))

E  = 10.0
nu = 0.3

mu    = E / (2*(1 + nu))
lmbda = E*nu / ((1 + nu)*(1 - 2*nu))

I = Identity(3)
#    mu    = Constant(cell)
#    lmbda = Constant(cell)

def epsilon(v):
    return 0.5*(grad(v) + grad(v).T)

def sigma(v):
    return 2*mu*epsilon(v) + lmbda*tr(epsilon(v))*I

a = inner(epsilon(v), sigma(u))*dx
L = dot(v, f)*dx

# Set up boundary condition at left end
c = Constant((0, 0, 0))
bcl = DirichletBC(V, c, Left())

# Set up boundary condition at right end
r = Rotation()
bcr = DirichletBC(V, r, Right())

# Set up boundary conditions
bcs = [bcl, bcr]

# Set up PDE and solve
u = Function(V)
solve(a==L, u, bcs)

# Save solution to VTK format
vtk_file = File("elasticity.pvd")
vtk_file << u

# Plot solution
plot(u, mode="displacement")

# Displace mesh and plot displaced mesh
mesh.move(u)
plot(mesh)

# Hold plots
interactive()
