/*
 *  Copyright (C) 2004-2013 Savoir-Faire Linux Inc.
 *  Author: Emmanuel Milou <emmanuel.milou@savoirfairelinux.com>
 *  Author: Alexandre Savard <alexandre.savard@savoirfairelinux.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA.
 *
 *  Additional permission under GNU GPL version 3 section 7:
 *
 *  If you modify this program, or any covered work, by linking or
 *  combining it with the OpenSSL project's OpenSSL library (or a
 *  modified version of that library), containing parts covered by the
 *  terms of the OpenSSL or SSLeay licenses, Savoir-Faire Linux Inc.
 *  grants you additional permission to convey the resulting work.
 *  Corresponding Source for a non-source form of such a combination
 *  shall include the source code for the parts of OpenSSL used as well
 *  as that of the covered work.
 */

#include "samplerateconverter.h"
#include "sfl_types.h"

SamplerateConverter::SamplerateConverter(int freq, size_t channels /* = 1 */) : floatBufferIn_(),
    floatBufferOut_(), scratchBuffer_(), samples_(0), channels_(channels), maxFreq_(freq), src_state_(0)
{
    int err;
    src_state_ = src_new(SRC_LINEAR, channels_, &err);

    samples_ = (freq * 20) / 1000; // start with 20 ms buffers

    floatBufferIn_.resize(samples_);
    floatBufferOut_.resize(samples_);
    scratchBuffer_.resize(samples_);
}

SamplerateConverter::~SamplerateConverter()
{
    src_delete(src_state_);
}

void
SamplerateConverter::Short2FloatArray(const SFLAudioSample *in, float *out, int len)
{
    // factor is 1/(2^15), used to rescale the short int range to the
    // [-1.0 - 1.0] float range.
    static const float FACTOR = 1.0f / (1 << 15);

    while (len--)
        out[len] = (float) in[len] * FACTOR;
}

void SamplerateConverter::resample(const AudioBuffer &dataIn, AudioBuffer &dataOut)
{
    const double inputFreq = dataIn.getSampleRate();
    const double outputFreq = dataOut.getSampleRate();
    const double sampleFactor = outputFreq / inputFreq;

    if (sampleFactor == 1.0)
        return;

    const size_t nbFrames = dataIn.frames();
    const size_t nbChans = dataIn.channels();

    if (nbChans != channels_) {
        // change channel num if needed
        int err;
        src_delete(src_state_);
        src_state_ = src_new(SRC_LINEAR, nbChans, &err);
        channels_ = nbChans;
    }

    size_t inSamples = nbChans * nbFrames;
    size_t outSamples = inSamples * sampleFactor;

    // grow buffer if needed
    samples_ = std::max(inSamples, outSamples);
    floatBufferIn_.resize(inSamples);
    floatBufferOut_.resize(outSamples);
    scratchBuffer_.resize(outSamples);

    SRC_DATA src_data;
    src_data.data_in = floatBufferIn_.data();
    src_data.data_out = floatBufferOut_.data();
    src_data.input_frames = nbFrames;
    src_data.output_frames = nbFrames * sampleFactor;
    src_data.src_ratio = sampleFactor;
    src_data.end_of_input = 0; // More data will come

    dataIn.interleaveFloat(floatBufferIn_.data());

    src_process(src_state_, &src_data);

    /*
    TODO: one-shot deinterleave and float-to-short conversion
    */
    src_float_to_short_array(floatBufferOut_.data(), scratchBuffer_.data(), outSamples);
    dataOut.deinterleave(scratchBuffer_.data(), src_data.output_frames, nbChans);
}
