#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

from zope.component import queryUtility
from zope.location import location
from zope.interface import implements
from zope.publisher.interfaces import IPublishTraverse, NotFound
from zope.security.checker import canAccess
from zope.security.interfaces import Unauthorized
from zope.security.proxy import removeSecurityProxy

from schooltool.app.interfaces import ISchoolToolApplication
from schooltool.intervention import interfaces
from schooltool.person.interfaces import IPerson
from schooltool.term.interfaces import IDateManager
from schooltool.traverser.traverser import Traverser, TraverserPlugin


class InterventionTabProxy(object):
    """Proxy used as context of the Intervention tab views."""
    implements(interfaces.IInterventionTabProxy)

    def __init__(self, user):
        self.user = user


class InterventionTabTraverserPlugin(TraverserPlugin):
    """Traverses from a user's BasicPerson object to the Intervention tab
       proxy which in turn will serve as context to the Intervention tab views.

    i.e. if you go to localhost/persons/someuser/intervention_tab this traverser
    will get called instead of standard traverser.  It returns an
    InterventionTabProxy for the user.
    """

    def traverse(self, name):
        person = IPerson(self.request.principal, None)
        if person is None or person.username != self.context.username:
            raise Unauthorized
        proxyInterventionTab = InterventionTabProxy(self.context)
        return location.LocationProxy(
            proxyInterventionTab, self.context, self.__name__)


class StudentSchoolYearsProxy(object):
    """Proxy used for traversing to a student message or goal."""
    implements(interfaces.IStudentSchoolYearsProxy)

    def __init__(self, student):
        self.student = student


class StudentSchoolYearsTraverserPlugin(TraverserPlugin):
    """Represents the first step in traversing from a student to a goal or
       message object.

    i.e. if you go to localhost/persons/somestudent/schoolyears this traverser
    will get called instead of standard traverser.  It returns the an
    StudentSchoolYearsProxy for the student.
    """

    def traverse(self, name):
        proxySchoolYears = StudentSchoolYearsProxy(self.context)
        return location.LocationProxy(
            proxySchoolYears, self.context, self.__name__)


class StudentSchoolYearProxy(object):
    """Proxy used for traversing to a student message or goal."""
    implements(interfaces.IStudentSchoolYearProxy)

    def __init__(self, year):
        self.year = removeSecurityProxy(year)


class StudentSchoolYearTraverser(Traverser):
    """Allows traversal to a student's schoolyear proxy.

    i.e. if you go to localhost/persons/somestudent/schoolyears/someyear this
    traverser will get called instead of standard traverser.  It also wraps the
    message in a location proxy so the url looks like what is above.

    The whole reason for having this traversal adaptation is to allow
    persons responsible to see messages or goals for students that they
    are not teaching, advising or administrating.
    """

    def traverse(self, name):
        app = ISchoolToolApplication(None)
        interventionRoot = app[u'schooltool.interventions']
        if name not in interventionRoot:
            raise NotFound(self.context, name, self.request)
        proxySchoolYear = StudentSchoolYearProxy(interventionRoot[name])
        return location.LocationProxy(
            proxySchoolYear, self.context, name)


class StudentMessagesGoalsProxy(object):
    """Proxy used for traversing to a student message or goal."""

    implements(interfaces.IStudentMessagesGoalsProxy)

    def __init__(self, type):
        self.type = type


class StudentMessagesGoalsTraverser(Traverser):
    """Allows traversal to a student's schoolyear proxy.

    i.e. go to localhost/persons/somestudent/schoolyears/someyear/messages or
    goals and this traverser will get called instead of standard traverser.
    It also wraps the message in a location proxy so the url looks like what
    is above.

    The whole reason for having this traversal adaptation is to allow
    persons responsible to see messages or goals for students that they
    are not teaching, advising or administrating.
    """

    def traverse(self, name):
        proxyMessagesGoals = StudentMessagesGoalsProxy(name)
        return location.LocationProxy(
            proxyMessagesGoals, self.context, name)


class StudentMessageGoalTraverser(Traverser):
    """Allows traversal to a student's schoolyear proxy.

    i.e. localhost/persons/somestudent/schoolyears/someyear/goals/somegoal or
    message and this traverser will get called instead of standard traverser.
    It also wraps the message in a location proxy so the url looks like what
    is above.

    The whole reason for having this traversal adaptation is to allow
    persons responsible to see messages or goals for students that they
    are not teaching, advising or administrating.
    """
    def traverse(self, name):
        context = removeSecurityProxy(self.context)
        yearproxy = removeSecurityProxy(context.__parent__)
        yearsproxy = removeSecurityProxy(yearproxy.__parent__)
        student = removeSecurityProxy(yearsproxy.student)
        interventionStudent = removeSecurityProxy(
            yearproxy.year[student.username])
        messagesGoals = interventionStudent[context.type]
        if name not in messagesGoals:
            raise NotFound(self.context, name, self.request)
        messageGoal = messagesGoals[name]
        if not canAccess(messageGoal, 'created'):
            raise Unauthorized
        return location.LocationProxy(
            messageGoal, self.context, name)


class InterventionStudentTraverserPlugin(TraverserPlugin):
    """Allows traversal to a student's intervention object.

    i.e. if you go to localhost/persons/somestudent/intervention this traverser
    will get called instead of standard traverser.  It returns the student's
    intervention object for the current school year.
    """

    def traverse(self, name):
        if queryUtility(IDateManager).current_term is None:
            return self.context
        interventionStudent = interfaces.IInterventionStudent(self.context)
        return interventionStudent


class SectionMessagesProxy(object):
    """Proxy used for traversing to a section's student messages."""

    implements(interfaces.ISectionMessagesProxy)

    def __init__(self, section):
        self.section = section
        self.student = None


class SectionMessagesTraverserPlugin(TraverserPlugin):
    """Allows traversal from a section object to a SectionMessagesProxy
    object, which in turn will be used to traverse to the messages
    container for a given student.

    i.e. if you go to localhost/sections/somesec/messages this traverser
    will get called instead of standard traverser.
    """

    def traverse(self, name):
        proxyMessages = SectionMessagesProxy(self.context)
        return location.LocationProxy(
            proxyMessages, self.context, self.__name__)

class SectionMessagesTraverser(Traverser):
    """Allows traversal to a student's messages container from the
    section messages proxy.

    i.e. if you go to localhost/sections/somesec/messages/someone this
    traverser will get called instead of standard traverser.  It also wraps the
    message in a location proxy so the url looks like what is above.

    The whole reason for having this traversal adaptation is to allow
    the message add view to know what section the message is for.
    """

    def traverse(self, name):
        student = ISchoolToolApplication(None)['persons'][name]
        interventionStudent = interfaces.IInterventionStudent(student)
        proxy = removeSecurityProxy(self.context)
        proxy.student = student
        messages = interventionStudent['messages']
        messages = location.LocationProxy(messages, self.context, name)
        return messages

