# Copyright (C) 2014-2020 all contributors <meta@public-inbox.org>
# License: AGPL-3.0+ <https://www.gnu.org/licenses/agpl-3.0.txt>
use strict;
use warnings;
use Test::More;
use PublicInbox::TestCommon;
my $psgi = "./examples/public-inbox.psgi";
my ($tmpdir, $for_destroy) = tmpdir();
my $pi_config = "$tmpdir/config";
my $inboxdir = "$tmpdir/main.git";
my $addr = 'test-public@example.com';
my @mods = qw(HTTP::Request::Common Plack::Test URI::Escape);
require_mods(@mods);
use_ok 'PublicInbox::Import';
use_ok 'PublicInbox::Git';
my @ls;

foreach my $mod (@mods) { use_ok $mod; }
local $ENV{PI_CONFIG} = $pi_config;
ok(-f $psgi, "psgi example file found");
my $pfx = 'http://example.com/test';
ok(run_script(['-init', 'test', $inboxdir, "$pfx/", $addr]),
	'initialized repo');
PublicInbox::Import::run_die([qw(git config -f), $pi_config,
	'publicinbox.test.newsgroup', 'inbox.test']);
open my $fh, '>', "$inboxdir/description" or die "open: $!\n";
print $fh "test for public-inbox\n";
close $fh or die "close: $!\n";
my $app = require $psgi;
my $git = PublicInbox::Git->new($inboxdir);
my $im = PublicInbox::Import->new($git, 'test', $addr);
# ensure successful message delivery
{
	my $mime = PublicInbox::Eml->new(<<EOF);
From: Me <me\@example.com>
To: You <you\@example.com>
Cc: $addr
Message-Id: <blah\@example.com>
Subject: hihi
Date: Fri, 02 Oct 1993 00:00:00 +0000

> quoted text
zzzzzz
EOF
	$im->add($mime);
	$im->done;
	my $rev = $git->qx(qw(rev-list HEAD));
	like($rev, qr/\A[a-f0-9]{40,}/, "good revision committed");
	@ls = $git->qx(qw(ls-tree -r --name-only HEAD));
	chomp @ls;

	# multipart with two text bodies
	$mime = eml_load 't/plack-2-txt-bodies.eml';
	$im->add($mime);

	# multipart with attached patch + filename
	$mime = eml_load 't/plack-attached-patch.eml';
	$im->add($mime);

	# multipart collapsed to single quoted-printable text/plain
	$mime = eml_load 't/plack-qp.eml';
	like($mime->body_raw, qr/hi =3D bye=/, 'our test used QP correctly');
	$im->add($mime);

	my $crlf = <<EOF;
From: Me
  <me\@example.com>
To: $addr
Message-Id: <crlf\@example.com>
Subject: carriage
  return
  in
  long
  subject
Date: Fri, 02 Oct 1993 00:00:00 +0000

:(
EOF
	$crlf =~ s/\n/\r\n/sg;
	$im->add(PublicInbox::Eml->new($crlf));

	$im->done;
}

test_psgi($app, sub {
	my ($cb) = @_;
	foreach my $u (qw(robots.txt favicon.ico .well-known/foo)) {
		my $res = $cb->(GET("http://example.com/$u"));
		is($res->code, 404, "$u is missing");
	}
});

test_psgi($app, sub {
	my ($cb) = @_;
	my $res = $cb->(GET('http://example.com/test/crlf@example.com/'));
	is($res->code, 200, 'retrieved CRLF as HTML');
	unlike($res->content, qr/\r/, 'no CR in HTML');
	$res = $cb->(GET('http://example.com/test/crlf@example.com/raw'));
	is($res->code, 200, 'retrieved CRLF raw');
	like($res->content, qr/\r/, 'CR preserved in raw message');
});

# redirect with newsgroup
test_psgi($app, sub {
	my ($cb) = @_;
	my $from = 'http://example.com/inbox.test';
	my $to = 'http://example.com/test/';
	my $res = $cb->(GET($from));
	is($res->code, 301, 'newsgroup name is permanent redirect');
	is($to, $res->header('Location'), 'redirect location matches');
	$from .= '/';
	is($res->code, 301, 'newsgroup name/ is permanent redirect');
	is($to, $res->header('Location'), 'redirect location matches');
});

# redirect with trailing /
test_psgi($app, sub {
	my ($cb) = @_;
	my $from = 'http://example.com/test';
	my $to = "$from/";
	my $res = $cb->(GET($from));
	is(301, $res->code, 'is permanent redirect');
	is($to, $res->header('Location'),
		'redirect location matches with trailing slash');
});

foreach my $t (qw(t T)) {
	test_psgi($app, sub {
		my ($cb) = @_;
		my $u = $pfx . "/blah\@example.com/$t";
		my $res = $cb->(GET($u));
		is(301, $res->code, "redirect for missing /");
		my $location = $res->header('Location');
		like($location, qr!/\Q$t\E/#u\z!,
			'redirected with missing /');
	});
}
foreach my $t (qw(f)) {
	test_psgi($app, sub {
		my ($cb) = @_;
		my $u = $pfx . "/blah\@example.com/$t";
		my $res = $cb->(GET($u));
		is(301, $res->code, "redirect for legacy /f");
		my $location = $res->header('Location');
		like($location, qr!/blah\@example\.com/\z!,
			'redirected with missing /');
	});
}

test_psgi($app, sub {
	my ($cb) = @_;
	my $atomurl = 'http://example.com/test/new.atom';
	my $res = $cb->(GET('http://example.com/test/new.html'));
	is(200, $res->code, 'success response received');
	like($res->content, qr!href="new\.atom"!,
		'atom URL generated');
	like($res->content, qr!href="blah\@example\.com/"!,
		'index generated');
	like($res->content, qr!1993-10-02!, 'date set');
});

test_psgi($app, sub {
	my ($cb) = @_;
	my $res = $cb->(GET($pfx . '/atom.xml'));
	is(200, $res->code, 'success response received for atom');
	my $body = $res->content;
	like($body, qr!link\s+href="\Q$pfx\E/blah\@example\.com/"!s,
		'atom feed generated correct URL');
	like($body, qr/<title>test for public-inbox/,
		"set title in XML feed");
	like($body, qr/zzzzzz/, 'body included');
	$res = $cb->(GET($pfx . '/description'));
	like($res->content, qr/test for public-inbox/, 'got description');
});

test_psgi($app, sub {
	my ($cb) = @_;
	my $path = '/blah@example.com/';
	my $res = $cb->(GET($pfx . $path));
	is(200, $res->code, "success for $path");
	my $html = $res->content;
	like($html, qr!<title>hihi - Me</title>!, 'HTML returned');
	like($html, qr!<a\nhref="raw"!s, 'raw link present');
	like($html, qr!&gt; quoted text!s, 'quoted text inline');

	$path .= 'f/';
	$res = $cb->(GET($pfx . $path));
	is(301, $res->code, "redirect for $path");
	my $location = $res->header('Location');
	like($location, qr!/blah\@example\.com/\z!,
		'/$MESSAGE_ID/f/ redirected to /$MESSAGE_ID/');

	$res = $cb->(GET($pfx . '/multipart@example.com/'));
	like($res->content,
		qr/hi\n.*-- Attachment #2.*\nbye\n/s, 'multipart split');

	$res = $cb->(GET($pfx . '/patch@example.com/'));
	$html = $res->content;
	like($html, qr!see attached!, 'original body');
	like($html, qr!.*Attachment #2: foo&(?:amp|#38);\.patch --!,
		'parts split with filename');

	$res = $cb->(GET($pfx . '/qp@example.com/'));
	like($res->content, qr/\bhi = bye\b/, "HTML output decoded QP");
});

test_psgi($app, sub {
	my ($cb) = @_;
	my $res = $cb->(GET($pfx . '/blah@example.com/raw'));
	is(200, $res->code, 'success response received for /*/raw');
	like($res->content, qr!^From !sm, "mbox returned");

	$res = $cb->(GET($pfx . '/blah@example.com/t.mbox.gz'));
	is(501, $res->code, '501 when overview missing');
	like($res->content, qr!\bOverview\b!, 'overview omission noted');
});

# legacy redirects
foreach my $t (qw(m f)) {
	test_psgi($app, sub {
		my ($cb) = @_;
		my $res = $cb->(GET($pfx . "/$t/blah\@example.com.txt"));
		is(301, $res->code, "redirect for old $t .txt link");
		my $location = $res->header('Location');
		like($location, qr!/blah\@example\.com/raw\z!,
			".txt redirected to /raw");
	});
}

my %umap = (
	'm' => '',
	'f' => '',
	't' => 't/',
);
while (my ($t, $e) = each %umap) {
	test_psgi($app, sub {
		my ($cb) = @_;
		my $res = $cb->(GET($pfx . "/$t/blah\@example.com.html"));
		is(301, $res->code, "redirect for old $t .html link");
		my $location = $res->header('Location');
		like($location,
			qr!/blah\@example\.com/$e(?:#u)?\z!,
			".html redirected to new location");
	});
}
foreach my $sfx (qw(mbox mbox.gz)) {
	test_psgi($app, sub {
		my ($cb) = @_;
		my $res = $cb->(GET($pfx . "/t/blah\@example.com.$sfx"));
		is(301, $res->code, 'redirect for old thread link');
		my $location = $res->header('Location');
		like($location,
		     qr!/blah\@example\.com/t\.mbox(?:\.gz)?\z!,
		     "$sfx redirected to /mbox.gz");
	});
}
test_psgi($app, sub {
	my ($cb) = @_;
	# for a while, we used to support /$INBOX/$X40/
	# when we "compressed" long Message-IDs to SHA-1
	# Now we're stuck supporting them forever :<
	foreach my $path (@ls) {
		$path =~ tr!/!!d;
		my $from = "http://example.com/test/$path/";
		my $res = $cb->(GET($from));
		is(301, $res->code, 'is permanent redirect');
		like($res->header('Location'),
			qr!/test/blah\@example\.com/!,
			'redirect from x40 MIDs works');
	}
});

# dumb HTTP clone/fetch support
test_psgi($app, sub {
	my ($cb) = @_;
	my $path = '/test/info/refs';
	my $req = HTTP::Request->new('GET' => $path);
	my $res = $cb->($req);
	is(200, $res->code, 'refs readable');
	my $orig = $res->content;

	$req->header('Range', 'bytes=5-10');
	$res = $cb->($req);
	is(206, $res->code, 'got partial response');
	is($res->content, substr($orig, 5, 6), 'partial body OK');

	$req->header('Range', 'bytes=5-');
	$res = $cb->($req);
	is(206, $res->code, 'got partial another response');
	is($res->content, substr($orig, 5), 'partial body OK past end');
});

# things which should fail
test_psgi($app, sub {
	my ($cb) = @_;

	my $res = $cb->(PUT('/'));
	is(405, $res->code, 'no PUT to / allowed');
	$res = $cb->(PUT('/test/'));
	is(405, $res->code, 'no PUT /$INBOX allowed');

	# TODO
	# $res = $cb->(GET('/'));
});

done_testing();
