/*	New_Conductor_Dialog

PIRL CVS ID: New_Conductor_Dialog.java,v 1.19 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Conductor.Maestro;

import	PIRL.Messenger.Message;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Viewers.Dialog_Box;

import	javax.swing.JDialog;
import	javax.swing.JPanel;
import	javax.swing.JLabel;
import	javax.swing.JTextField;
import	javax.swing.JCheckBox;
import	javax.swing.JButton;
import	javax.swing.JFrame;
import	javax.swing.Box;
import	java.awt.BorderLayout;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.Insets;
import	java.awt.Dimension;
import	java.awt.event.ActionListener;
import	java.awt.event.ActionEvent;
import	java.awt.Frame;


/**	<I>New_Conductor_Dialog</I> provides a dialog used to interactively
	obtain the information needed to form a command line that will start
	a Conductor.
<p>
	New_Conductor_Dialog is a reusable dialog. Once constructed it only
	needs to be setVisible to be activated. If it had been used
	previously the previous field values will remain.
<p>
	@author	Bradford Castalia, UA/PIRL
	@version 1.19
*/
public class New_Conductor_Dialog
	extends JDialog
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.New_Conductor_Dialog (1.19 2012/04/16 06:04:11)";


public static final boolean
	DEFAULT_WAIT_TO_START		= true;
private static boolean
	Default_Wait_to_Start		= DEFAULT_WAIT_TO_START;

private static final String
	PIPELINE		= Conductor_Definition.PIPELINE_PARAMETER_NAME,
	CATALOG			= Conductor_Definition.CATALOG_PARAMETER_NAME,
	CONFIGURATION	= Conductor_Definition.CONFIGURATION_PARAMETER_NAME,
		DEFAULT_CONFIGURATION
			= Conductor_Definition.DEFAULT_CONFIGURATION_FILENAME,
	SERVER			= Conductor_Definition.SERVER_PARAMETER_NAME,
	COUNT			= Conductor_Definition.COUNT_PARAMETER_NAME,
	WAIT_TO_START	= Conductor_Definition.WAIT_TO_START_PARAMETER_NAME;

private	JTextField
	Name_Field					= new JTextField (),
	Pipeline_Field				= new JTextField (),
	Catalog_Field				= new JTextField (),
	Configuration_Field			= new JTextField (),
	Server_Field				= new JTextField (),
	Count_Field					= new JTextField ();

private String
	Previous_Pipeline_Name		= null;

private JCheckBox
	Wait_to_Start_Selection		= new JCheckBox ();

private Message
	Start_Conductor_Message		= null;

//	System new-line.
private static final String
	NL							= Stage_Manager.NL;


//  DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_SETUP		= 1 << 0,
	DEBUG_GET		= 1 << 1,
	DEBUG_CONFIG	= 1 << 2,
	DEBUG_UI		= 1 << 3,
	DEBUG_HELPERS	= 1 << 4,
	DEBUG_ALL		= -1,

	DEBUG       	= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a New_Conductor_Dialog.
<p>
	@param	title	The title String for the dialog window. If null
		"Start Conductor" will be used.
	@param	owner	The Frame with which the dialog is associated,
		which may be null in which case the dialog will be associated
		with the display.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
	@param	name The name of the Conductor definition. If null the {@link
		#Pipeline() pipeline} name is used.
	@param	pipeline	The name of the pipeline to be managed by the
		Conductor. May be null, but if the corresponding field is not
		filled in when the Accept button is pressed a Notice Dialog_Box
		will be shown and the Accept will be aborted.
	@param	catalog	The name of the database catalog containing the
		pipeline tables. May be null.
	@param	configuration	The pathname for the configuration file to be
		used by Conductor. If null, the {@link
		PIRL.Conductor.Conductor#DEFAULT_CONFIGURATION_FILENAME} will be
		used.
	@param	server	The name of the database server access parameters
		group in the configuration file. May be null.
	@param	count	The number of Conductors to start.
	@param	wait_to_start	The {@link #WAIT_TO_START} mode for a
		Conductor will be set to this value.
*/
public New_Conductor_Dialog
	(
	String			title,
	Frame			owner,
	boolean			modal,
	String			name,
	String			pipeline,
	String			catalog,
	String			configuration,
	String			server,
	int				count,
	boolean			wait_to_start
	)
{
super (owner, (title == null ? "New Conductor" : title), modal);
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> New_Conductor_Dialog");

if (configuration == null)
	configuration = DEFAULT_CONFIGURATION;

Name (name);
Pipeline (pipeline);
Catalog (catalog);
Configuration (configuration);
Server (server);
Count (count);
Wait_to_Start (wait_to_start);

//	Build the panels.
getContentPane ().add (Panels ());

pack ();
getRootPane ().setMinimumSize (getContentPane ().getSize ());
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("<<< New_Conductor_Dialog");
}

/**	Constructs a New_Conductor_Dialog.
<p>
	The default name, catalog, configuration pathname, server name and
	count will be used and wait-to-start will be true.
<p>
	@param	title	The title String for the dialog window. If null
		"Start Conductor" will be used.
	@param	owner	The Frame with which the dialog is associated,
		which may be null in which case the dialog will be associated
		with the display.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
	@param	pipeline	The name of the pipeline to be managed by the
		Conductor. May be null, but if the corresponding field is not
		filled in when the Accept button is pressed a Notice Dialog_Box
		will be shown and the Accept will be aborted.
*/
public New_Conductor_Dialog
	(
	String			title,
	Frame			owner,
	boolean			modal,
	String			pipeline
	)
{this (title, owner, modal, pipeline, pipeline, null, null, null, 1,
	Default_Wait_to_Start);}

/**	Constructs a New_Conductor_Dialog.
<p>
	The default name, catalog, configuration pathname, pipeline name,
	server name and count will be used and wait-to-start will be true.
<p>
	@param	title	The title String for the dialog window. If null
		"Start Conductor" will be used.
	@param	owner	The Frame with which the dialog is associated,
		which may be null in which case the dialog will be associated
		with the display.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public New_Conductor_Dialog
	(
	String			title,
	Frame			owner,
	boolean			modal
	)
{this (title, owner, modal, null, null, null, null, null, 1,
	Default_Wait_to_Start);}

/**	Constructs a New_Conductor_Dialog.
<p>
	The default name, catalog, configuration pathname, pipeline name,
	server name and count will be used and wait-to-start will be true.
	The dialog will be modal.
<p>
	@param	title	The title String for the dialog window. If null
		"Start Conductor" will be used.
	@param	owner	The Frame with which the dialog is associated,
		which may be null in which case the dialog will be associated
		with the display.
*/
public New_Conductor_Dialog
	(
	String			title,
	Frame			owner
	)
{this (title, owner, true, null, null, null, null, null, 1,
	Default_Wait_to_Start);}

/**	Constructs a default New_Conductor_Dialog.
*/
public New_Conductor_Dialog ()
{this (null, null, false, null, null, null, null, null, 1,
	Default_Wait_to_Start);}

/**	Get a Message that defines a command line to start a Conductor.
<p>
	A New_Conductor_Dialog is constructed, set visible and the
	resulting {@link #Start_Conductor_Message() message}, if any, is
	returned. The dialog will be modal and the default pipeline name,
	catalog, configuration pathname, server name and count will be used
	and wait-to-start will be true.
<p>
	@param	title	The title String for the dialog window. If null
		"Start Conductor" will be used.
	@param	owner	The Frame with which the dialog is associated,
		which may be null in which case the dialog will be associated
		with the display.
*/
public static Message Start_Conductor_Message
	(
	String			title,
	JFrame			owner
	)
{
New_Conductor_Dialog
	dialog = new New_Conductor_Dialog
		(title, owner, true, null, null, null, null, null, 1,
			Default_Wait_to_Start);
dialog.setVisible (true);
return dialog.Start_Conductor_Message ();
}

/*==============================================================================
	Accessors
*/
/**	Get the dialog title.
<p>
	@return	The title String to be displayed in the dialog window header bar.
	@see	#Title(String)
*/
public String Title ()
{return getTitle ();}

/**	Set the dialog title.
<p>
	@param	title	The title String to be displayed in the dialog window
		header bar.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Title
	(
	String	title
	)
{setTitle (title); return this;}

/**	Get the Conductor name.
<p>
	@return	The name of the Conductor definition. This will be null if
		no name has been specified.
	@see	#Name(String)
*/
public String Name ()
{
String
	name = Name_Field.getText ();
if (name.length () == 0)
	name = null;
return name;
}

/**	Set the Conductor name.
<p>
	@param	name	The name of the Conductor definition. If null the
		{@link #Pipeline() pipeline} name is used.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Name
	(
	String	name
	)
{
if (name == null)
	name = Pipeline ();
Name_Field.setText (name);
return this;
}

/**	Get the pipeline name.
<p>
	@return	The name of the pipeline. This will be null if no pipeline
		has been specified.
	@see	#Pipeline(String)
*/
public String Pipeline ()
{
String
	name = Pipeline_Field.getText ();
if (name.length () == 0)
	name = null;
return name;
}

/**	Set the pipeline name.
<p>
	If the {@link #Name() Conductor name} is null it is set to the pipeline
	name.
<p>
	@param	name	The name of the pipeline. May be null.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Pipeline
	(
	String	name
	)
{
Previous_Pipeline_Name = Pipeline_Field.getText ();
Pipeline_Field.setText (name);
if (Name () == null)
	Name (name);
return this;
}

/**	Get the database catalog name.
<p>
	@return	The name of the catalog. This will be null if no catalog has
		been specified.
	@see	#Catalog(String)
*/
public String Catalog ()
{
String
	name = Catalog_Field.getText ();
if (name.length () == 0)
	name = null;
return name;
}

/**	Set the database catalog name.
<p>
	@param	name	The name of the database catalog. May be null.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Catalog
	(
	String	name
	)
{Catalog_Field.setText (name); return this;}

/**	Get the configuration source name.
<p>
	@return	The source name for the configuration file. This will be null
		if no configuration source has been specified.
	@see	#Configuration(String)
*/
public String Configuration ()
{
String
	name = Configuration_Field.getText ();
if (name.length () == 0)
	name = null;
return name;
}

/**	Set the configuration source name.
<p>
	@param	name	The source name for the configuration file. May be null.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Configuration
	(
	String	name
	)
{Configuration_Field.setText (name); return this;}

/**	Get the database server name.
<p>
	@return	The name of the database server as listed in the Server list
		of the configuration file used by the Conductor. This will be
		null if no database server has been specified.
	@see	#Server(String)
*/
public String Server ()
{
String
	name = Server_Field.getText ();
if (name.length () == 0)
	name = null;
return name;
}

/**	Set the database server name.
<p>
	@param	name	The name of the database server as listed in the
		Server list of the configuration file used by the Conductor. May
		be null.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Server
	(
	String	name
	)
{Server_Field.setText (name); return this;}

/**	Test if wait-to-start is selected.
<p>
	@return	true if wait-to-start has been selected; false otherwise.
	@see	#Wait_to_Start(boolean)
*/
public boolean Wait_to_Start ()
{
return Wait_to_Start_Selection.isSelected ();
}

/**	Set the wait-to-start mode.
<p>
	When wait-to-start mode is enabled the Conductor will be told to wait
	for a start message from a Manager before source processing begins.
	Otherwise the Conductor will begin processing sources immediately
	after it has initialized.
<p>
	@param	wait_to_start	The value of the Coductor wait-to-start mode.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Wait_to_Start
	(
	boolean	wait_to_start
	)
{Wait_to_Start_Selection.setSelected (wait_to_start); return this;}

/**	Set the default wait-to-start mode.
<p>
	@return	The default value of the Coductor wait-to-start mode.
	@see	#Default_Wait_to_Start(boolean)
*/
public static boolean Default_Wait_to_Start ()
{return Default_Wait_to_Start;}

/**	Get the default wait-to-start mode.
<p>
	@param	wait_to_start	The default value of the Coductor wait-to-start
		mode if it is not otherwise specified.
	@see	#Wait_to_Start(boolean)
*/
public static void Default_Wait_to_Start
	(
	boolean	wait_to_start
	)
{Default_Wait_to_Start = wait_to_start;}

/**	Get the Conductor count.
<p>
	@return	The number of Conductors to be started.
	@see	#Count(int)
*/
public int Count ()
{return Integer.parseInt (Count_Field.getText ());}

/**	Set the Conductor count.
<p>
	@param count	The number of Conductors to be started. The value
		will be limited to the range 1 - {@link
		Stage_Manager#DEFAULT_MAX_START_CONDUCTORS_COUNT}. <b>N.B.</b>:
		The Stage_Manager being employed may have been configured to
		impose a lower maximum value.
	@return	This New_Conductor_Dialog.
*/
public New_Conductor_Dialog Count
	(
	int		count
	)
{
if (count <= 0)
	count = 1;
else
if (count > Stage_Manager.DEFAULT_MAX_START_CONDUCTORS_COUNT)
	count = Stage_Manager.DEFAULT_MAX_START_CONDUCTORS_COUNT;
Count_Field.setText (String.valueOf (count));
return this;
}

/**	Get the Conductor command line description message.
<p> @return	The Message that was generated the last time this dialog was
		used and Accept was pressed. This will be null if the dialog was
		not previously used or the last time it was used Cancel was
		pressed.
*/
public Message Start_Conductor_Message ()
{return Start_Conductor_Message;}

/*==============================================================================
	Panels
*/
private JPanel Panels ()
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println (">>> New_Conductor_Dialog.Panels");
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

//	Text fields:

//	Name.
location.anchor		= GridBagConstraints.EAST;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (5, 5, 5, 3);
panel.add (new JLabel ("Conductor name:"), location);

Name_Field.setToolTipText
	("<html>A name for this Conductor definition<br>"
	+"<b>Default</b>: The pipeline name");
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (5, 0, 5, 5);
panel.add (Name_Field, location);

//	Pipeline.
location.anchor		= GridBagConstraints.EAST;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (5, 5, 5, 3);
panel.add (new JLabel ("Pipeline name:"), location);

Pipeline_Field.setToolTipText ("Pipeline for Conductor to manage");
Dimension
	dimension = Pipeline_Field.getPreferredSize ();
dimension.width = 175;
Pipeline_Field.setMinimumSize (dimension);
Pipeline_Field.setPreferredSize (dimension);
Pipeline_Field.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{
	if (Pipeline () != null &&
		(Name () == null ||
		 Name ().equals (Previous_Pipeline_Name)))
		Name (Pipeline ());
	}});
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (5, 0, 5, 5);
panel.add (Pipeline_Field, location);

//	Configuration.
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 5, 5, 3);
panel.add (new JLabel ("Configuration file:"), location);

Configuration_Field.setToolTipText
	("<html>Conductor configuration file pathname<br>" + NL
	+"<b>Default</b>: " + DEFAULT_CONFIGURATION);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (Configuration_Field, location);

//	Server.
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 5, 5, 3);
panel.add (new JLabel ("Database Server:"), location);

Server_Field.setToolTipText
	("<html>Server name in the configuration file<br>" + NL
	+"<b>Default</b>: First Server name listed");
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (Server_Field, location);

//	Catalog.
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 5, 5, 3);
panel.add (new JLabel ("Database catalog:"), location);

Catalog_Field.setToolTipText
	("<html>Database catalog containing the pipeline tables<br>" + NL
	+"<b>Default</b>: Catalog in the configuratin file");
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (Catalog_Field, location);

//	Count.
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 5, 5, 3);
panel.add (new JLabel ("Conductor count:"), location);

Count_Field.setToolTipText
	("<html>Number of Conductors to start<br>" + NL
	+"<b>Default</b>: 1");
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (Count_Field, location);

//	Wait-to-Start.
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 5, 5, 3);
panel.add (new JLabel ("Wait to start:"), location);

Wait_to_Start_Selection.setToolTipText
	("Wait for a Manager to start the Conductor");
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (Wait_to_Start_Selection, location);


//	Buttons panel:

JPanel
	button_panel = new JPanel (new GridBagLayout ());
JButton
	button;
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 0, 0);

//	Accept.
button				= new JButton ("OK");
button.setMnemonic ('O');
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Completed (true);}});
button.setDefaultCapable (true);
getRootPane ().setDefaultButton (button);
location.anchor		= GridBagConstraints.WEST;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
button_panel.add (Box.createHorizontalGlue (), location);

//	Cancel.
button		= new JButton ("Cancel");
button.setMnemonic ('C');
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Completed (false);}});
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.anchor		= GridBagConstraints.EAST;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 5, 5, 5);
panel.add (button_panel, location);

if ((DEBUG & DEBUG_UI) != 0)
	System.out.println ("<<< New_Conductor_Dialog.Panels");
return panel;
}

/*==============================================================================
	Actions
*/
/**	Completes the dialog interaction.
*/
private void Completed
	(
	boolean	accept
	)
{
Start_Conductor_Message = null;

if (accept)
	{
	String
		name = Pipeline ();
	if (name == null)
		{
		Dialog_Box.Notice ("A pipeline name must be specified.", this);
		return;
		}

	int
		count = 1;
	try {count = Count ();}
	catch (NumberFormatException exception)
		{
		Dialog_Box.Notice ("The Conductor count must be a number.");
		return;
		}
	if (count < 1 ||
		count > Stage_Manager.DEFAULT_MAX_START_CONDUCTORS_COUNT)
		{
		Dialog_Box.Notice
			("The Conductor count must be in the range 1 - "
				+ Stage_Manager.DEFAULT_MAX_START_CONDUCTORS_COUNT + '.');
		return;
		}

	if (Start_Conductor_Message == null)
		Start_Conductor_Message =
			Message.Action (Stage_Manager.START_CONDUCTORS_ACTION);
	Start_Conductor_Message.Name (Name ());
	Start_Conductor_Message
			.Set (PIPELINE, name)
			.Set (CATALOG, Catalog ())
			.Set (CONFIGURATION, Configuration ())
			.Set (SERVER, Server ())
			.Set (COUNT, count);
	if (Wait_to_Start ())
		Start_Conductor_Message
			.Set_Token (WAIT_TO_START);
	}
setVisible (false);
}


/*
//	Test stub.
public static void main
	(
	String[]	args
	)
{
Message
	message = Start_Conductor_Message ("Start Conductor", null);
if (message != null)
	System.out.println (message);
System.exit (0);
}
*/


}	//	End of New_Conductor_Dialog class.

