<?php

/**
* Does everything relevant to creating and drawing or filling an n-point polygon
*
* The routine for getting a polygon onto an image is generally to create one,
* add all its points (The order in which the points are added determines how
* the polygon will be drawn), set the image to draw on, and draw.
*
* @version 0.3
* @author  Matt McClanahan <cardinal@dodds.net>
* @package Imlib
* @access public
*/
class ImlibPoly extends ImlibCliprect
{
   /**
   * Resource id# of the image to draw on
   *
   * @access private
   */
   var $im;

   /**
   * The resource id# of the current polygon
   *
   * @access private
   */
   var $poly;

   /**
   * ImlibPoly constructor
   *
   * @access public
   */
   function ImlibPoly()
   {
      $this->cliprect = 0;
      $this->cliprect_inuse = 0;
      $this->color = 0;
      $this->im = 0;
      $this->poly = 0;
   }

   /**
   * Add a point to the current polygon
   *
   * @param int X coordinate
   * @param int Y coordinate
   * @return bool False if there is no polygon set
   * @access public
   */
   function add_point($x,$y)
   {
      if (!is_resource($this->poly))
         return false;

      imlib_polygon_add_point($this->poly,$x,$y);
   }

   /**
   * Checks if the current polygon contains a given point
   *
   * @param int X coordinate
   * @param int Y coordinate
   * @return bool True if it does, false otherwise
   * @access public
   */
   function contains_point($x,$y)
   {
      if (!is_resource($this->poly))
         return false;

      return imlib_polygon_contains_point($this->poly,$x,$y);
   }

   /**
   * Draw the current polygon on the current image
   *
   * @param integer If true, the polygon will be drawn with the endpoints connected
   * @return bool False if there is no polygon or image set
   * @access public
   */
   function draw($closed=1)
   {
      if (!is_resource($this->im) || !is_resource($this->poly))
         return false;

      if (!$this->get_color($r,$g,$b,$a))
         list($r,$g,$b,$a) = Array(255,255,255,255);
 
      if ($this->cliprect_inuse)
         imlib_image_draw_polygon($this->im,$this->poly,$closed,$r,$g,$b,$a,
                                  $this->get_cliprect_array());
      else
         imlib_image_draw_polygon($this->im,$this->poly,$closed,$r,$g,$b,$a);
   }

   /**
   * Fill the current polygon on the current image
   *
   * @return bool False if there is no polygon or image set
   * @access public
   */
   function fill()
   {
      if (!is_resource($this->im) || !is_resource($this->poly))
         return false;

      if (!$this->get_color($r,$g,$b,$a))
         list($r,$g,$b,$a) = Array(255,255,255,255);

      if ($this->cliprect_inuse)
         imlib_image_fill_polygon($this->im,$this->poly,$r,$g,$b,$a,
                                  $this->get_cliprect_array());
      else
         imlib_image_fill_polygon($this->im,$this->poly,$r,$g,$b,$a);
   }

   /**
   * Free the current polygon
   *
   * @return bool False if there is no polygon or image set
   * @access public
   */
   function free()
   {
      if (!is_resource($this->poly))
         return false;

      imlib_polygon_free($this->poly);
      $this->poly = 0;
   }

   /**
   * Get the bounds of the polygon
   *
   * @param int Upper left X coordinate
   * @param int Upper left Y coordinate
   * @param int Lower right X coordinate
   * @param int Lower right Y coordinate
   * @return bool False if there is no polygon or image set
   * @access public
   */
   function get_bounds(&$x1,&$y1,&$x2,&$y2)
   {
      if (!is_resource($this->poly))
         return false;

      imlib_polygon_get_bounds($this->poly,&$x1,&$y1,&$x2,&$y2);
   }

   /**
   * Get the current image resource id#
   *
   * @return int Current image resource id#
   * @access public
   */
   function get_image()
   {
      return $this->im;
   }

   /**
   * Create a new polygon
   *
   * @return int Resource id# of the new polygon
   * @access public
   */
   function new_poly()
   {
      if (is_resource($this->poly))
         return false;

      return $this->poly = imlib_polygon_new();
   }

   /**
   * Set the image resource id# to draw on
   *
   * @param int Image resource id#
   * @access public
   */
   function set_image($im)
   {
      $this->im = $im;
   }
};

?>
