/*
 * window_main.cc
 * Copyright (C) Martin Singer 2009-2013 <m_power3@users.sourceforge.net>
 * 
 * pdfchain is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * pdfchain is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "window_main.h"


/*** Notebook Sections ********************************************************/

// Constructor
cNotebook_Sections::cNotebook_Sections( Gtk::Window& ref_window )
:

	mSection_Cat(    ref_window ),
	mSection_Burst(  ref_window ),
	mSection_BgSt(   ref_window ),
	mSection_Attach( ref_window ),
	mSection_Tool(   ref_window ),

	mLabel_Cat(    _("_Concatenate")        , true ),	// label , mnemonic
	mLabel_Burst(  _("_Burst")              , true ),
	mLabel_BgSt(   _("Background / _Stamp") , true ),
	mLabel_Attach( _("Attach_ment")         , true ),
	mLabel_Tool(   _("_Tools")              , true )
{
	set_scrollable( true );

	mLabel_Cat.set_tooltip_text(    _("Concatenates selected pages of various PDF files into one PDF file") );
	mLabel_Burst.set_tooltip_text(  _("Bursts a PDF file into single pages") );
	mLabel_BgSt.set_tooltip_text(   _("Sets a background or a stamp on each page of a PDF file") );
	mLabel_Attach.set_tooltip_text( _("Adds several attachment files to a PDF file") );
	mLabel_Tool.set_tooltip_text(   _("Some different Tools") );
	
	mAlign_Cat.set_padding(    0                , 0                , 0                , 0                );
	mAlign_Burst.set_padding(  Pdfchain::BORDER , Pdfchain::BORDER , Pdfchain::BORDER , Pdfchain::BORDER );
	mAlign_BgSt.set_padding(   Pdfchain::BORDER , Pdfchain::BORDER , Pdfchain::BORDER , Pdfchain::BORDER );
	mAlign_Attach.set_padding( 0                , 0                , 0                , 0                );
	mAlign_Tool.set_padding(   0                , 0                , 0                , 0                );

	mAlign_Cat.add(    mSection_Cat );
	mAlign_Burst.add(  mSection_Burst );
	mAlign_BgSt.add(   mSection_BgSt );
	mAlign_Attach.add( mSection_Attach );
	mAlign_Tool.add(   mSection_Tool );
	
	append_page( mAlign_Cat    , mLabel_Cat );	// child_widget , tab_label_widget
	append_page( mAlign_Burst  , mLabel_Burst );
	append_page( mAlign_BgSt   , mLabel_BgSt );
	append_page( mAlign_Attach , mLabel_Attach );
	append_page( mAlign_Tool   , mLabel_Tool );
}


// Destructor (virtual)
cNotebook_Sections::~cNotebook_Sections()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cNotebook_Sections::~cNotebook_Sections()";	//TEST
#endif
}

// Method (public) : clear()
void
cNotebook_Sections::clear()
{
	mSection_Cat.clear();
	mSection_Burst.clear();
	mSection_BgSt.clear();
	mSection_Attach.clear();
	mSection_Tool.clear();
	
	set_current_page( 0 );
	return;
}


/*** HBox : Execute ***********************************************************/

// Constructor
cHBox_Execute::cHBox_Execute()
:
	Gtk::HBox( false , Pdfchain::SPACING ),	// homogenous , spacing
	mButton_About( Gtk::Stock::ABOUT ),
	mButton_Clear( Gtk::Stock::CLEAR ),
//	mButton_Execute( Gtk::Stock::EXECUTE )	//TODO: Change button dependent on notebook section
	mButton_Execute( Gtk::Stock::SAVE_AS )
{
	pack_start( mButton_About   , false , false , 0 );	// widget , expand , fill , padding
	pack_start( mButton_Clear   , false , false , 0 );
	pack_start( mSBar_Main      , true  , true  , 0 );
	pack_start( mButton_Execute , false , false , 0 );
	
	Pdfchain::pSBar = &mSBar_Main;
}


// Destructor (virtual)
cHBox_Execute::~cHBox_Execute()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cHBox_Execute::~cHBox_Execute()";	//TEST
#endif
}



/*** Window : Main ************************************************************/

// Constructor
cWindow_Main::cWindow_Main()
:
	mNotebook_Sections( *this ),
	mDialog_About( *this ),
	mMDialog_PdftkError( *this )
{
	Glib::RefPtr<Gio::File>	ref_icon_path_A =
		Gio::File::create_for_path( Pdfchain::APPICON_PATH_A );
		
	Glib::RefPtr<Gio::File>	ref_icon_path_B =
		Gio::File::create_for_path( Pdfchain::APPICON_PATH_B );

	if		( ref_icon_path_A->query_exists() )
		set_default_icon_from_file( Pdfchain::APPICON_PATH_A );
	else if	( ref_icon_path_B->query_exists() )
		set_default_icon_from_file( Pdfchain::APPICON_PATH_B );
	else
		std::cerr << std::endl << "Icon file doesn't exist! " <<
			Pdfchain::APPICON_PATH_A << " , " << Pdfchain::APPICON_PATH_B;

	set_title( Pdfchain::Window::TITLE );
	set_default_size( Pdfchain::Window::WIDTH , Pdfchain::Window::HEIGHT );
	set_position( Gtk::WIN_POS_CENTER );
	
	mAlign_Sections.set_padding(   0                , 0                , 0                , 0                );	// top , bottom , left , right
	mAlign_Permission.set_padding( 0                , 0                , Pdfchain::BORDER , Pdfchain::BORDER );
	mAlign_Execute.set_padding(    Pdfchain::BORDER , Pdfchain::BORDER , Pdfchain::BORDER , Pdfchain::BORDER );
	
	mAlign_Sections.add(   mNotebook_Sections );
	mAlign_Permission.add( mExpander_Permission );
	mAlign_Execute.add(    mHBox_Execute );
	
	mVBox_Main.pack_start( mAlign_Sections   , true  , true  , 0 );	// widget , expand , fill , padding
	mVBox_Main.pack_start( mAlign_Permission , false , false , 0 );
	mVBox_Main.pack_start( mAlign_Execute    , false , false , 0 );
	
	add( mVBox_Main );

	mHBox_Execute.mButton_About.signal_clicked().connect( sigc::mem_fun(
		*this , &cWindow_Main::onButton_About_clicked ) );
	
	mHBox_Execute.mButton_Clear.signal_clicked().connect( sigc::mem_fun(
		*this , &cWindow_Main::onButton_Clear_clicked ) );
	
	mHBox_Execute.mButton_Execute.signal_clicked().connect( sigc::mem_fun(
		*this , &cWindow_Main::onButton_Execute_clicked ) );

	mConnection_NotebookSections_signal_switch_page =
		mNotebook_Sections.signal_switch_page().connect( sigc::mem_fun(
			*this , &cWindow_Main::onNotebook_Sections_switch_page ) );
	
	init();
	show_all_children();
}


// Destructor (virtual)
cWindow_Main::~cWindow_Main()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cWindow_Main::~cWindow_Main()";	//TEST
#endif

	// The signal handler "onNotebook_Sections_switch_page" calls a method at the expander "mExpander_Permissions-object".
	// While destroying the notebook object, the signal "mNotebook_Sections.signal_switch_page()" could be emitted
	// while the mExpander_Permissions object is already destroyed.
	// To prevent a segmentation fault the signal handler is disconnected from the signal.
	mConnection_NotebookSections_signal_switch_page.disconnect();
}


// Method (public) : clear()
void
cWindow_Main::clear()
{
	mNotebook_Sections.clear();
	mExpander_Permission.clear();
//	mHBox_Execute.clear();

	return;
}


// Method (protected) : init()
void
cWindow_Main::init()
{
	return;
}


// Method (protected) : confirm pdftk error code
void
cWindow_Main::confirmErrorCode( int error_code )
{	
	switch ( error_code )
	{
		case 0:
			std::cout << std::endl << "PDFTK returned: " << error_code	<< std::endl;
			break;
		case 256:
			mMDialog_PdftkError.popupMessage( _("Input/Output Error") , error_code );
			break;
		case 512:
			mMDialog_PdftkError.popupMessage( _("Unhandled Java exception at PDFTK") , error_code );
			break;
		case 32512:
			mMDialog_PdftkError.popupMessage( _("PDFTK-Command not found") , error_code );
			break;
		default:
			mMDialog_PdftkError.popupMessage( _("Unknown Error") , error_code );
			break;
	}

	return;
}


// Signal Handler (protected) : on Button About clicked
void
cWindow_Main::onButton_About_clicked()
{
	mDialog_About.run();
	mDialog_About.hide();
	
	return;
}


// Signal Handler (protected) : on Button Clear clicked
void
cWindow_Main::onButton_Clear_clicked()
{
	cWindow_Main::clear();
	return;
}


// Signal Handler (protected) : on Button Execute clicked
void
cWindow_Main::onButton_Execute_clicked()
{
	std::string        str_command = "";
	std::string        str_section = "";
	int                error_code  = 0;
	const Gtk::Widget* pt_page     = mNotebook_Sections.get_nth_page( mNotebook_Sections.get_current_page() );

	if ( pt_page == &(mNotebook_Sections.mAlign_Cat) )
		str_section = mNotebook_Sections.mSection_Cat.createCommand();
	else if ( pt_page == &(mNotebook_Sections.mAlign_Burst) )
		str_section = mNotebook_Sections.mSection_Burst.createCommand();
	else if ( pt_page == &(mNotebook_Sections.mAlign_BgSt) )
		str_section = mNotebook_Sections.mSection_BgSt.createCommand();
	else if ( pt_page == &(mNotebook_Sections.mAlign_Attach) )
		str_section = mNotebook_Sections.mSection_Attach.createCommand();
	else if ( pt_page == &(mNotebook_Sections.mAlign_Tool) )
		str_section = mNotebook_Sections.mSection_Tool.createCommand();
	else
		std::cerr << std::endl << 
			"ERROR: At method cWindow_Main::onButton_Execute_clicked() \
			no valid mNotebook_Sections.get_current_page() can't be found!";

	if ( "" != str_section )
	{
		str_command  = Pdfchain::Cmd::PDFTK;
		str_command += str_section;

		if ( pt_page != &(mNotebook_Sections.mAlign_Tool) )
			str_command += mExpander_Permission.createCommand();

		confirmErrorCode( Pdfchain::execute_command( str_command ) );
	}

	return;
}


// Signal Handler (protected) : on Notebook Sections switch page
void
cWindow_Main::onNotebook_Sections_switch_page( Widget* pt_page , guint page_num )
{
	if ( pt_page == &(mNotebook_Sections.mAlign_Tool) )
	{
		mExpander_Permission.set_expanded( false );
		mExpander_Permission.set_sensitive( false );
	}
	else
	{
		mExpander_Permission.set_sensitive( true );
	}

	return;
}
