///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file CircleObject.h
 * \brief Contains the definition of the StdObjects::CircleObject class.
 */

#ifndef __STDOBJ_CIRCLE_OBJECT_H
#define __STDOBJ_CIRCLE_OBJECT_H

#include <core/Core.h>
#include <core/scene/objects/shapes/SimpleShapeObject.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/gui/properties/PropertiesEditor.h>

#include <stdobjects/StdObjects.h>

namespace StdObjects {

/**
 * \brief The procedural circle shape object.
 *
 * \author Alexander Stukowski
 */
class STDOBJECTS_DLLEXPORT CircleObject : public SimpleShapeObject
{
public:
	/// \brief Default constructors.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	CircleObject(bool isLoading = false);

	/// \brief Returns the controller that controls the radius of the circle object.
	/// \return The internal controller object.
	/// \sa setRadiusController()
	FloatController* radiusController() const { return _radius; }

	/// \brief Replaces the controller that controls the radius of the circle object with a new controller.
	/// \param ctrl The new controller for the radius parameter.
	/// \undoable
	/// \sa radiusController()
	void setRadiusController(FloatController* ctrl) { _radius = ctrl; }

	// From SceneObject:

	/// \brief Asks the object for its validity interval at the given time.
	/// \param time The animation time at which the validity interval should be computed.
	/// \return The maximum time interval that contains \a time and during which the object does not change.
	virtual TimeInterval objectValidity(TimeTicks time);

protected:

	// From SimpleShapeObject:

	/// \brief Builds the shape representation of this procedural object.
	/// \param[in] time The animation time at which to generate the shape.
	/// \param[out] shape The shape object that is be filled by the method.
	/// \param[out] shapeValidity The method stores the validity interval of the generated shape in this output variable.
	virtual void buildShape(TimeTicks time, BezierShape& shape, TimeInterval& shapeValidity);

protected:

	/// This controller stores the radius of the circle.
	ReferenceField<FloatController> _radius;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CircleObject)
	DECLARE_REFERENCE_FIELD(_radius)
};


/**
 * \brief A properties editor for the CircleObject class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the object's parameters.
 *
 * \author Alexander Stukowski
 */
class CircleObjectEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CircleObjectEditor)
};

};

#endif // __STDOBJ_CIRCLE_OBJECT_H
