///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/properties/FloatPropertyUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/data/units/ParameterUnit.h>

#include <atomviz/modifier/analysis/cna/CNAModifier.h>
#include <atomviz/atoms/datachannels/OrientationDataChannel.h>
#include "AnalyzeMicrostructureModifier.h"

namespace CrystalAnalysis {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(AnalyzeMicrostructureModifier, AtomsObjectAnalyzerBase)
DEFINE_REFERENCE_FIELD(AnalyzeMicrostructureModifier, DataChannel, "ClusterChannel", _clusterChannel)
DEFINE_REFERENCE_FIELD(AnalyzeMicrostructureModifier, DataChannel, "MisorientationChannel", _misorientationChannel)
DEFINE_PROPERTY_FIELD(AnalyzeMicrostructureModifier, "GrainMisorientationThreshold", _grainMisorientationThreshold)
DEFINE_PROPERTY_FIELD(AnalyzeMicrostructureModifier, "MinimumGrainSize", _minimumGrainSize)
DEFINE_PROPERTY_FIELD(AnalyzeMicrostructureModifier, "AssignRandomColors", _assignRandomColors)
DEFINE_PROPERTY_FIELD(AnalyzeMicrostructureModifier, "CalculateAtomicMisorientations", _calculateAtomicMisorientations)
SET_PROPERTY_FIELD_LABEL(AnalyzeMicrostructureModifier, _grainMisorientationThreshold, "Misorientation threshold angle")
SET_PROPERTY_FIELD_LABEL(AnalyzeMicrostructureModifier, _minimumGrainSize, "Minimum grain size (# of atoms)")
SET_PROPERTY_FIELD_LABEL(AnalyzeMicrostructureModifier, _assignRandomColors, "Assign random colors to grains")
SET_PROPERTY_FIELD_LABEL(AnalyzeMicrostructureModifier, _calculateAtomicMisorientations, "Calculate atomic misorientations")
SET_PROPERTY_FIELD_UNITS(AnalyzeMicrostructureModifier, _grainMisorientationThreshold, AngleParameterUnit)

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
AnalyzeMicrostructureModifier::AnalyzeMicrostructureModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading), _grainMisorientationThreshold(2.0/180.0*FLOATTYPE_PI),
	_minimumGrainSize(100), _assignRandomColors(true), _calculateAtomicMisorientations(false)
{
	INIT_PROPERTY_FIELD(AnalyzeMicrostructureModifier, _clusterChannel);
	INIT_PROPERTY_FIELD(AnalyzeMicrostructureModifier, _misorientationChannel);
	INIT_PROPERTY_FIELD(AnalyzeMicrostructureModifier, _grainMisorientationThreshold);
	INIT_PROPERTY_FIELD(AnalyzeMicrostructureModifier, _minimumGrainSize);
	INIT_PROPERTY_FIELD(AnalyzeMicrostructureModifier, _assignRandomColors);
	INIT_PROPERTY_FIELD(AnalyzeMicrostructureModifier, _calculateAtomicMisorientations);
	if(!isLoading) {
		_clusterChannel = new DataChannel(DataChannel::ClusterChannel);
		_misorientationChannel = new DataChannel(qMetaTypeId<FloatType>(), sizeof(FloatType), 1);
		_misorientationChannel->setName(tr("Atomic Misorientation"));
	}
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus AnalyzeMicrostructureModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	if(input()->atomsCount() != clusterChannel()->size())
		throw Exception(tr("Number of atoms of input object has changed. Analysis results became invalid."));

	// Create a copy of the internal buffer channels and assign them to the output AtomsObject.
	CloneHelper cloneHelper;

	DataChannel::SmartPtr clusterClone = cloneHelper.cloneObject(clusterChannel(), true);
	output()->replaceDataChannel(outputStandardChannel(DataChannel::ClusterChannel), clusterClone.get());

	if(_calculateAtomicMisorientations && _misorientationChannel && _misorientationChannel->size() == input()->atomsCount()) {
		DataChannel::SmartPtr misorientationClone = cloneHelper.cloneObject(_misorientationChannel, true);
		output()->insertDataChannel(misorientationClone.get());
	}

	if(_assignRandomColors && _findGrains.grains().size() > 0) {
		const int* clusterIter = clusterChannel()->constDataInt();
		const int* clusterIterEnd = clusterIter + clusterChannel()->size();
		Vector3* colorIter = outputStandardChannel(DataChannel::ColorChannel)->dataVector3();
		for(; clusterIter != clusterIterEnd; ++clusterIter, ++colorIter) {
			OVITO_ASSERT(*clusterIter < _findGrains.grains().size());
			if(*clusterIter >= 0 && *clusterIter < _findGrains.grains().size())
				*colorIter = _findGrains.grains()[*clusterIter].color();
			else
				*colorIter = Color(1,1,1);
		}
	}

	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, tr("Number of identified grains: %1\n").arg(_findGrains.grains().size()));
}

/******************************************************************************
* This is the actual analysis method.
******************************************************************************/
EvaluationStatus AnalyzeMicrostructureModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	if(calculate(input(), suppressDialogs))
		return EvaluationStatus();
	else
		return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Calculation has been canceled by the user."));
}

/******************************************************************************
* Performs the analysis.
* Throws an exception on error.
* Returns false when the operation has been canceled by the user.
******************************************************************************/
bool AnalyzeMicrostructureModifier::calculate(AtomsObject* atomsObject, bool suppressDialogs)
{
	FloatType nearestNeighborCutoff = nearestNeighborList()->nearestNeighborCutoff();

	if(_grainMisorientationThreshold <= 0.0)
		throw Exception(tr("The grain misorientation threshold angle must be positive."));

	// Decompose structure into individual grains.
	if(!_findGrains.performAnalysis(atomsObject, _clusterChannel, nearestNeighborCutoff, _grainMisorientationThreshold,
			_minimumGrainSize, _calculateAtomicMisorientations ? _misorientationChannel : (DataChannel*)NULL, suppressDialogs))
		return false;

	return true;
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void AnalyzeMicrostructureModifier::saveToStream(ObjectSaveStream& stream)
{
	AtomsObjectAnalyzerBase::saveToStream(stream);

	stream.beginChunk(0x10000000);
	_findGrains.saveToStream(stream);
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void AnalyzeMicrostructureModifier::loadFromStream(ObjectLoadStream& stream)
{
	AtomsObjectAnalyzerBase::loadFromStream(stream);

	stream.expectChunk(0x10000000);
	_findGrains.loadFromStream(stream);
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr AnalyzeMicrostructureModifier::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	AnalyzeMicrostructureModifier::SmartPtr clone = static_object_cast<AnalyzeMicrostructureModifier>(AtomsObjectAnalyzerBase::clone(deepCopy, cloneHelper));
	clone->_findGrains = this->_findGrains;
	return clone;
}

IMPLEMENT_PLUGIN_CLASS(AnalyzeMicrostructureModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void AnalyzeMicrostructureModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Analyze Microstructure"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	QGridLayout* layout2 = new QGridLayout();
	layout2->setContentsMargins(0,0,0,0);
	layout2->setSpacing(0);
	layout2->setColumnStretch(1, 1);
	layout1->addLayout(layout2);

	// Angle threshold parameter
	FloatPropertyUI* misorientationThresholdPUI = new FloatPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AnalyzeMicrostructureModifier, _grainMisorientationThreshold));
	layout2->addWidget(misorientationThresholdPUI->label(), 0, 0);
	layout2->addWidget(misorientationThresholdPUI->textBox(), 0, 1);
	layout2->addWidget(misorientationThresholdPUI->spinner(), 0, 2);
	misorientationThresholdPUI->setMinValue(0);

	// Minimum grain size parameter
	IntegerPropertyUI* minimumGrainSizePUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AnalyzeMicrostructureModifier, _minimumGrainSize));
	layout2->addWidget(minimumGrainSizePUI->label(), 1, 0);
	layout2->addWidget(minimumGrainSizePUI->textBox(), 1, 1);
	layout2->addWidget(minimumGrainSizePUI->spinner(), 1, 2);
	minimumGrainSizePUI->setMinValue(0);

	BooleanPropertyUI* assignColorsUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AnalyzeMicrostructureModifier, _assignRandomColors));
	layout1->addWidget(assignColorsUI->checkBox());

	BooleanPropertyUI* calcAtomicMisorientionsUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AnalyzeMicrostructureModifier, _calculateAtomicMisorientations));
	layout1->addWidget(calcAtomicMisorientionsUI->checkBox());

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	QPushButton* recalcButton = new QPushButton(tr("Calculate"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onRecalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());

	// Open a sub-editor for the NearestNeighborList sub-object.
	new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _nearestNeighborList), rolloutParams.before(rollout));
}

/******************************************************************************
* Is called when the user presses the Recalculate button.
******************************************************************************/
void AnalyzeMicrostructureModifierEditor::onRecalculate()
{
	if(!editObject()) return;
	AnalyzeMicrostructureModifier* modifier = static_object_cast<AnalyzeMicrostructureModifier>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to analyze microstructure."));
		ex.showError();
	}
}

};	// End of namespace CrystalAnalysis
