///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>

#include "AbstractFileColumnParser.h"

namespace AtomViz {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(AbstractFileColumnParser, AtomsFileParser)

/******************************************************************************
 * Sets the name of the input file for this parser.
 *****************************************************************************/
bool AbstractFileColumnParser::setInputFile(const QString& filename)
{
	int numberOfColumns = 0;
	QStringList columnNames;

	// Determine the number of data columns in the input file.
	if(!inspectFileHeader(filename, numberOfColumns, columnNames))
		return false;

	if(numberOfColumns <= 0)
		throw Exception(tr("The atoms file '%1' is invalid. It contains no data columns.").arg(filename));

	// Resize the mapping array.
	if(_columnMapping.columnCount() == 0) {
		// Use the last mapping by default.
		QSettings settings;
		settings.beginGroup("atomviz/io/columnmapping/");
		if(settings.contains(pluginClassDescriptor()->name())) {
			try {
				VerboseLogger() << "Loading last column-to-channel mapping for " << pluginClassDescriptor()->name() << " importer from settings store." << endl;
				_columnMapping.fromByteArray(settings.value(pluginClassDescriptor()->name()).toByteArray());
			}
			catch(Exception& ex) {
				ex.prependGeneralMessage(tr("Failed to load last used column-to-channel mapping from application settings store."));
				ex.logError();
			}
		}
		settings.endGroup();
		// Clear the old column names because they probably stem from another input file.
		_columnMapping.clearColumnNames();
	}
	_columnMapping.setColumnCount(numberOfColumns, columnNames);

	return AtomsFileParser::setInputFile(filename);
}

/******************************************************************************
 * Sets the mapping between data columns in the input file and
 * the DataChannel objects in the final AtomsObject.
 *****************************************************************************/
void AbstractFileColumnParser::setColumnMapping(const ColumnChannelMapping& mapping)
{
	_columnMapping = mapping;

	// Remember the mapping for the next time.
	VerboseLogger() << "Saving most recent column-to-channel mapping for " << pluginClassDescriptor()->name() << " importer to settings store." << endl;
	QSettings settings;
	settings.beginGroup("atomviz/io/columnmapping/");
	settings.setValue(pluginClassDescriptor()->name(), mapping.toByteArray());
	OVITO_ASSERT(settings.contains(pluginClassDescriptor()->name()));
	settings.endGroup();

	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
 * Saves the class' contents to the given stream.
 *****************************************************************************/
void AbstractFileColumnParser::saveToStream(ObjectSaveStream& stream)
{
	AtomsFileParser::saveToStream(stream);
	stream.beginChunk(0x442AAB);
	// Save the parser's column to data channel mapping into the scene file.
	_columnMapping.saveToStream(stream);
	stream.endChunk();
}

/******************************************************************************
 * Loads the class' contents from the given stream.
 *****************************************************************************/
void AbstractFileColumnParser::loadFromStream(ObjectLoadStream& stream)
{
	AtomsFileParser::loadFromStream(stream);
	stream.expectChunk(0x442AAB);
	// Load the parser's column to data channel mapping from the scene file.
	_columnMapping.loadFromStream(stream);
	stream.closeChunk();
}

/******************************************************************************
 * Creates a copy of this object.
 *****************************************************************************/
RefTarget::SmartPtr AbstractFileColumnParser::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	AbstractFileColumnParser::SmartPtr clone = static_object_cast<AbstractFileColumnParser>(AtomsFileParser::clone(deepCopy, cloneHelper));
	clone->_columnMapping = this->_columnMapping;
	return clone;
}

};	// End of namespace AtomViz
