///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __WRAP_PERIODIC_IMAGES_MODIFIER_H
#define __WRAP_PERIODIC_IMAGES_MODIFIER_H

#include <core/Core.h>
#include <atomviz/AtomViz.h>
#include <atomviz/modifier/AtomsObjectModifierBase.h>

namespace AtomViz {

/**
 * \brief This modifier wraps around the atomic positions in directions with periodic boundary
 *        conditions.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT WrapPeriodicImagesModifier : public AtomsObjectModifierBase
{
public:

	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	WrapPeriodicImagesModifier(bool isLoading = false);

	/// \brief Asks the modifier for its validity interval at the given time.
	///
	/// This method returns the maximum time interval during which
	/// the object's parameters stay constant and that includes the given
	/// point in time.
	virtual TimeInterval modifierValidity(TimeTicks time) { return TimeForever; }

protected:

	/// \brief Modifies the atoms object.
	///
	/// The time interval passed to the function is cut down by this method to the interval
	/// where the output object produced by this modifier is valid/constant.
	/// This method is part of the implementation of the abstract AtomsObjectModifierBase class.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

private:

	/// This helper class is used to split up the computation into small
	/// operations that can be performed on multiple processors in parallel.
	template<int dim>
	class Kernel {
	public:
		// Constructor that stores the parameters.
		Kernel(const AffineTransformation& _simCell, const AffineTransformation& _inverseSimCell) : simCell(_simCell), inverseSimCell(_inverseSimCell) {}

		// The actual kernel function that is called by the Qt concurrent framework for each atom.
		void operator()(Point3& p) const {
			p -= simCell.column(dim) * floor((inverseSimCell * p)[dim]);
		}

	private:
		const AffineTransformation simCell;
		const AffineTransformation inverseSimCell;
	};

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(WrapPeriodicImagesModifier)
};

/**
 * \brief A properties editor for the WrapPeriodicImagesModifier class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the modifier's parameters.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT WrapPeriodicImagesModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(WrapPeriodicImagesModifierEditor)
};

};	// End of namespace AtomViz

#endif // __WRAP_PERIODIC_IMAGES_MODIFIER_H
