/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "SourcesCollection.h"

// LLVM
#ifdef LLVM_27_OR_28
#include <llvm/System/Path.h>
#else
#include <llvm/Support/Path.h>
#endif

#include "GTLCore/String.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/Utils_p.h"

#include "Debug.h"
#include "Source.h"

using namespace OpenShiva;

struct SourcesCollection::Private {
  std::list< Source > kernels;
  void addDirectory( const llvm::sys::Path& path );
};


void SourcesCollection::Private::addDirectory( const llvm::sys::Path& path )
{
  std::set< llvm::sys::Path > content;
  path.getDirectoryContents( content, 0 );
  SHIVA_DEBUG( "Looking for kernel in: " << path.c_str() );
  foreach( const llvm::sys::Path& path, content )
  {
    SHIVA_DEBUG( "Kernel: " << path.c_str() << "?" );
    if( GTLCore::String( path.getSuffix() ).toLower() == "shiva" )
    {
      SHIVA_DEBUG("Loading: " << path.c_str() );
      Source kernel;
      kernel.loadFromFile( path.c_str() );
      kernels.push_back( kernel );
    }
  }
}

SourcesCollection::SourcesCollection() : d(new Private)
{
  addDirectory( _OPENSHIVA_SHIVA_KERNELS_DIR_ );
  addDirectory( llvm::sys::Path::GetUserHomeDirectory().str() + "/.OpenGTL/shiva/kernels" );
}

SourcesCollection::~SourcesCollection()
{
  delete d;
}

void SourcesCollection::addDirectory(const GTLCore::String& directory)
{
  d->addDirectory( llvm::sys::Path( (const std::string&)directory ) );
}

void SourcesCollection::registerSource( const Source& kernel )
{
  d->kernels.push_back( kernel );
}

std::list< Source> SourcesCollection::sources() const
{
  return d->kernels;
}

std::list< Source > SourcesCollection::sources(Source::SourceType _source) const
{
  std::list< Source > sources;
  foreach(const Source& source,d->kernels)
  {
    if(source.sourceType() == _source)
    {
      sources.push_back(source);
    }
  }
  return sources;
}

Source SourcesCollection::source(const GTLCore::String& _string) const
{
  foreach(Source source,d->kernels)
  {
    if(source.name() == _string)
    {
      return source;
    }
  }
  return Source();
}
