# listenable.py
#
#   Copyright (C) 2004 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# A generic class that you can "listen" to events on.  You can derive
# from this, or you can instantiate it as a field (eg, if multiple
# events are available).

import sets
import weak

class Connection:
    """Represents a single conenction between a Listenable and a
    callback.  Used mainly to make an easy unique ID (the object
    identity)."""
    def __init__(self, listener, as_weak):
        if as_weak:
            self.__wrapper=weak.WeakCallableRef(listener)
        else:
            self.__wrapper=lambda:listener

    def live(self):
        """Returns the underlying function if it is live, None otherwise."""
        return self.__wrapper()

class Listenable:
    """An object with which callbacks can be registered.  The callbacks are
    weakly held (with object methods being handled in the expected way) and
    invoked when the 'comments' of this object change."""
    def __init__(self):
        self.listeners=sets.Set()

    def add_listener(self, listener, as_weak=True):
        """Register a new callback with this object.  If as_weak is
        False, the callback will be strongly held."""

        conn=Connection(listener, as_weak)
        self.listeners.add(conn)
        return conn

    def remove_listener(self, conn):
        """Remove a callback based on its connection id, or on its
        target."""

        if isinstance(conn, Connection):
            self.listeners.remove(conn)
        else:
            for conn in list(self.listeners):
                f=conn.live()

                if f == None or conn == f:
                    self.listeners.remove(conn)

    def call_listeners(self, *args, **kw):
        """Calls all the listeners with the given arguments."""

        # NOTE: copy the set of listeners so that if some get deleted
        # while we're iterating, we don't do weird things.
        for conn in list(self.listeners):
            f=conn.live()

            if f == None:
                # Use "discard", not "remove", because conn might have
                # been removed already by a callback; this avoids
                # throwing a KeyError.
                self.listeners.discard(conn)
            else:
                f(*args, **kw)
