-*
   Copyright 2020, Luigi Ferraro, Federico Galetto,
   Francesca Gandini, Hang Huang, Matthew Mastroeni, Xianglong Ni.

   You may redistribute this file under the terms of the GNU General Public
   License as published by the Free Software Foundation, either version 2 of
   the License, or any later version.
*-

document {
	Key => {actionMatrix, (actionMatrix, LinearlyReductiveAction)},
	
	Headline => "matrix of a linearly reductive action",
	
	Usage => "actionMatrix L",
	
	Inputs => {
	        "L" => LinearlyReductiveAction,
		},
	    
	Outputs => {
		Matrix => {"of a group action on a ring"}
		},
	    
	"This function is provided by the package ", TO InvariantRing,
	".",
	
	PARA { 
	    "Suppose the action ", TT "L", " consists of the linearly
	    reductive group with coordinate ring ", TT "S/I",
	    " (where ", TT "S", " is a polynomial ring) acting on 
	    a (quotient of)
	    a polynomial ring ", TT "R", " via the action matrix ",
	    TT "M", ". This function returns the action matrix ", TT "M",
	    ".",
	    },
	EXAMPLE {
		"S = QQ[z]",
		"I = ideal(z^2 - 1)",
		"M = matrix{{(z+1)/2, (1-z)/2},{(1-z)/2, (z+1)/2}}",
		"R = QQ[x,y]",
		"L = linearlyReductiveAction(I, M, R)",
		"actionMatrix L",
		},
	    }

document {
	Key => {groupIdeal, (groupIdeal, LinearlyReductiveAction)},
	
	Headline => "ideal defining a linearly reductive group",
	
	Usage => "groupIdeal L",
	
	Inputs => {
	        "L" => LinearlyReductiveAction => {"of a group with coordinate ring ", TT "S/I"},
		},
	    
	Outputs => {
		Ideal => {TT "I"}
		},
	    
	"This function is provided by the package ", TO InvariantRing,
	".",
	
	PARA { 
	    "Suppose the action ", TT "L", " consists of the linearly
	    reductive group with coordinate ring ", TT "S/I",
	    " (where ", TT "S", " is a polynomial ring) acting on 
	    a (quotient of) 
	    a polynomial ring ", TT "R", " via the action matrix ",
	    TT "M", ". This function returns the ideal ", TT "I",
	    ".",
	    },
	EXAMPLE {
		"S = QQ[z]",
		"I = ideal(z^2 - 1)",
		"M = matrix{{(z+1)/2, (1-z)/2},{(1-z)/2, (z+1)/2}}",
		"R = QQ[x,y]",
		"L = linearlyReductiveAction(I, M, R)",
		"groupIdeal L",
		},
	    }



document {
	Key => {hilbertIdeal, (hilbertIdeal, LinearlyReductiveAction)},
	
	Headline => "compute generators for the Hilbert ideal",
	
	Usage => "hilbertIdeal L",
	
	Inputs => {
	        "L" => LinearlyReductiveAction,
		},
	    
	Outputs => {
		Ideal => {"the ideal generated by all ring elements invariant under the action"}
		},
	    
	"This function is provided by the package ", TO InvariantRing,
	".",
	
	PARA { 
	    "This function computes the Hilbert ideal
	    for the action of a linearly reductive group
	    on a (quotient of a) polynomial ring, i.e., the ideal
	    generated by all ring elements of positive degree
	    invariant under the action. The algorithm is based on: ",
	    },
       	
	UL { 
	    {"Derksen, H. & Kemper, G. (2015).", EM " Computational Invariant Theory", 
	   ". Heidelberg: Springer. pp 159-164"}
        },
	
	PARA {
	    "The next example constructs a cyclic group of order 2
	    as a set of two affine points. Then it introduces an
	    action of this group on a polynomial ring in two variables
	    and computes the Hilbert ideal. The action permutes the
	    variables of the polynomial ring. Note that the
	    generators of the Hilbert ideal need not be invariant."
	    },
	
	EXAMPLE {
		"S = QQ[z]",
		"I = ideal(z^2 - 1)",
		"M = matrix{{(z+1)/2, (1-z)/2},{(1-z)/2, (z+1)/2}}",
		"sub(M,z=>1),sub(M,z=>-1)",
		"R = QQ[x,y]",
		"L = linearlyReductiveAction(I, M, R)",
		"H = hilbertIdeal L",
		"apply(H_*, f -> isInvariant(f,L))"
		},
	PARA {
	    "We offer a slight variation on the previous example
	    to illustrate this method at work on a quotient of a
	    polynomial ring."
	    },
	EXAMPLE {
		"S = QQ[z];",
		"I = ideal(z^2 - 1);",
		"M = matrix{{(z+1)/2, (1-z)/2},{(1-z)/2, (z+1)/2}};",
		"Q = QQ[x,y] / ideal(x*y)",
		"L = linearlyReductiveAction(I, M, Q)",
		"H = hilbertIdeal L"
		},
	PARA {
	    "The algorithm performs an elimination using Groebner
	    bases. The options ", TO DegreeLimit, " and ",
	    TO SubringLimit, " are standard ", TO gb, " options
	    that can be used to interrupt the computation
	    before it is complete, yielding a partial list of
	    generators for the Hilbert ideal."
	    },
    	 Caveat => "The generators of the Hilbert ideal computed
	 by this function need not be invariant."	    
	    }


document {
	Key => {linearlyReductiveAction,
	    (linearlyReductiveAction, Ideal, Matrix, PolynomialRing),
	    (linearlyReductiveAction, Ideal, Matrix, QuotientRing)},
	
	Headline => "Linearly reductive group action",
	
	Usage => "linearlyReductiveAction(I, M, R) \n linearlyReductiveAction(I, M, Q) ",
	
	Inputs => {
	        "I" => Ideal => {"of a polynomial ring ", TT "S", " defining a group as an affine variety"},
		"M" => Matrix => {"whose entries are in ", TT "S", ", that encodes the group action on ", TT "R"},
	    	"R" => PolynomialRing => {"on which the group acts"},
	    	"Q" => QuotientRing => {"on which the group acts"},
		},
	    
	Outputs => {
		LinearlyReductiveAction => {"the linearly reductive action of ", TT "S/I",
		    " on ", TT "R", " or ", TT "Q", " via the matrix ", TT "M"}
		},
	    
	"This function is provided by the package ", TO InvariantRing, ".",
    
    	PARA {
	    "In order to encode a linearly reductive group action,
	    we represent the group as an affine variety.
	    The polynomial ring ", TT "S", " is the coordinate ring
	    of the ambient affine space containing the group, 
	    while ", TT "I", " is the ideal of ", TT "S",
	    " defining the group as a subvariety. In other words,
	    the elements of the group are the points of the affine
	    variety with coordinate ring ", TT "S/I", ". ",
	    "The group acts linearly on the
	    polynomial ring ", TT "R", " via the matrix ",
	    TT "M", " with entries in ", TT "S", ".",
	},
	PARA {
	    "The next example constructs a cyclic group of order 2
	    as a set of two affine points. Then it introduces an
	    action of this group on a polynomial ring in two variables."
	    },
	EXAMPLE {
		"S = QQ[z]",
		"I = ideal(z^2 - 1)",
		"M = matrix{{(z+1)/2, (1-z)/2},{(1-z)/2, (z+1)/2}}",
		"sub(M,z=>1),sub(M,z=>-1)",
		"R = QQ[x,y]",
		"L = linearlyReductiveAction(I, M, R)",
		},
	PARA {
	    "This function is also used to define linearly reductive
	    group actions on quotients of polynomial rings.
	    We illustrate by a slight variation on the previous example."
	    },
	EXAMPLE {
		"S = QQ[z];",
		"I = ideal(z^2 - 1);",
		"M = matrix{{(z+1)/2, (1-z)/2},{(1-z)/2, (z+1)/2}};",
		"Q = QQ[x,y] / ideal(x*y)",
		"L = linearlyReductiveAction(I, M, Q)",
		},
}


document {
	Key => {LinearlyReductiveAction},
	
	Headline => "the class of all (non finite, non toric) linearly reductive group actions",
	
	"This class is provided by the package ", TO InvariantRing,".",
	
	PARA {
	    	TT "LinearlyReductiveAction", " is the class of all
		linearly reductive group actions on (quotients of)
		polynomial rings
		for the	purpose of computing invariants.
		It is created using ", TO "linearlyReductiveAction", ".",
		" This class should not be used for actions of
		tori or finite groups, as its methods for computing
		invariants are in general less efficient than
		specialized methods for the classes ",
		TO "FiniteGroupAction", ", and ", TO "DiagonalAction", "."
	    },
	}





