package org.apache.commons.openpgp;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRingCollection;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRingCollection;
import org.bouncycastle.openpgp.PGPUtil;

import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;

/**
 * Bouncy Castle implementation of the OpenPGP key ring.
 *
 * @author <a href="mailto:brett@apache.org">Brett Porter</a>
 * @todo password is not secure
 */
public class BouncyCastleKeyRing
    implements KeyRing
{
    private final PGPSecretKeyRingCollection pgpSec;

    private final char[] password;

    private final PGPPublicKeyRingCollection pgpPub;

    private static final long MASK = 0xFFFFFFFFL;

    public BouncyCastleKeyRing( InputStream secretKeyRingStream, InputStream publicKeyRingStream, char[] password )
        throws IOException, PGPException
    {
        pgpSec = new PGPSecretKeyRingCollection( PGPUtil.getDecoderStream( secretKeyRingStream ) );

        pgpPub = new PGPPublicKeyRingCollection( PGPUtil.getDecoderStream( publicKeyRingStream ) );

        this.password = password;
    }

    public char[] getPassword()
    {
        return password;
    }

    public PGPSecretKey getSecretKey( String keyId )
    {
        Iterator rIt = pgpSec.getKeyRings();

        while ( rIt.hasNext() )
        {
            PGPSecretKeyRing kRing = (PGPSecretKeyRing) rIt.next();
            Iterator kIt = kRing.getSecretKeys();

            while ( kIt.hasNext() )
            {
                PGPSecretKey k = (PGPSecretKey) kIt.next();

                // TODO: do long conversion in other direction
                if ( k.isSigningKey() && Long.toHexString( k.getKeyID() & MASK ).equals( keyId.toLowerCase() ) )
                {
                    return k;
                }
            }
        }

        return null;
    }

    public PGPPublicKey getPublicKey( String keyId )
    {
        Iterator rIt = pgpPub.getKeyRings();

        while ( rIt.hasNext() )
        {
            PGPPublicKeyRing kRing = (PGPPublicKeyRing) rIt.next();
            Iterator kIt = kRing.getPublicKeys();

            while ( kIt.hasNext() )
            {
                PGPPublicKey k = (PGPPublicKey) kIt.next();

                // TODO: do long conversion in other direction
                if ( Long.toHexString( k.getKeyID() & MASK ).equals( keyId.toLowerCase() ) )
                {
                    return k;
                }
            }
        }

        return null;
    }

    public PGPSecretKey getSecretKey( long keyId )
    {
        // TODO: switch with above
        return getSecretKey( Long.toHexString( keyId & MASK ) );
    }

    public PGPPublicKey getPublicKey( long keyId )
    {
        // TODO: switch with above
        return getPublicKey( Long.toHexString( keyId & MASK ) );
    }
}
