/*
 * Copyright (C) 2017 Francis Deslauriers <francis.deslauriers@efficios.com>
 *
 * SPDX-License-Identifier: LGPL-2.1-only
 *
 */

#include <fcntl.h>
#include <stdio.h>
#include <sys/syscall.h>
#include <unistd.h>
#include <common/error.h>
#include <common/align.h>

#include "utils.h"

#define MAX_LEN 16

/*
 * The LTTng system call tracing facilities can't handle page faults at the
 * moment. If a fault would occur while reading a syscall argument, the
 * tracer will report an empty string (""). Since the proper execution of the
 * tests which use this generator depends on some syscall string arguments being
 * present, this util allows us to mitigate the page-fault risk.
 *
 * This isn't a proper fix; it is simply the best we can do for now.
 * See bug #1261 for more context.
 */
static
void prefault_string(const char *p)
{
	const char * const end = p + strlen(p) + 1;

	while (p < end) {
		/*
		 * Trigger a read attempt on *p, faulting-in the pages
		 * for reading.
		 */
		asm volatile("" : : "m"(*p));
		p += PAGE_SIZE;
	}
}

static
int open_read_close(const char *path)
{
	int fd, ret;
	char buf[MAX_LEN];

	/*
	 * Start generating syscalls. We use syscall(2) to prevent libc from
	 * changing the underlying syscall (e.g. calling openat(2) instead of
	 * open(2)).
	 */
	prefault_string(path);
	fd = syscall(SYS_openat, AT_FDCWD, path, O_RDONLY);
	if (fd < 0) {
		PERROR_NO_LOGGER("Failed to open file with openat(): path = '%s'", path);
		ret = -1;
		goto error;
	}

	ret = syscall(SYS_read, fd, buf, MAX_LEN);
	if (ret < 0) {
		PERROR_NO_LOGGER("Failed to read file: path = '%s', fd = %d, length = %d",
				path, fd, MAX_LEN);
		ret = -1;
		goto error;
	}

	ret = syscall(SYS_close, fd);
	if (ret == -1) {
		PERROR_NO_LOGGER("Failed to close file: path = '%s', fd = %d", path, fd);
		ret = -1;
		goto error;
	}

error:
	return ret;
}

/*
 * The process waits for the creation of a file passed as argument from an
 * external processes to execute a syscall and exiting. This is useful for tests
 * in combinaison with LTTng's PID tracker feature where we can trace the kernel
 * events generated by our test process only.
 */
int main(int argc, char **argv)
{
	int ret;
	const char *start_file, *path1, *path2;

	if (argc != 4) {
		fprintf(stderr, "Error: Missing argument\n");
		fprintf(stderr, "USAGE: %s PATH_WAIT_FILE PATH1_TO_OPEN PATH2_TO_OPEN\n", argv[0]);
		ret = -1;
		goto error;
	}

	start_file = argv[1];
	path1 = argv[2];
	path2 = argv[3];

	/*
	 * Wait for the start_file to be created by an external process
	 * (typically the test script) before executing the syscalls.
	 */
	ret = wait_on_file(start_file);
	if (ret != 0) {
		goto error;
	}

	/*
	 * Start generating syscalls. We use syscall(2) to prevent libc to change
	 * the underlying syscall. e.g. calling openat(2) instead of open(2).
	 */
	ret = open_read_close(path1);
	if (ret == -1) {
		ret = -1;
		goto error;
	}

	ret = open_read_close(path2);
	if (ret == -1) {
		ret = -1;
		goto error;
	}

error:
	return ret;
}
