// Protocol Buffers - Google's data interchange format
// Copyright 2008 Google Inc.  All rights reserved.
// http://code.google.com/p/protobuf/
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// Author: kenton@google.com (Kenton Varda) and others
// Modified by Michal Brzozowski to support non-void callbacks.

#ifndef CALLBACK_H_
#define CALLBACK_H_

#define GOOGLE_DISALLOW_EVIL_CONSTRUCTORS(TypeName)    \
  TypeName(const TypeName&);                           \
  void operator=(const TypeName&)

class Closure {
 public:
  Closure() {}
  virtual ~Closure(){}

  virtual void Run() = 0;

 private:
  GOOGLE_DISALLOW_EVIL_CONSTRUCTORS(Closure);
};

void DoNothing(){}

namespace internal {

class FunctionClosure0 : public Closure {
 public:
  typedef void (*FunctionType)();

  FunctionClosure0(FunctionType function, bool self_deleting)
    : function_(function), self_deleting_(self_deleting) {}
  ~FunctionClosure0(){}

  void Run() {
    function_();
    if (self_deleting_) delete this;
  }

 private:
  FunctionType function_;
  bool self_deleting_;
};

template <typename Class>
class MethodClosure0 : public Closure {
 public:
  typedef void (Class::*MethodType)();

  MethodClosure0(Class* object, MethodType method, bool self_deleting)
    : object_(object), method_(method), self_deleting_(self_deleting) {}
  ~MethodClosure0() {}

  void Run() {
    (object_->*method_)();
    if (self_deleting_) delete this;
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
};

template <typename Arg1>
class FunctionClosure1 : public Closure {
 public:
  typedef void (*FunctionType)(Arg1 arg1);

  FunctionClosure1(FunctionType function, bool self_deleting,
                   Arg1 arg1)
    : function_(function), self_deleting_(self_deleting),
      arg1_(arg1) {}
  ~FunctionClosure1() {}

  void Run() {
    function_(arg1_);
    if (self_deleting_) delete this;
  }

 private:
  FunctionType function_;
  bool self_deleting_;
  Arg1 arg1_;
};

template <typename Class, typename Arg1>
class MethodClosure1 : public Closure {
 public:
  typedef void (Class::*MethodType)(Arg1 arg1);

  MethodClosure1(Class* object, MethodType method, bool self_deleting,
                 Arg1 arg1)
    : object_(object), method_(method), self_deleting_(self_deleting),
      arg1_(arg1) {}
  ~MethodClosure1() {}

  void Run() {
    (object_->*method_)(arg1_);
    if (self_deleting_) delete this;
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
  Arg1 arg1_;
};

template <typename Arg1, typename Arg2>
class FunctionClosure2 : public Closure {
 public:
  typedef void (*FunctionType)(Arg1 arg1, Arg2 arg2);

  FunctionClosure2(FunctionType function, bool self_deleting,
                   Arg1 arg1, Arg2 arg2)
    : function_(function), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2) {}
  ~FunctionClosure2() {}

  void Run() {
    function_(arg1_, arg2_);
    if (self_deleting_) delete this;
  }

 private:
  FunctionType function_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
};

template <typename Class, typename Arg1, typename Arg2>
class MethodClosure2 : public Closure {
 public:
  typedef void (Class::*MethodType)(Arg1 arg1, Arg2 arg2);

  MethodClosure2(Class* object, MethodType method, bool self_deleting,
                 Arg1 arg1, Arg2 arg2)
    : object_(object), method_(method), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2) {}
  ~MethodClosure2() {}

  void Run() {
    (object_->*method_)(arg1_, arg2_);
    if (self_deleting_) delete this;
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
};

template <typename Arg1, typename Arg2, typename Arg3>
class FunctionClosure3 : public Closure {
 public:
  typedef void (*FunctionType)(Arg1 arg1, Arg2 arg2, Arg3 arg3);

  FunctionClosure3(FunctionType function, bool self_deleting,
                   Arg1 arg1, Arg2 arg2, Arg3 arg3)
    : function_(function), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2), arg3_(arg3) {}
  ~FunctionClosure3() {}

  void Run() {
    function_(arg1_, arg2_, arg3_);
    if (self_deleting_) delete this;
  }

 private:
  FunctionType function_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
  Arg3 arg3_;
};

template <typename Class, typename Arg1, typename Arg2, typename Arg3>
class MethodClosure3 : public Closure {
 public:
  typedef void (Class::*MethodType)(Arg1 arg1, Arg2 arg2, Arg3 arg3);

  MethodClosure3(Class* object, MethodType method, bool self_deleting,
                 Arg1 arg1, Arg2 arg2, Arg3 arg3)
    : object_(object), method_(method), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2), arg3_(arg3) {}
  ~MethodClosure3() {}

  void Run() {
    (object_->*method_)(arg1_, arg2_, arg3_);
    if (self_deleting_) delete this;
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
  Arg3 arg3_;
};

template <typename Arg1, typename Arg2, typename Arg3, typename Arg4>
class FunctionClosure4 : public Closure {
 public:
  typedef void (*FunctionType)(Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4);

  FunctionClosure4(FunctionType function, bool self_deleting,
                   Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4)
    : function_(function), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2), arg3_(arg3), arg4_(arg4) {}
  ~FunctionClosure4() {}

  void Run() {
    function_(arg1_, arg2_, arg3_, arg4_);
    if (self_deleting_) delete this;
  }

 private:
  FunctionType function_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
  Arg3 arg3_;
  Arg4 arg4_;
};

template <typename Class, typename Arg1, typename Arg2, typename Arg3, typename Arg4>
class MethodClosure4 : public Closure {
 public:
  typedef void (Class::*MethodType)(Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4);

  MethodClosure4(Class* object, MethodType method, bool self_deleting,
                 Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4)
    : object_(object), method_(method), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2), arg3_(arg3), arg4_(arg4) {}
  ~MethodClosure4() {}

  void Run() {
    (object_->*method_)(arg1_, arg2_, arg3_, arg4_);
    if (self_deleting_) delete this;
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
  Arg3 arg3_;
  Arg4 arg4_;
};

}  // namespace internal

// See Closure.
inline Closure* NewCallback(void (*function)()) {
  return new internal::FunctionClosure0(function, true);
}

// See Closure.
inline Closure* NewPermanentCallback(void (*function)()) {
  return new internal::FunctionClosure0(function, false);
}

// See Closure.
template <typename Class>
inline Closure* NewCallback(Class* object, void (Class::*method)()) {
  return new internal::MethodClosure0<Class>(object, method, true);
}

// See Closure.
template <typename Class>
inline Closure* NewPermanentCallback(Class* object, void (Class::*method)()) {
  return new internal::MethodClosure0<Class>(object, method, false);
}

// See Closure.
template <typename Arg1>
inline Closure* NewCallback(void (*function)(Arg1),
                            Arg1 arg1) {
  return new internal::FunctionClosure1<Arg1>(function, true, arg1);
}

// See Closure.
template <typename Arg1>
inline Closure* NewPermanentCallback(void (*function)(Arg1),
                                     Arg1 arg1) {
  return new internal::FunctionClosure1<Arg1>(function, false, arg1);
}

// See Closure.
template <typename Class, typename Arg1>
inline Closure* NewCallback(Class* object, void (Class::*method)(Arg1),
                            Arg1 arg1) {
  return new internal::MethodClosure1<Class, Arg1>(object, method, true, arg1);
}

// See Closure.
template <typename Class, typename Arg1>
inline Closure* NewPermanentCallback(Class* object, void (Class::*method)(Arg1),
                                     Arg1 arg1) {
  return new internal::MethodClosure1<Class, Arg1>(object, method, false, arg1);
}

// See Closure.
template <typename Arg1, typename Arg2>
inline Closure* NewCallback(void (*function)(Arg1, Arg2),
                            Arg1 arg1, Arg2 arg2) {
  return new internal::FunctionClosure2<Arg1, Arg2>(
    function, true, arg1, arg2);
}

// See Closure.
template <typename Arg1, typename Arg2>
inline Closure* NewPermanentCallback(void (*function)(Arg1, Arg2),
                                     Arg1 arg1, Arg2 arg2) {
  return new internal::FunctionClosure2<Arg1, Arg2>(
    function, false, arg1, arg2);
}

// See Closure.
template <typename Class, typename Arg1, typename Arg2>
inline Closure* NewCallback(Class* object, void (Class::*method)(Arg1, Arg2),
                            Arg1 arg1, Arg2 arg2) {
  return new internal::MethodClosure2<Class, Arg1, Arg2>(
    object, method, true, arg1, arg2);
}

// See Closure.
template <typename Class, typename Arg1, typename Arg2>
inline Closure* NewPermanentCallback(
    Class* object, void (Class::*method)(Arg1, Arg2),
    Arg1 arg1, Arg2 arg2) {
  return new internal::MethodClosure2<Class, Arg1, Arg2>(
    object, method, false, arg1, arg2);
}

// See Closure.
template <typename Arg1, typename Arg2, typename Arg3>
inline Closure* NewCallback(void (*function)(Arg1, Arg2, Arg3),
                            Arg1 arg1, Arg2 arg2, Arg3 arg3) {
  return new internal::FunctionClosure3<Arg1, Arg2, Arg3>(
    function, true, arg1, arg2, arg3);
}

// See Closure.
template <typename Arg1, typename Arg2, typename Arg3>
inline Closure* NewPermanentCallback(void (*function)(Arg1, Arg2, Arg3),
                                     Arg1 arg1, Arg2 arg2, Arg3 arg3) {
  return new internal::FunctionClosure3<Arg1, Arg2, Arg3>(
    function, false, arg1, arg2, arg3);
}

// See Closure.
template <typename Class, typename Arg1, typename Arg2, typename Arg3>
inline Closure* NewCallback(Class* object, void (Class::*method)(Arg1, Arg2, Arg3),
                            Arg1 arg1, Arg2 arg2, Arg3 arg3) {
  return new internal::MethodClosure3<Class, Arg1, Arg2, Arg3>(
    object, method, true, arg1, arg2, arg3);
}

// See Closure.
template <typename Class, typename Arg1, typename Arg2, typename Arg3>
inline Closure* NewPermanentCallback(
    Class* object, void (Class::*method)(Arg1, Arg2, Arg3),
		   Arg1 arg1, Arg2 arg2, Arg3 arg3) {
  return new internal::MethodClosure3<Class, Arg1, Arg2, Arg3>(
    object, method, false, arg1, arg2, arg3);
}

// See Closure.
template <typename Arg1, typename Arg2, typename Arg3, typename Arg4>
inline Closure* NewCallback(void (*function)(Arg1, Arg2, Arg3, Arg4),
                            Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4) {
  return new internal::FunctionClosure4<Arg1, Arg2, Arg3, Arg4>(
    function, true, arg1, arg2, arg3, arg4);
}

// See Closure.
template <typename Arg1, typename Arg2, typename Arg3, typename Arg4>
inline Closure* NewPermanentCallback(void (*function)(Arg1, Arg2, Arg3, Arg4),
                                     Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4) {
  return new internal::FunctionClosure4<Arg1, Arg2, Arg3, Arg4>(
    function, false, arg1, arg2, arg3, arg4);
}

// See Closure.
template <typename Class, typename Arg1, typename Arg2, typename Arg3, typename Arg4>
inline Closure* NewCallback(Class* object, void (Class::*method)(Arg1, Arg2, Arg3, Arg4),
                            Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4) {
  return new internal::MethodClosure4<Class, Arg1, Arg2, Arg3, Arg4>(
    object, method, true, arg1, arg2, arg3, arg4);
}

// See Closure.
template <typename Class, typename Arg1, typename Arg2, typename Arg3, typename Arg4>
inline Closure* NewPermanentCallback(
    Class* object, void (Class::*method)(Arg1, Arg2, Arg3, Arg4),
		   Arg1 arg1, Arg2 arg2, Arg3 arg3, Arg4 arg4) {
  return new internal::MethodClosure4<Class, Arg1, Arg2, Arg3, Arg4>(
    object, method, false, arg1, arg2, arg3, arg4);
}

template <typename Return>
class ReturnClosure {
 public:
  ReturnClosure() {}
  virtual ~ReturnClosure(){}

  virtual Return Run() = 0;

 private:
  GOOGLE_DISALLOW_EVIL_CONSTRUCTORS(ReturnClosure);
};

namespace internal {

template <typename PointerType>
class Deleter {
 public:
  Deleter(PointerType p):pointer_(p){}
  ~Deleter(){
    delete pointer_;
  }

 private:
  PointerType pointer_;
};

template <typename PointerType>
Deleter<PointerType> MakeDeleter(PointerType p){
  return Deleter<PointerType>(p);
}

template <typename Return>
class FunctionReturnClosure0 : public ReturnClosure<Return> {
 public:
  typedef Return (*FunctionType)();

  FunctionReturnClosure0(FunctionType function, bool self_deleting)
    : function_(function), self_deleting_(self_deleting) {}
  ~FunctionReturnClosure0(){}

  Return Run() {
    MakeDeleter(self_deleting_ ? this : 0);
    return function_();
  }

 private:
  FunctionType function_;
  bool self_deleting_;
};

template <typename Return, typename Class>
class MethodReturnClosure0 : public ReturnClosure<Return> {
 public:
  typedef Return (Class::*MethodType)();

  MethodReturnClosure0(Class* object, MethodType method, bool self_deleting)
    : object_(object), method_(method), self_deleting_(self_deleting) {}
  ~MethodReturnClosure0() {}

  Return Run() {
    MakeDeleter(self_deleting_ ? this : 0);
    return (object_->*method_)();
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
};

template <typename Return, typename Arg1>
class FunctionReturnClosure1 : public ReturnClosure<Return> {
 public:
  typedef Return (*FunctionType)(Arg1 arg1);

  FunctionReturnClosure1(FunctionType function, bool self_deleting,
                   Arg1 arg1)
    : function_(function), self_deleting_(self_deleting),
      arg1_(arg1) {}
  ~FunctionReturnClosure1() {}

  Return Run() {
    MakeDeleter(self_deleting_ ? this : 0);
    return function_(arg1_);
  }

 private:
  FunctionType function_;
  bool self_deleting_;
  Arg1 arg1_;
};

template <typename Return, typename Class, typename Arg1>
class MethodReturnClosure1 : public ReturnClosure<Return> {
 public:
  typedef Return (Class::*MethodType)(Arg1 arg1);

  MethodReturnClosure1(Class* object, MethodType method, bool self_deleting,
                 Arg1 arg1)
    : object_(object), method_(method), self_deleting_(self_deleting),
      arg1_(arg1) {}
  ~MethodReturnClosure1() {}

  Return Run() {
    MakeDeleter(self_deleting_ ? this : 0);
    return (object_->*method_)(arg1_);
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
  Arg1 arg1_;
};

template <typename Return, typename Arg1, typename Arg2>
class FunctionReturnClosure2 : public ReturnClosure<Return> {
 public:
  typedef Return (*FunctionType)(Arg1 arg1, Arg2 arg2);

  FunctionReturnClosure2(FunctionType function, bool self_deleting,
                   Arg1 arg1, Arg2 arg2)
    : function_(function), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2) {}
  ~FunctionReturnClosure2() {}

  Return Run() {
    MakeDeleter(self_deleting_ ? this : 0);
    return function_(arg1_, arg2_);
  }

 private:
  FunctionType function_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
};

template <typename Return, typename Class, typename Arg1, typename Arg2>
class MethodReturnClosure2 : public ReturnClosure<Return> {
 public:
  typedef Return (Class::*MethodType)(Arg1 arg1, Arg2 arg2);

  MethodReturnClosure2(Class* object, MethodType method, bool self_deleting,
                 Arg1 arg1, Arg2 arg2)
    : object_(object), method_(method), self_deleting_(self_deleting),
      arg1_(arg1), arg2_(arg2) {}
  ~MethodReturnClosure2() {}

  Return Run() {
    MakeDeleter(self_deleting_ ? this : 0);
    return (object_->*method_)(arg1_, arg2_);
  }

 private:
  Class* object_;
  MethodType method_;
  bool self_deleting_;
  Arg1 arg1_;
  Arg2 arg2_;
};

}  // namespace internal

// See Closure.
template <typename Return>
inline ReturnClosure<Return>* NewReturnCallback(Return (*function)()) {
  return new internal::FunctionReturnClosure0<Return>(function, true);
}

// See Closure.
template <typename Return>
inline ReturnClosure<Return>* NewPermanentReturnCallback(Return (*function)()) {
  return new internal::FunctionReturnClosure0<Return>(function, false);
}

// See Closure.
template <typename Return, typename Class>
inline ReturnClosure<Return>* NewReturnCallback(Class* object, Return (Class::*method)()) {
  return new internal::MethodReturnClosure0<Return, Class>(object, method, true);
}

// See Closure.
template <typename Return, typename Class>
inline ReturnClosure<Return>* NewPermanentReturnCallback(Class* object, Return (Class::*method)()) {
  return new internal::MethodReturnClosure0<Return, Class>(object, method, false);
}

// See Closure.
template <typename Return, typename Arg1>
inline ReturnClosure<Return>* NewReturnCallback(Return (*function)(Arg1),
                            Arg1 arg1) {
  return new internal::FunctionReturnClosure1<Return, Arg1>(function, true, arg1);
}

// See Closure.
template <typename Return, typename Arg1>
inline ReturnClosure<Return>* NewPermanentReturnCallback(Return (*function)(Arg1),
                                     Arg1 arg1) {
  return new internal::FunctionReturnClosure1<Return, Arg1>(function, false, arg1);
}

// See Closure.
template <typename Return, typename Class, typename Arg1>
inline ReturnClosure<Return>* NewReturnCallback(Class* object, Return (Class::*method)(Arg1),
                            Arg1 arg1) {
  return new internal::MethodReturnClosure1<Return, Class, Arg1>(object, method, true, arg1);
}

// See Closure.
template <typename Return, typename Class, typename Arg1>
inline ReturnClosure<Return>* NewPermanentReturnCallback(Class* object, Return (Class::*method)(Arg1),
                                     Arg1 arg1) {
  return new internal::MethodReturnClosure1<Return, Class, Arg1>(object, method, false, arg1);
}

// See Closure.
template <typename Return, typename Arg1, typename Arg2>
inline ReturnClosure<Return>* NewReturnCallback(Return (*function)(Arg1, Arg2),
                            Arg1 arg1, Arg2 arg2) {
  return new internal::FunctionReturnClosure2<Return, Arg1, Arg2>(
    function, true, arg1, arg2);
}

// See Closure.
template <typename Return, typename Arg1, typename Arg2>
inline ReturnClosure<Return>* NewPermanentReturnCallback(Return (*function)(Arg1, Arg2),
                                     Arg1 arg1, Arg2 arg2) {
  return new internal::FunctionReturnClosure2<Return, Arg1, Arg2>(
    function, false, arg1, arg2);
}

// See Closure.
template <typename Return, typename Class, typename Arg1, typename Arg2>
inline Closure* NewReturnCallback(Class* object, Return (Class::*method)(Arg1, Arg2),
                            Arg1 arg1, Arg2 arg2) {
  return new internal::MethodReturnClosure2<Return, Class, Arg1, Arg2>(
    object, method, true, arg1, arg2);
}

// See Closure.
template <typename Return, typename Class, typename Arg1, typename Arg2>
inline Closure* NewReturnPermanentCallback(
    Class* object, Return (Class::*method)(Arg1, Arg2),
    Arg1 arg1, Arg2 arg2) {
  return new internal::MethodReturnClosure2<Return, Class, Arg1, Arg2>(
    object, method, false, arg1, arg2);
}


#endif /*CALLBACK_H_*/
