#include "LaunchtoolOutput.h"
#include <syslog.h>
#include <errno.h>

using namespace std;

void StdoutOutput::out(const string& str) throw ()
{
	fwrite(str.data(), str.size(), 1, stdout);
	fputc('\n', stdout);
}

void StderrOutput::out(const string& str) throw ()
{
	fwrite(str.data(), str.size(), 1, stderr);
	fputc('\n', stderr);
}

FileOutput::FileOutput(const string& file) throw (FileException)
	: oh(0)
{
	oh = fopen(file.c_str(), "at");
	if (!oh)
		throw FileException(errno, "Trying to open " + file + " for appending");
}

FileOutput::~FileOutput()
{
	if (oh)
		fclose(oh);
}

void FileOutput::out(const string& str) throw ()
{
	fwrite(str.data(), str.size(), 1, oh);
	fputc('\n', oh);
	fflush(oh);
}

void SyslogOutput::open() throw ()
{
	openlog(_identity.c_str(), LOG_PID, _facility);
}

SyslogOutput::SyslogOutput(const string& identity, const string& facility, const string& level)
	throw () : _identity(identity)
{
	if (facility == "LOG_AUTH")
		_facility = LOG_AUTH;
	else if (facility == "LOG_AUTHPRIV")
		_facility = LOG_AUTHPRIV;
	else if (facility == "LOG_CRON")
		_facility = LOG_CRON;
	else if (facility == "LOG_DAEMON")
		_facility = LOG_DAEMON;
	else if (facility == "LOG_FTP")
		_facility = LOG_FTP;
	else if (facility == "LOG_KERN")
		_facility = LOG_KERN;
	else if (facility == "LOG_LPR")
		_facility = LOG_LPR;
	else if (facility == "LOG_MAIL")
		_facility = LOG_MAIL;
	else if (facility == "LOG_NEWS")
		_facility = LOG_NEWS;
	else if (facility == "LOG_SYSLOG")
		_facility = LOG_SYSLOG;
	else if (facility == "LOG_USER")
		_facility = LOG_USER;
	else if (facility == "LOG_UUCP")
		_facility = LOG_UUCP;
	else if (facility == "LOG_LOCAL0")
		_facility = LOG_LOCAL0;
	else if (facility == "LOG_LOCAL1")
		_facility = LOG_LOCAL1;
	else if (facility == "LOG_LOCAL2")
		_facility = LOG_LOCAL2;
	else if (facility == "LOG_LOCAL3")
		_facility = LOG_LOCAL3;
	else if (facility == "LOG_LOCAL4")
		_facility = LOG_LOCAL4;
	else if (facility == "LOG_LOCAL5")
		_facility = LOG_LOCAL5;
	else if (facility == "LOG_LOCAL6")
		_facility = LOG_LOCAL6;
	else if (facility == "LOG_LOCAL7")
		_facility = LOG_LOCAL7;
	else
		_facility = LOG_USER;

	if (level == "LOG_EMERG")
		_prio = LOG_ALERT;
	else if (level == "LOG_CRIT")
		_prio = LOG_CRIT;
	else if (level == "LOG_ERR")
		_prio = LOG_ERR;
	else if (level == "LOG_WARNING")
		_prio = LOG_WARNING;
	else if (level == "LOG_NOTICE")
		_prio = LOG_NOTICE;
	else if (level == "LOG_INFO")
		_prio = LOG_INFO;
	else if (level == "LOG_DEBUG")
		_prio = LOG_DEBUG;
	else
		_prio = LOG_WARNING;

	open();
}

SyslogOutput::~SyslogOutput()
{
	closelog();
}

void SyslogOutput::out(const string& str) throw ()
{
	syslog(_prio, "%.*s\n", PFSTR(str));
}

OutputMethod* Output::_lauout = 0;
OutputMethod* Output::_lauerr = 0;
OutputMethod* Output::_cldout = 0;
OutputMethod* Output::_clderr = 0;

OutputMethod* Output::getMethod(const string& str) throw (ConsistencyCheckException)
{
	size_t a = str.find(':');
	if (a == string::npos)
		if (str == "ignore")
			return 0;
		else if (str == "stdout")
			return new StdoutOutput();
		else if (str == "stderr")
			return new StderrOutput();
		else
			throw ConsistencyCheckException("Unknown output method: " + str);

	string tmp = str.substr(0, a);
	if (tmp == "file")
	{
		try {
			return new FileOutput(str.substr(a + 1));
		} catch (Exception& e) {
			throw ConsistencyCheckException(string("FileOutput did not initialize because of a ") + e.type() + ": " + e.desc());
		}
	}
	else if (tmp == "syslog")
	{
		tmp = str.substr(a + 1);
		a = tmp.find(',');
		if (a == string::npos)
			throw ConsistencyCheckException("No facility provided for log method: " + str);
		string identity = tmp.substr(0, a);
		tmp = tmp.substr(a + 1);
		a = tmp.find(',');
		if (a == string::npos)
			throw ConsistencyCheckException("No level provided for log method: " + str);
		string facility = tmp.substr(0, a);
		return new SyslogOutput(identity, facility, tmp.substr(a + 1));
	} else
		throw ConsistencyCheckException("Unknown output method: " + str);
}


// vim:set ts=4 sw=4:
