/* $Id: network.h,v 1.14 2003/06/14 12:41:20 sjoerd Exp $ */
#ifndef FFRENZY_NETWORK_H
#define FFRENZY_NETWORK_H

#include <stdio.h>
#include <sys/socket.h>

#include "global.h"
#include "list.h"

/** \defgroup network network
 * Network interface
 */

/** \ingroup network
 * A network packet
 */
typedef struct {
  int size; /**< size of the \p data */
  char *data; /**< the data */
  int  refcount; /**< reference counter, for internal use only */
} Network_packet;

/** \ingroup network
 * Status codes of a network connection
 */
typedef enum {
  NW_ERROR, /**< An error has occurred */
  NW_OK /**< everything OK */
} Network_state;

/* a connection */
typedef struct {
  int fd;               /* socket file descriptor */
  char *hostname;       /* hostname of the remote host in numeric form */
  char *localhost;     /* hostname of the local side of the connection */
  char *shostname;     /* hostname of the remote host in human readable form */
  List *in_queue;       /* queue of input packets */
  List *out_queue;      /* queue of output packets */

  Network_packet *inp;  /* package that is being recieved */
  int bytes_received;   /* number of bytes already received of current packet*/ 
  uint32_t size;

  Network_packet *outp; /* package that is being sent */
  int bytes_sent;       /* number of bytes already sent of current packet */ 
} Network_connection;

typedef struct {
  int fd;               /* socket of the file descriptor */ 
  List *new_connection; /* List of new connections */
} Network_listener;

/** \ingroup network
 * Create a new network packet of \a size. 
 */
Network_packet *new_network_packet(int size);

/** \ingroup network
 *  free a network packet
 */
void del_network_packet(Network_packet *packet);

/** \ingroup network 
 * Network_listener constructor.
 * Starts listening on \a port and address family \a ai_family. 
 */
Network_listener *new_network_listener(int port, int ai_family);

/** \ingroup network
 * Network_listener destructor.
 * Stops the listener. Note that al non-popped new connections will be closed
 */
void del_network_listener(Network_listener *listener);

/** \ingroup network
 * Get a new connection from \a listener. \c NULL is returned when there are no
 * new connection waiting
 */
Network_connection *network_listener_pop(Network_listener *listener);

/** \ingroup network
 * Network_connection constructor.
 * Creates a new connection to \a hostname on \a port. Returns \c NULL if the
 * connection failed
 */
Network_connection *new_network_connection(char *hostname,int port);

/** \ingroup network
 * Network_connection destructor.
 * Closes the Network_connection \a con. All non-popped messages and unsent
 * data will be lost! 
 */
void del_network_connection(Network_connection *con);

/** \ingroup network
 * Get a packet from the connection. \c NULL is returned when there is no data
 * waiting 
 */
Network_packet *network_connection_pop(Network_connection *con);

/** \ingroup network
 * Send \a data of \a size through the network connection.
 * Returns \c TRUE on success or FALSE it the connection is close already
 * \warning don't change a network packet after sending it
 */
int network_connection_sent(Network_connection *con, Network_packet *packet);

/** \ingroup network
 * Get the current state of a network connection 
 */
Network_state network_connection_state(Network_connection *con);

char *network_connection_get_remote(Network_connection *con);
char *network_connection_get_sremote(Network_connection *con);
char *network_connection_get_local(Network_connection *con);

/** \ingroup network
 * Update information on all Network_listeners \a listeners and
 * Network_connections \a connections.
 * Returns after some data has been received of more then \a timeout
 * microseconds if \a timeout > 0
 */
int network_update(List *listeners,List *connections,int timeout);

#endif /* FFRENZY_NETWORK_H */
