/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file fax_phone.c
 * \brief Fax and softphone routines
 */

#include <ffgtk.h>
#include <faxophone/faxophone.h>
#include <faxophone/phone.h>
#include <faxophone/fax.h>

static struct sSession *psSession = NULL;
static void *pAudioPriv = NULL;
static struct sCapiConnection *psPhoneConnection = NULL;
static struct sCapiConnection *psFaxConnection = NULL;

static GtkWidget *psStatusLabel, *psPageNumber, *psBadRowsNumber, *psErrorLabel,
	*psEncodingNumber, *psBitrateNumber, *psOkButton, *psCancelButton, *psProgressBar,
	*psRemoteLabel, *psRemoteName;
static char *pnCurrentFile = NULL;
static int nCurrentBitrate = 0;
static int nCurrentEcm = 0;
static int nCurrentController = 0;
static gchar *pnCurrentNumber = NULL;
static gchar *pnCurrentSenderMsn = NULL;
static gchar *pnCurrentSenderNumber = NULL;
static gchar *pnCurrentSenderName = NULL;
static int nCurrentCallAnonymous = 0;
static gboolean bSending = FALSE;
static int nTimer = 0;
static char *pnCurrentReportDir = NULL;
static GtkWidget *psSendWindow = NULL;

static gboolean bDtmfCheck = FALSE;

/**
 * \brief Convert error id into string (TODO: set routine with translation to libcapi?)
 * \param nReason error id
 * \return error in string
 */
char *capiInfo2String( int nReason ) {
	switch ( nReason ) {
		/*-- informative values (corresponding message was processed) -----*/
		case 0x0001:
			return _( "NCPI not supported by current protocol, NCPI ignored" );
		case 0x0002:
			return _( "Flags not supported by current protocol, flags ignored" );
		case 0x0003:
		   return _( "Alert already sent by another application" );

		/*-- error information concerning CAPI_REGISTER -----*/
		case 0x1001:
		   return _( "Too many applications" );
		case 0x1002:
		   return _( "Logical block size too small, must be at least 128 Bytes" );
		case 0x1003:
		   return _( "Buffer exceeds 64 kByte" );
		case 0x1004:
		   return _( "Message buffer size too small, must be at least 1024 Bytes" );
		case 0x1005:
		   return _( "Max. number of logical connections not supported" );
		case 0x1006:
		   return _( "Reserved" );
		case 0x1007:
		   return _( "The message could not be accepted because of an internal busy condition" );
		case 0x1008:
		   return _( "OS resource error (no memory ?)" );
		case 0x1009:
		   return _( "CAPI not installed" );
		case 0x100A:
		   return _( "Controller does not support external equipment" );
		case 0x100B:
		   return _( "Controller does only support external equipment" );

		/*-- error information concerning message exchange functions -----*/
		case 0x1101:
		   return _( "Illegal application number" );
		case 0x1102:
		   return _( "Illegal command or subcommand or message length less than 12 bytes" );
		case 0x1103:
		   return _( "The message could not be accepted because of a queue full condition !! The error code does not imply that CAPI cannot receive messages directed to another controller, PLCI or NCCI" );
		case 0x1104:
		   return _( "Queue is empty" );
		case 0x1105:
		   return _( "Queue overflow, a message was lost !! This indicates a configuration error. The only recovery from this error is to perform a CAPI_RELEASE" );
		case 0x1106:
		   return _( "Unknown notification parameter" );
		case 0x1107:
		   return _( "The Message could not be accepted because of an internal busy condition" );
		case 0x1108:
		   return _( "OS Resource error (no memory ?)" );
		case 0x1109:
		   return _( "CAPI not installed" );
		case 0x110A:
		   return _( "Controller does not support external equipment" );
		case 0x110B:
		   return _( "Controller does only support external equipment" );

		/*-- error information concerning resource / coding problems -----*/
		case 0x2001:
		   return _( "Message not supported in current state" );
		case 0x2002:
		   return _( "Illegal Controller / PLCI / NCCI" );
		case 0x2003:
		   return _( "Out of PLCI" );
		case 0x2004:
		   return _( "Out of NCCI" );
		case 0x2005:
		   return _( "Out of LISTEN" );
		case 0x2006:
		   return _( "Out of FAX resources (protocol T.30)" );
		case 0x2007:
		   return _( "Illegal message parameter coding" );

		/*-- error information concerning requested services  -----*/
		case 0x3001:
		   return _( "B1 protocol not supported" );
		case 0x3002: 
		   return _( "B2 protocol not supported" );
		case 0x3003: 
		   return _( "B3 protocol not supported" );
		case 0x3004: 
		   return _( "B1 protocol parameter not supported" );
		case 0x3005: 
		   return _( "B2 protocol parameter not supported" );
		case 0x3006: 
		   return _( "B3 protocol parameter not supported" );
		case 0x3007: 
		   return _( "B protocol combination not supported" );
		case 0x3008: 
		   return _( "NCPI not supported" );
		case 0x3009: 
		   return _( "CIP Value unknown" );
		case 0x300A: 
		   return _( "Flags not supported (reserved bits)" );
		case 0x300B: 
		   return _( "Facility not supported" );
		case 0x300C: 
		   return _( "Data length not supported by current protocol" );
		case 0x300D: 
		   return _( "Reset procedure not supported by current protocol" );

		/*-- informations about the clearing of a physical connection -----*/
		case 0x3301: 
		   return _( "Protocol error layer 1 (broken line or B-channel removed by signalling protocol)" );
		case 0x3302: 
		   return _( "Protocol error layer 2" );
		case 0x3303: 
		   return _( "Protocol error layer 3" );
		case 0x3304: 
		   return _( "Another application got that call" );

		/*-- T.30 specific reasons -----*/
		case 0x3311: 
		   return _( "Connecting not successful (remote station is no FAX G3 machine)" );
		case 0x3312: 
		   return _( "Connecting not successful (training error)" );
		case 0x3313: 
		   return _( "Disconnected before transfer (remote station does not support transfer mode, e.g. resolution)" );
		case 0x3314: 
		   return _( "Disconnected during transfer (remote abort)" );
		case 0x3315: 
		   return _( "Disconnected during transfer (remote procedure error, e.g. unsuccessful repetition of T.30 commands)" );
		case 0x3316: 
		   return _( "Disconnected during transfer (local tx data underrun)" );
		case 0x3317: 
		   return _( "Disconnected during transfer (local rx data overflow)" );
		case 0x3318: 
		   return _( "Disconnected during transfer (local abort)" );
		case 0x3319: 
		   return _( "Illegal parameter coding (e.g. SFF coding error)" );

		/*-- disconnect causes from the network according to ETS 300 102-1/Q.931 -----*/
		case 0x3481: return _( "Unallocated (unassigned) number" );
		case 0x3482: return _( "No route to specified transit network" );
		case 0x3483: return _( "No route to destination" );
		case 0x3486: return _( "Channel unacceptable" );
		case 0x3487: 
		   return _( "Call awarded and being delivered in an established channel" );
		case 0x3490: return _( "Normal call clearing" );
		case 0x3491: return _( "User busy" );
		case 0x3492: return _( "No user responding" );
		case 0x3493: return _( "No answer from user (user alerted)" );
		case 0x3495: return _( "Call rejected" );
		case 0x3496: return _( "Number changed" );
		case 0x349A: return _( "Non-selected user clearing" );
		case 0x349B: return _( "Destination out of order" );
		case 0x349C: return _( "Invalid number format" );
		case 0x349D: return _( "Facility rejected" );
		case 0x349E: return _( "Response to STATUS ENQUIRY" );
		case 0x349F: return _( "Normal, unspecified" );
		case 0x34A2: return _( "No circuit / channel available" );
		case 0x34A6: return _( "Network out of order" );
		case 0x34A9: return _( "Temporary failure" );
		case 0x34AA: return _( "Switching equipment congestion" );
		case 0x34AB: return _( "Access information discarded" );
		case 0x34AC: return _( "Requested circuit / channel not available" );
		case 0x34AF: return _( "Resources unavailable, unspecified" );
		case 0x34B1: return _( "Quality of service unavailable" );
		case 0x34B2: return _( "Requested facility not subscribed" );
		case 0x34B9: return _( "Bearer capability not authorized" );
		case 0x34BA: return _( "Bearer capability not presently available" );
		case 0x34BF: return _( "Service or option not available, unspecified" );
		case 0x34C1: return _( "Bearer capability not implemented" );
		case 0x34C2: return _( "Channel type not implemented" );
		case 0x34C5: return _( "Requested facility not implemented" );
		case 0x34C6: return _( "Only restricted digital information bearer capability is available" );
		case 0x34CF: return _( "Service or option not implemented, unspecified" );
		case 0x34D1: return _( "Invalid call reference value" );
		case 0x34D2: return _( "Identified channel does not exist" );
		case 0x34D3: return _( "A suspended call exists, but this call identity does not" );
		case 0x34D4: return _( "Call identity in use" );
		case 0x34D5: return _( "No call suspended" );
		case 0x34D6: return _( "Call having the requested call identity has been cleared" );
		case 0x34D8: return _( "Incompatible destination" );
		case 0x34DB: return _( "Invalid transit network selection" );
		case 0x34DF: return _( "Invalid message, unspecified" );
		case 0x34E0: return _( "Mandatory information element is missing" );
		case 0x34E1: return _( "Message type non-existent or not implemented" );
		case 0x34E2: return _( "Message not compatible with call state or message type non-existent or not implemented" );
		case 0x34E3: return _( "Information element non-existent or not implemented" );
		case 0x34E4: return _( "Invalid information element contents" );
		case 0x34E5: return _( "Message not compatible with call state" );
		case 0x34E6: return _( "Recovery on timer expiry" );
		case 0x34EF: return _( "Protocol error, unspecified" );
		case 0x34FF: return _( "Interworking, unspecified" );

		case 0x3500: return _( "Normal end of connection" );
		case 0x3501: return _( "Carrier lost" );
		case 0x3502: return _( "Error in negotiation, i.e. no modem with error correction at the other end" );
		case 0x3503: return _( "No answer to protocol request" );
		case 0x3504: return _( "Remote modem only works in synchronous mode" );
		case 0x3505: return _( "Framing fails" );
		case 0x3506: return _( "Protocol negotiation fails" );
		case 0x3507: return _( "Other modem sends wrong protocol request" );
		case 0x3508: return _( "Sync information (data or flags) missing" );
		case 0x3509: return _( "Normal end of connection from the other modem" );
		case 0x350A: return _( "No answer from other modem" );
		case 0x350B: return _( "Protocol error" );
		case 0x350C: return _( "Error in compression" );
		case 0x350D: return _( "No conenct (timeout or wrong modulation)" );
		case 0x350E: return _( "No protocol fall-back allowed" );
		case 0x350F: return _( "No modem or fax at requested number" );
		case 0x3510: return _( "Handshake error" );

		default: return _( "No additional information" );
    }
}

/**
 * \brief Set fax button according to window type
 * \param nType type of window
 */
static void faxSetButtons( gint nType ) {
	if ( psCancelButton != NULL ) {
		gtk_widget_destroy( psCancelButton );
	}
	if ( psOkButton != NULL ) {
		gtk_widget_destroy( psOkButton );
	}

	if ( nType == 0 ) {
		/* Add cancel button and enable it */
		psCancelButton = gtk_dialog_add_button( GTK_DIALOG( psSendWindow ), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL );
		gtk_widget_set_sensitive( psCancelButton, TRUE );

		/* Add ok button and disable it */
		psOkButton = gtk_dialog_add_button( GTK_DIALOG( psSendWindow ), GTK_STOCK_OK, GTK_RESPONSE_OK );
		gtk_widget_set_sensitive( psOkButton, FALSE );
	} else {
		/* Add redo button and enable it */
		psCancelButton = gtk_dialog_add_button( GTK_DIALOG( psSendWindow ), GTK_STOCK_REDO, 1 );
		gtk_widget_set_sensitive( psCancelButton, TRUE );

		/* Add ok button and disable it */
		psOkButton = gtk_dialog_add_button( GTK_DIALOG( psSendWindow ), GTK_STOCK_OK, GTK_RESPONSE_OK );
		gtk_widget_set_sensitive( psOkButton, TRUE );
	}

	gtk_widget_show( psCancelButton );
	gtk_widget_show( psOkButton );
}

/**
 * \brief Open audio interface
 * \return error code by audio plugin
 */
int OpenAudio( void ) {
	struct sAudio *psAudio = getDefaultAudioPlugin( getActiveProfile() );

	if ( psAudio == NULL ) {
		return -1;
	}

	pAudioPriv = psAudio -> Open();
	
	return 0;
};

/**
 * \brief Read audio interface
 * \param pnData data pointer
 * \param nSize size of data
 * \return error code by audio plugin or length of read data
 */
int ReadAudio( unsigned char *pnData, unsigned int nSize ) {
	struct sAudio *psAudio = getDefaultAudioPlugin( getActiveProfile() );

	if ( psAudio == NULL ) {
		return -1;
	}

	return psAudio -> Read( pAudioPriv, pnData, nSize );
}

/**
 * \brief Write audio interface
 * \param pnData data pointer
 * \param nSize size of data
 * \return error code by audio plugin
 */
int WriteAudio( unsigned char *pnData, unsigned int nSize ) {
	struct sAudio *psAudio = getDefaultAudioPlugin( getActiveProfile() );

	if ( psAudio == NULL ) {
		return -1;
	}

	return psAudio -> Write( pAudioPriv, pnData, nSize );
}

/**
 * \brief Close audio interface
 * \return error code by audio plugin
 */
int CloseAudio( void ) {
	struct sAudio *psAudio = getDefaultAudioPlugin( getActiveProfile() );

	if ( psAudio == NULL ) {
		return -1;
	}

	return psAudio -> Close( pAudioPriv, FALSE );
}

/**
 * \brief Connection established callback
 * \param psConnection capi connection pointer
 */
void connectionEstablished( struct sCapiConnection *psConnection ) {
	if ( psPhoneConnection == psConnection ) {
		Debug( KERN_DEBUG, "Softphone connected!!\n" );

		//ffgtkLock();
		dialWindowSetStatus( dialGetDialog(), CALL_TYPE_CONNECT );
		//ffgtkUnlock();
	}
	if ( psFaxConnection == psConnection ) {
		Debug( KERN_DEBUG, "Fax connected!!\n" );
	}
}

void connectionStatus( struct sCapiConnection *psConnection, int nStatus );

/**
 * \brief Connection terminated callback
 * \param psConnection capi connection pointer
 */
void connectionTerminated( struct sCapiConnection *psConnection ) {
	if ( psPhoneConnection != NULL && psPhoneConnection == psConnection ) {
		psPhoneConnection = NULL;
		Debug( KERN_DEBUG, "Terminated!!\n" );

		ffgtkLock();
		dialWindowSetStatus( dialGetDialog(), CALL_TYPE_DISCONNECT );
		ffgtkUnlock();

		bDtmfCheck = false;
	}
	if ( psFaxConnection != NULL && psFaxConnection == psConnection ) {
		struct sFaxStatus *psStatus = psConnection -> pPrivate;

		/* Temporary workaround for buggy spandsp */
		if ( psStatus -> ePhase != FAX_OK && psStatus -> nPageTotal == psStatus -> nPageCurrent ) {
			psStatus -> ePhase = FAX_OK;
		}
		connectionStatus( psConnection, 0 );

		psFaxConnection = NULL;
		Debug( KERN_DEBUG, "Terminated!!\n" );
		Debug( KERN_DEBUG, "Error 0x%x (%s)\n", psConnection -> nCapiCode, capiInfo2String( psConnection -> nCapiCode ) );
		/*ffgtkLock();
		if ( psErrorLabel != NULL ) {
			gtk_label_set_text( GTK_LABEL( psErrorLabel ), capiInfo2String( psConnection -> nCapiCode ) );
		}

		if ( psStatus -> ePhase != FAX_OK ) {
			gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Fax transfer failed</b>" ) );
			if ( bSending == FALSE ) {
				g_source_remove( nTimer );

				gtk_widget_set_sensitive( psOkButton, TRUE );
				gtk_widget_set_sensitive( psCancelButton, FALSE );
				gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psProgressBar ), 1.0 );
			} else {
				faxSetButtons( 1 );
			}
			CreateFaxReport( psStatus, pnCurrentReportDir );
		}
		ffgtkUnlock();*/
	}
}

void createFaxStatusWindow( const char *pnSenderNumber, char *pnNumber );
int FaxReceive( struct sCapiConnection *psConnection, const char *pnTiffFile, int nModem, int nEcm, const char *pnSrcNo, char *pnTrgNo, int nManualHookup );

/**
 * \brief Connection DTMF code callback
 * \param psConnection capi connection pointer
 */
void connectionRing( struct sCapiConnection *psConnection ) {
	gchar *pnTrg = psConnection -> pnTarget;

	if ( strchr( pnTrg, '#' ) ) {
		pnTrg = strchr( pnTrg, '#' ) + 1;
	}
	
	Debug( KERN_DEBUG, "%s <-> %s\n", pnTrg, faxGetSenderMsn( getActiveProfile() ) ); 
	if ( psFaxConnection == NULL && !strcmp( pnTrg, faxGetSenderMsn( getActiveProfile() ) ) && ( faxGetAccept( getActiveProfile() ) == TRUE ) ) {
		time_t tim = time( NULL );
		struct tm *psNow = localtime( &tim );
		gchar *pnFile = g_strdup_printf( "%s/ffgtk_%s_%s-%02d_%02d_%d_%02d_%02d_%02d.tiff",
		   faxGetReportDir( getActiveProfile() ), psConnection -> pnSource, pnTrg,
		   psNow -> tm_mday, psNow -> tm_mon + 1, psNow -> tm_year + 1900,
		   psNow -> tm_hour, psNow -> tm_min, psNow -> tm_sec );
		Debug( KERN_DEBUG, "Incoming fax!\n" );

		FaxReceive( psConnection, pnFile, faxGetBitrate( getActiveProfile() ), FALSE, pnTrg, psConnection -> pnSource, FALSE );

		capiPickup( psConnection, SESSION_FAX ); 
		psFaxConnection = psConnection;
		bSending = FALSE;

		ffgtkLock();
		createFaxStatusWindow( psConnection -> pnSource, pnTrg );
		ffgtkUnlock();
	} else if ( psPhoneConnection == NULL ) {
		//struct sProfile *psProfile = getActiveProfile();
		psPhoneConnection = psConnection;
		Debug( KERN_DEBUG, "Ring!!\n" );

		/*Debug( KERN_DEBUG, "Checking caller: %s <-> %s\n", dtmfGetCaller( psProfile ), psPhoneConnection -> pnSource );
		Debug( KERN_DEBUG, "Checking called: %s <-> %s\n", dtmfGetCalled( psProfile ), pnTrg );
		if ( !strcmp( dtmfGetCaller( psProfile ), psPhoneConnection -> pnSource ) && !strcmp( dtmfGetCalled( psProfile ), pnTrg ) ) {
			Debug( KERN_INFO, "DTMF Action: pickup call\n" );
			bDtmfCheck = true;
			softphonePickup();
		}*/

		if ( validDtmfAction( psPhoneConnection -> pnSource, pnTrg ) == TRUE ) {
			Debug( KERN_INFO, "DTMF Action: pickup call\n" );
			bDtmfCheck = true;
			softphonePickup();
		}
	}
}

/**
 * \brief Connection DTMF code callback
 * \param psConnection capi connection pointer
 * \param nCode DTMF code
 */
void connectionCode( struct sCapiConnection *psConnection, int nCode ) {
	static GString *psDtmfCode = NULL;

	if ( psPhoneConnection == NULL || psPhoneConnection != psConnection ) {
		return;
	}

	Debug( KERN_DEBUG, "Code!! (%c)\n", nCode );

	if ( bDtmfCheck == FALSE ) {
		return;
	}

	switch ( nCode ) {
		case '#':
			if ( psDtmfCode != NULL ) {
				Debug( KERN_DEBUG, "psDtmfCode %s\n", psDtmfCode -> str );

				executeDtmfActions( psConnection -> pnSource, psConnection -> pnTarget, psDtmfCode -> str );
				/*if ( !strcmp( dtmfGetCode( getActiveProfile() ), psDtmfCode -> str ) ) {
					Debug( KERN_INFO, "DTMF-Action: Valid code entered!\n" );
					g_spawn_command_line_async( dtmfGetExecute( getActiveProfile() ), NULL );
				}*/

				g_string_free( psDtmfCode, TRUE );
				psDtmfCode = NULL;
			}
			break;
		default:
			if ( psDtmfCode == NULL ) {
				psDtmfCode = g_string_new("");
			}

			psDtmfCode = g_string_append_c( psDtmfCode, nCode );
			break;
	}
}

/**
 * \brief Connection status callback
 * \param psConnection capi connection pointer
 * \param nStatus status code
 */
void connectionStatus( struct sCapiConnection *psConnection, int nStatus ) {
	struct sFaxStatus *psStatus;
	gchar anBuffer[ 256 ];

	Debug( KERN_DEBUG, "Called\n" );
	if ( psConnection == NULL ) {
		return;
	}

	if ( psPhoneConnection == psConnection ) {
		return;
	}

	psStatus = psConnection -> pPrivate;
	if ( psStatus == NULL ) {
		return;
	}

	if ( nStatus == 0 && psStatus -> nDone == 0 ) {
		ffgtkLock();
		Debug( KERN_DEBUG, "Phase: %d\n", psStatus -> ePhase );

		if ( psStatus -> ePhase == PHASE_E ) {
			Debug( KERN_DEBUG, "Temporary spandsp bug workaround (%d/%d)\n", psStatus -> nPageTotal, psStatus -> nPageCurrent ); 
			if ( psStatus -> nPageTotal == psStatus -> nPageCurrent ) {
				psStatus -> ePhase = FAX_OK;
			}
		}
		
		switch ( psStatus -> ePhase ) {
			case PHASE_B:
				if ( bSending == TRUE ) {
					gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Sending fax (START)</b>" ) );
					Debug( KERN_DEBUG, "Sending fax (START)\n" );
				} else {
					gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Receiving fax (START)</b>" ) );
					Debug( KERN_DEBUG, "Receiving fax (START)\n" );
				}
				Debug( KERN_DEBUG, "Ident: %s\n", psStatus -> anIdent );
				Debug( KERN_DEBUG, "Bitrate: %d\n", psStatus -> nBitrate );
				gtk_label_set_text( GTK_LABEL( psRemoteName ), psStatus -> anIdent );
				snprintf( anBuffer, sizeof( anBuffer ), "%d", psStatus -> nBitrate );
				gtk_label_set_text( GTK_LABEL( psBitrateNumber ), anBuffer );
				snprintf( anBuffer, sizeof( anBuffer ), "%d", psStatus -> nBadRows );
				gtk_label_set_text( GTK_LABEL( psBadRowsNumber ), anBuffer );
				break;
			case FAX_OK:
				gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Fax transfer successful</b>" ) );
				Debug( KERN_DEBUG, "Fax transfer successful\n" );
				gtk_widget_set_sensitive( psOkButton, TRUE );
				gtk_widget_set_sensitive( psCancelButton, FALSE );
				if ( bSending == FALSE ) {
					g_source_remove( nTimer );
					gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psProgressBar ), 1.0 );
					gtk_progress_bar_set_text( GTK_PROGRESS_BAR( psProgressBar ), "100%" );
				}
				CreateFaxReport( psStatus, pnCurrentReportDir );
				phoneHangup( psConnection );
				psStatus -> nDone = 1;
				break;
			case FAX_FAILED:
				gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Fax transfer failed</b>" ) );
				Debug( KERN_DEBUG, "Fax transfer failed\n" );
				if ( bSending == FALSE ) {
					g_source_remove( nTimer );

					gtk_widget_set_sensitive( psOkButton, TRUE );
					gtk_widget_set_sensitive( psCancelButton, FALSE );
					gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psProgressBar ), 1.0 );
				} else {
					faxSetButtons( 1 );
				}
				CreateFaxReport( psStatus, pnCurrentReportDir );
				phoneHangup( psConnection );
				psStatus -> nDone = 1;
				break;
			case PHASE_D:
				// Pages transferred
				if ( bSending == TRUE ) {
					gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Sending fax (PAGE)</b>" ) );
					Debug( KERN_DEBUG, "Sending fax (PAGE)\n" );
				} else {
					gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Receiving fax (PAGE)</b>" ) );
					Debug( KERN_DEBUG, "Receiving fax (PAGE)\n" );
				}
				snprintf( anBuffer, sizeof( anBuffer ), "%d/%d", psStatus -> nPageCurrent, psStatus -> nPageTotal );
				Debug( KERN_DEBUG, "Pages: %s\n", anBuffer );
				gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psProgressBar ), (float)psStatus -> nPageCurrent / (float)psStatus -> nPageTotal );
				gtk_label_set_text( GTK_LABEL( psPageNumber ), anBuffer );
				break;
			default:
				Debug( KERN_DEBUG, "Unhandled phase\n" );
				phoneHangup( psConnection );
				break;
		}
		ffgtkUnlock();
	} else if ( nStatus == 1 ) {
		float fPercentage = 0.0f;
		gchar *pnText = NULL;

		fPercentage = ( float ) psStatus -> nBytesSent / ( float ) psStatus -> nBytesTotal;

		if ( fPercentage > 1.0f ) {
			fPercentage = 1.0f;
		}

		ffgtkLock();
		gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psProgressBar ), fPercentage );
		fPercentage *= 100;

		pnText = g_strdup_printf( "%d%%", ( int ) fPercentage );
		Debug( KERN_DEBUG, "Transfer at %s\n", pnText );
		gtk_progress_bar_set_text( GTK_PROGRESS_BAR( psProgressBar ), pnText );
		g_free( pnText );
		ffgtkUnlock();
	}
}

/** session handlers */
static struct sSessionHandlers sHandlers = {
		OpenAudio,
		ReadAudio,
		WriteAudio,
		CloseAudio,

		connectionEstablished,
		connectionTerminated,
		connectionRing,
		connectionCode,
		connectionStatus
};

/**
 * \brief Dial a number
 * \param pnNumber target number
 * \param bAnonymous dial anonymous flag
 */
int softphoneDial( gchar *pnNumber, gboolean bAnonymous ) {
	struct sProfile *psProfile = getActiveProfile();

	if ( psSession == NULL ) {
		Debug( KERN_DEBUG, "Session is NULL!!\n" );
		return -1;
	}

	Debug( KERN_DEBUG, "Calling '%s' from '%s' on controller '%d'\n", pnNumber, softphoneGetMsn( psProfile ), softphoneGetController( psProfile ) + 1 );
	psPhoneConnection = phoneCall( softphoneGetController( psProfile ) + 1, softphoneGetMsn( psProfile ), pnNumber, bAnonymous );
	if ( psPhoneConnection == NULL ) {
		Debug( KERN_DEBUG, "Dial failed!\n" );
		return -1;
	}

	return 0;
}

/**
 * \brief Hangup current connection
 */
void softphoneHangup( void ) {
	if ( psSession == NULL ) {
		Debug( KERN_DEBUG, "Session is NULL!!\n" );
		return;
	}

	phoneHangup( psPhoneConnection );
}

/**
 * \brief Pickup current connection
 */
void softphonePickup( void ) {
	if ( psSession == NULL ) {
		Debug( KERN_DEBUG, "Session is NULL!!\n" );
		return;
	}

	phonePickup( psPhoneConnection );
}

/**
 * \brief Send DTMF on current connection
 * \param nNum dtmf code
 */
void softphoneSendDtmfCode( gint nNum ) {
	if ( psPhoneConnection != NULL ) {
		phoneSendDtmfCode( psPhoneConnection, nNum );
	}
}

/**
 * \brief Mute current connection
 * \param bMute mute flag
 */
void softphoneMute( gboolean bMute ) {
	if ( psPhoneConnection != NULL ) {
		phoneMute( psPhoneConnection, bMute );
	}
}

/**
 * \brief Hold current connection
 * \param bHold hold flag
 */
void softphoneHold( gboolean bHold ) {
	if ( psPhoneConnection != NULL ) {
		phoneHold( psPhoneConnection, bHold );
	}
}

/**
 * \brief Record current connection
 * \param bRecord record flag
 */
void softphoneRecord( gboolean bRecord ) {
	if ( psPhoneConnection != NULL ) {
		phoneRecord( psPhoneConnection, bRecord );
	}
}

void softphoneFlush( void ) {
	if ( psPhoneConnection != NULL ) {
		phoneFlush( psPhoneConnection );
	}
}

/**
 * \brief React on bluetooth headset buttons (pickup/hangup current connection)
 */
void softphoneHandleBluetooth( void ) {
	if ( psPhoneConnection != NULL ) {
		if ( psPhoneConnection -> nType == SESSION_NONE ) {
			//dialSetNumber( psPhoneConnection -> pnSource );
			dialingDialog( psPhoneConnection -> pnSource, _( "Softphone" ) );

			softphonePickup();
		} else {
			softphoneHangup();
		}
	}
}
	
/**
 * \brief Move progress bar
 * \param pData user data pointer
 * \return TRUE (do not delete timer)
 */
static gboolean updateProgressBar( gpointer pData ) {
	gtk_progress_bar_pulse( GTK_PROGRESS_BAR( pData ) );

	return TRUE;
}

/**
 * \brief Send window response callback
 * \param psDialog dialog window pointer
 * \param nResponse response id
 * \param pUserData user data pointer
 */
static void sendWindowResponse( GtkDialog *psDialog, gint nResponse, gpointer pUserData ) {
	switch ( nResponse ) {
		case 1:
			psFaxConnection = faxSend( pnCurrentFile, nCurrentBitrate, nCurrentEcm, nCurrentController, pnCurrentSenderMsn, pnCurrentNumber, pnCurrentSenderNumber, pnCurrentSenderName, nCurrentCallAnonymous );
			if ( bSending == FALSE ) {
				nTimer = g_timeout_add( 100, updateProgressBar, psProgressBar );
			}
			gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Outgoing fax</b>" ) );
			gtk_label_set_text( GTK_LABEL( psErrorLabel ), "" );
			gtk_progress_bar_set_text( GTK_PROGRESS_BAR( psProgressBar ), "0%" );
			gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psProgressBar ), 0.0f );
			faxSetButtons( 0 );
			return;
		case GTK_RESPONSE_OK:
			break;
		case GTK_RESPONSE_CANCEL:
		case GTK_RESPONSE_DELETE_EVENT:
			if ( psFaxConnection != NULL ) {
				phoneHangup( psFaxConnection );
			}
			if ( bSending == FALSE ) {
				g_source_remove( nTimer );
			}
			break;
	}

	gtk_widget_destroy( GTK_WIDGET( psDialog ) );
	psSendWindow = NULL;

	if ( pnCurrentSenderMsn != NULL ) {
		g_free( pnCurrentSenderMsn );
		pnCurrentSenderMsn = NULL;
	}
	if ( pnCurrentNumber != NULL ) {
		g_free( pnCurrentNumber );
		pnCurrentNumber = NULL;
	}
	if ( pnCurrentSenderNumber != NULL ) {
		g_free( pnCurrentSenderNumber );
		pnCurrentSenderNumber = NULL;
	}
	if ( pnCurrentSenderName != NULL ) {
		g_free( pnCurrentSenderName );
		pnCurrentSenderName = NULL;
	}

	if ( pnCurrentReportDir != NULL ) {
		g_free( pnCurrentReportDir );
		pnCurrentReportDir = NULL;
	}

	if ( bSending == TRUE ) {
		unlink( pnCurrentFile );
	}

	if ( pnCurrentFile != NULL ) {
		g_free( pnCurrentFile );
		pnCurrentFile = NULL;
	}
}

/**
 * \brief Creates fax status windows, used by sending/receiving
 * \param pnSenderNumber sender number
 * \param pnNumber target number
 */
void createFaxStatusWindow( const char *pnSenderNumber, char *pnNumber ) {
	GtkWidget *psTable;
	GtkWidget *psExpander;
	GtkWidget *psExpanderTable;
	GtkWidget *psSourceLabel, *psSourceNumber;
	GtkWidget *psDestinationLabel, *psDestinationNumber;
	GtkWidget *psBitrateLabel;
	GtkWidget *psEncodingLabel;
	GtkWidget *psBadRowsLabel;
	GtkWidget *psPageLabel;
	GtkWidget *psBox = NULL;

	psSendWindow = gtk_dialog_new_with_buttons( _( "Fax Status" ), NULL, GTK_DIALOG_DESTROY_WITH_PARENT, NULL, NULL );

	psCancelButton = NULL;
	psOkButton = NULL;
	faxSetButtons( 0 );

	/* create main table */
	psTable = gtk_table_new( 4, 2, false );

	/* Create fax status label */
	psStatusLabel = gtk_label_new( "" );
	gtk_misc_set_alignment( GTK_MISC( psStatusLabel ), 0.5, 0.5 );
	gtk_table_attach( GTK_TABLE( psTable ), psStatusLabel, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND, 0, 0, 0 );
	if ( bSending == TRUE ) {
		gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Outgoing fax</b>" ) );
	} else {
		gtk_label_set_markup( GTK_LABEL( psStatusLabel ), _( "<b>Incoming fax</b>" ) );
	}

	psErrorLabel = gtk_label_new( "" );
	gtk_misc_set_alignment( GTK_MISC( psErrorLabel ), 0.5, 0.5 );
	gtk_table_attach( GTK_TABLE( psTable ), psErrorLabel, 0, 2, 1, 2, GTK_FILL | GTK_EXPAND, 0, 0, 0 );

	/* Create Progressbar */
	psProgressBar = gtk_progress_bar_new();
	if ( bSending == false ) {
		gtk_progress_bar_pulse( GTK_PROGRESS_BAR( psProgressBar ) );
	}
	gtk_progress_bar_set_text( GTK_PROGRESS_BAR( psProgressBar ), "0%" );
	gtk_table_attach( GTK_TABLE( psTable ), psProgressBar, 0, 2, 2, 3, GTK_FILL | GTK_EXPAND, 0, 0, 0 );

	/* Create expander */
	psExpander = gtk_expander_new( _( "More Information" ) );
	gtk_table_attach( GTK_TABLE( psTable ), psExpander, 0, 2, 3, 4, GTK_FILL, 0, 0, 0 );

	psExpanderTable = gtk_table_new( 4, 2, false );
	gtk_container_add( GTK_CONTAINER( psExpander ), GTK_WIDGET( psExpanderTable ) );

	/* Source label and number */
	psSourceLabel = gtk_label_new( _( "Sender-Number:" ) );
	gtk_misc_set_alignment( GTK_MISC( psSourceLabel ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psSourceLabel ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psSourceLabel, 0, 1, 0, 1, GTK_FILL, 0, 0, 0 );

	psSourceNumber = gtk_label_new( pnSenderNumber );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psSourceNumber, 1, 2, 0, 1, GTK_FILL, 0, 0, 0 );

	/* Destination label and number */
	psDestinationLabel = gtk_label_new( _( "Destination-Number:" ) );
	gtk_misc_set_alignment( GTK_MISC( psDestinationLabel ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psDestinationLabel ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psDestinationLabel, 0, 1, 1, 2, GTK_FILL, 0, 0, 0 );

	psDestinationNumber = gtk_label_new( pnNumber );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psDestinationNumber, 1, 2, 1, 2, GTK_FILL, 0, 0, 0 );

	/* Remote label and name */
	psRemoteLabel = gtk_label_new( _( "Remote-Name:" ) );
	gtk_misc_set_alignment( GTK_MISC( psRemoteLabel ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psRemoteLabel ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psRemoteLabel, 0, 1, 2, 3, GTK_FILL, 0, 0, 0 );

	psRemoteName = gtk_label_new( "" );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psRemoteName, 1, 2, 2, 3, GTK_FILL, 0, 0, 0 );

	/* Bitrate label and number */
	psBitrateLabel = gtk_label_new( _( "Bitrate:" ) );
	gtk_misc_set_alignment( GTK_MISC( psBitrateLabel ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psBitrateLabel ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psBitrateLabel, 0, 1, 3, 4, GTK_FILL, 0, 0, 0 );

	psBitrateNumber = gtk_label_new( "-----" );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psBitrateNumber, 1, 2, 3, 4, GTK_FILL, 0, 0, 0 );

	/* Encoding label and number */
	psEncodingLabel = gtk_label_new( _( "Encoding:" ) );
	gtk_misc_set_alignment( GTK_MISC( psEncodingLabel ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psEncodingLabel ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psEncodingLabel, 0, 1, 4, 5, GTK_FILL, 0, 0, 0 );

	psEncodingNumber = gtk_label_new( "-" );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psEncodingNumber, 1, 2, 4, 5, GTK_FILL, 0, 0, 0 );

	/* BadRows label and number */
	psBadRowsLabel = gtk_label_new( _( "Bad Rows:" ) );
	gtk_misc_set_alignment( GTK_MISC( psBadRowsLabel ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psBadRowsLabel ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psBadRowsLabel, 0, 1, 5, 6, GTK_FILL, 0, 0, 0 );

	psBadRowsNumber = gtk_label_new( "-" );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psBadRowsNumber, 1, 2, 5, 6, GTK_FILL, 0, 0, 0 );

	/* Page label and number */
	psPageLabel = gtk_label_new( _( "Page:" ) );
	gtk_misc_set_alignment( GTK_MISC( psPageLabel ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psPageLabel ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psPageLabel, 0, 1, 6, 7, GTK_FILL, 0, 0, 0 );

	psPageNumber = gtk_label_new( "-" );
	gtk_table_attach( GTK_TABLE( psExpanderTable ), psPageNumber, 1, 2, 6, 7, GTK_FILL, 0, 0, 0 );

	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psSendWindow ) );
	if ( psBox != NULL ) {
		gtk_box_pack_start( GTK_BOX( psBox ), psTable, false, false, 10 );
	}

	g_signal_connect( G_OBJECT( psSendWindow ), "response", G_CALLBACK( sendWindowResponse ), NULL );

	if ( bSending == FALSE ) {
		nTimer = g_timeout_add( 100, updateProgressBar, psProgressBar );
	}

	gtk_window_set_default_size( GTK_WINDOW( psSendWindow ), 380, -1 );
	gtk_widget_show_all( psSendWindow );
}

/**
 * \brief Send fax via faxohpone
 * \param pnFaxFile file to transfer
 * \param nBitRate transfer bitrate
 * \param nEcm error correction mode flag
 * \param nController controller
 * \param pnSender sender number
 * \param pnTarget target number
 * \param pnHeader fax header
 * \param pnIdent ident
 * \param nAnonymous call anonymous flag
 * \param pnReportDir optional report directory
 * \return error code
 */
int faxophoneSendFax( gchar *pnFaxFile, gint nBitRate, gint nEcm, gint nController, const gchar *pnSender, gchar *pnTarget, const gchar *pnHeader, const gchar *pnIdent, gint nAnonymous, const gchar *pnReportDir ) {
	if ( psSession == NULL ) {
		Debug( KERN_DEBUG, "Session is NULL!!\n" );
		return -1;
	}

	/* Call number */
	psFaxConnection = faxSend( pnFaxFile, nBitRate, nEcm, nController + 1, pnSender, pnTarget, pnHeader, pnIdent, nAnonymous );
	if ( psFaxConnection != NULL ) {
		if ( pnReportDir != NULL ) {
			pnCurrentReportDir = g_strdup( pnReportDir );
		} else {
			pnCurrentReportDir = NULL;
		}

		nCurrentBitrate = nBitRate;
		nCurrentEcm = nEcm;
		nCurrentController = nController + 1;
		pnCurrentSenderMsn = g_strdup( pnSender );
		pnCurrentNumber = g_strdup( pnTarget );
		pnCurrentSenderNumber = g_strdup( pnHeader );
		pnCurrentSenderName = g_strdup( pnIdent );
		nCurrentCallAnonymous = nAnonymous;
		pnCurrentFile = g_strdup( pnFaxFile );
		bSending = TRUE;

		createFaxStatusWindow( pnSender, ( char * ) pnTarget );
	} else {
		Debug( KERN_DEBUG, "Dial failed!\n" );
		return -1;
	}

	return 0;
}

/**
 * \brief Try to start faxophone session
 * \param pData UNUSED
 * \return NULL
 */
static gpointer startSession( gpointer pData ) {
	while ( 1 ) {
		struct sProfile *psProfile = getActiveProfile();

		psSession = faxophoneInit( &sHandlers, psProfile ? routerGetHost( psProfile ) : NULL );
		if ( psSession != NULL ) {
			break;
		}

		g_usleep( 2 * G_USEC_PER_SEC );
	}
	return NULL;
}

void faxophoneLog( int nLevel, const char *pnString ) {
	Debug( KERN_DEBUG, pnString );
}

void FopSetLogHandler( void ( *log )( int nLevel, const char *pnString ) );

/**
 * \brief Start faxophone session thread
 */
void InitFaxPhone( void ) {
	FopSetLogHandler( faxophoneLog );
	CREATE_THREAD("faxophone", startSession, NULL );
}

/**
 * \brief Close faxophone
 */
void CloseFaxPhone( void ) {
	faxophoneClose();
}
