/*
 * $Id: conv_cd.c,v 1.12 2012-02-13 10:01:21 vrsieh Exp $
 *
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue.h"

#include "conv_cd.h"

#define BLOCKSIZE	(2352 + 10 + 10)

#define MAX_DATA_TRACKS 100
#define MAX_CD_AREAS    100

struct conv_cd {
	void *file[MAX_DATA_TRACKS];
	unsigned int nfiles;
	struct {
		enum {
			CD_LEADIN,
			CD_PREGAP,
			CD_AUDIO,
			CD_DATA,
			CD_POSTGAP,
			CD_LEADOUT,
		} type;
		unsigned int tracknum;
		unsigned int indexnum;
		unsigned int filenum;
		unsigned long long fileoffset;
		unsigned long long size;
	} area[MAX_CD_AREAS];
	unsigned int nareas;
};

enum token {
	PARSE_ERROR,

	PARSE_EOF,
	PARSE_FILE,
	PARSE_TRACK,
	PARSE_PREGAP,
	PARSE_INDEX,
	PARSE_POSTGAP,
	PARSE_STRING,
	PARSE_PARAM,
};

static enum token
conv_cd_lex(const char **cfp, char *buf)
{
	const char *cf;
	enum token ret;

	cf = *cfp;

	ret = PARSE_ERROR;

	for (;;) {
		if (*cf == '\0') {
			/* EOF */
			ret = PARSE_EOF;
			break;

		} else if (*cf == '\t'
			|| *cf == '\n'
			|| *cf == '\r'
			|| *cf == ' ') {
			/* White Space */
			cf++;
			continue;

		} else if (strncmp(cf, "REM", 3) == 0
			|| strncmp(cf, "rem", 3) == 0) {
			/* REMark Line */
			cf += strlen("REM");
			while (*cf != '\0'
					&& *cf != '\n') {
				cf++;
			}
			continue;

		} else if (strncmp(cf, "FILE", 4) == 0
			|| strncmp(cf, "file", 4) == 0) {
			/* FILE Token */
			cf += 4;
			ret = PARSE_FILE;
			break;

		} else if (strncmp(cf, "TRACK", 5) == 0
			|| strncmp(cf, "track", 5) == 0) {
			/* TRACK Token */
			cf += 5;
			ret = PARSE_TRACK;
			break;

		} else if (strncmp(cf, "PREGAP", 6) == 0
			|| strncmp(cf, "pregap", 6) == 0) {
			/* PREGAP Token */
			cf += 6;
			ret = PARSE_PREGAP;
			break;

		} else if (strncmp(cf, "INDEX", 5) == 0
			|| strncmp(cf, "index", 5) == 0) {
			/* TRACK Token */
			cf += 5;
			ret = PARSE_INDEX;
			break;

		} else if (strncmp(cf, "POSTGAP", 7) == 0
			|| strncmp(cf, "postgap", 7) == 0) {
			/* POSTGAP Token */
			cf += 7;
			ret = PARSE_POSTGAP;
			break;

		} else if (*cf == '"') {
			/* String Token */
			cf++;
			while (*cf != '"'
					&& *cf != '\n'
					&& *cf != '\0') {
				*buf++ = *cf++;
			}
			if (*cf != '"') break;
			cf++;
			*buf = '\0';
			ret = PARSE_STRING;
			break;

		} else {
			/* Parameter Token */
			/* E.g.: 01, 00:02:00, BINARY, MODE1/2048, ... */
			while (*cf != '\0'
					&& *cf != '\t'
					&& *cf != '\n'
					&& *cf != '\r'
					&& *cf != ' ') {
				*buf++ = *cf++;
			}
			*buf = '\0';
			ret = PARSE_PARAM;
			break;
		}
	}

	*cfp = cf;
	return ret;
}

static void *
conv_cd_open(const char *cf, const char *path)
{
	struct conv_cd *s;
	enum token token;
	char param[256];
	int filenum;
	int tracknum;
	int indexnum;
	int lastarea;
	unsigned long long size;

	s = shm_alloc(sizeof(*s));
	assert(s);

	/*
	 * Parse cue sheet (simple version). FIXME VOSSI
	 */
	filenum = -1;
	tracknum = -1;
	indexnum = -1;
	lastarea = -1;
	size = 0;

	s->nareas = 0;

	/* Add leadin. */
	s->area[s->nareas].type = CD_LEADIN;
	s->area[s->nareas].tracknum = 0;
	s->area[s->nareas].indexnum = 0;
	s->area[s->nareas].filenum = 0;
	s->area[s->nareas].fileoffset = 0;
	s->area[s->nareas].size = 0;
	s->nareas++;

	s->nfiles = 0;

	for (;;) {
		char filename[1024];
		unsigned int arg;

		token = conv_cd_lex(&cf, param);
		if (token == PARSE_ERROR) {
syntax:;
			fprintf(stderr, "%s: WARNING: parse error in cue sheet\n",
					path);
			free(s);
			return NULL;
		}
		if (token == PARSE_EOF) {
			break;
		}
		switch (token) {
		case PARSE_FILE:
			/*
			 * FILE "<name>" <type>
			 */
			/* Read filename. */
			token = conv_cd_lex(&cf, param);
			if (token != PARSE_STRING) goto syntax;

			if (param[0] == '/') {
				strcpy(filename, param);
			} else if (strchr(path, '/')) {
				strcpy(filename, path);
				strcpy(strrchr(filename, '/') + 1, param);
			} else {
				strcpy(filename, param);
			}

			/* Read type. */
			token = conv_cd_lex(&cf, param);
			if (token != PARSE_PARAM) goto syntax;

			/* Ignore it... */ /* FIXME */

			if (0 <= lastarea) {
				/* Patch last index area. */
				s->area[lastarea].size = size;
			}

			s->file[s->nfiles] = storage_simple_open(filename, 0);
			assert(s->file[s->nfiles]);
			lastarea = -1;
			size = storage_simple_size(s->file[s->nfiles]);

			filenum = s->nfiles;

			s->nfiles++;
			break;

		case PARSE_TRACK:
			/*
			 * TRACK <no> <mode>
			 */
			if (filenum < 0) goto syntax;

			/* Read track number. */
			token = conv_cd_lex(&cf, param);
			tracknum = atoi(param);
			if (tracknum <= 0 || 100 <= tracknum) goto syntax;

			/* Read mode. */
			token = conv_cd_lex(&cf, param);
			/* Ignoring mode... FIXME */
			break;

		case PARSE_PREGAP:
			/*
			 * PREGAP <min>:<sec>:<frame>
			 */
			/* Read <min>:<sec>:<frame>. */
			token = conv_cd_lex(&cf, param);
			if (strlen(param) != 8) goto syntax;
			if (param[2] != ':') goto syntax;
			if (param[5] != ':') goto syntax;
			arg = atoi(&param[0]);
			arg *= 60;
			arg += atoi(&param[3]);
			arg *= 75;
			arg += atoi(&param[6]);
			arg *= 2048;

			if (s->nareas == 1
					&& arg != 2 * 75 * 2048) {
				fprintf(stderr, "WARNING: PREGAP of first trackhas bad size!\n");
				arg = 2 * 75 * 2048;
			}

			s->area[s->nareas].type = CD_PREGAP;
			s->area[s->nareas].tracknum = tracknum;
			s->area[s->nareas].indexnum = 0;
			s->area[s->nareas].filenum = 0;
			s->area[s->nareas].fileoffset = 0;
			s->area[s->nareas].size = arg;
			s->nareas++;
			break;

		case PARSE_INDEX:
			/*
			 * INDEX <no> <min>:<sec>:<frame>
			 */
			if (s->nareas == 1) {
				/* Insert standard pregap 00:02:00 */
				s->area[s->nareas].type = CD_PREGAP;
				s->area[s->nareas].tracknum = tracknum;
				s->area[s->nareas].indexnum = 0;
				s->area[s->nareas].filenum = 0;
				s->area[s->nareas].fileoffset = 0;
				s->area[s->nareas].size = 2 * 75 * 2048;
				s->nareas++;
			}

			/* Read <no>. */
			token = conv_cd_lex(&cf, param);
			if (strlen(param) != 2) goto syntax;
			indexnum = atoi(param);
			if (indexnum < 0 || 100 <= indexnum) goto syntax;

			/* Read <min>:<sec>:<frame>. */
			token = conv_cd_lex(&cf, param);
			if (strlen(param) != 8) goto syntax;
			if (param[2] != ':') goto syntax;
			if (param[5] != ':') goto syntax;
			arg = atoi(&param[0]);
			arg *= 60;
			arg += atoi(&param[3]);
			arg *= 75;
			arg += atoi(&param[6]);
			arg *= 2048;

			if (0 <= lastarea) {
				/* Patch last index area. */
				s->area[lastarea].size
					= arg - s->area[lastarea].fileoffset;
				size -= s->area[lastarea].size;
			}
			lastarea = s->nareas;

			s->area[s->nareas].type = CD_DATA;
			s->area[s->nareas].tracknum = tracknum;
			s->area[s->nareas].indexnum = indexnum;
			s->area[s->nareas].filenum = filenum;
			s->area[s->nareas].fileoffset = arg;
			s->area[s->nareas].size = 0; /* will be patched later */
			s->nareas++;
			break;

		case PARSE_POSTGAP:
			/*
			 * POSTGAP <min>:<sec>:<frame>
			 */
			/* Read <min>:<sec>:<frame>. */
			token = conv_cd_lex(&cf, param);
			if (strlen(param) != 8) goto syntax;
			if (param[2] != ':') goto syntax;
			if (param[5] != ':') goto syntax;
			arg = atoi(&param[0]);
			arg *= 60;
			arg += atoi(&param[3]);
			arg *= 75;
			arg += atoi(&param[6]);
			arg *= 2048;

			s->area[s->nareas].type = CD_POSTGAP;
			s->area[s->nareas].tracknum = tracknum;
			s->area[s->nareas].indexnum = 0;
			s->area[s->nareas].filenum = 0;
			s->area[s->nareas].fileoffset = 0;
			s->area[s->nareas].size = arg;
			s->nareas++;
			break;

		default:
			assert(0);
		}
	}
	if (0 <= lastarea) {
		/* Patch last index area. */
		s->area[lastarea].size = size;
	}

	s->area[s->nareas].type = CD_LEADOUT;
	s->area[s->nareas].tracknum = 0xaa;
	s->area[s->nareas].indexnum = 0;
	s->area[s->nareas].filenum = 0;
	s->area[s->nareas].fileoffset = 0;
	s->area[s->nareas].size = 0;
	s->nareas++;

	return s;
}

static int
conv_cd_close(void *_c)
{
	struct conv_cd *c = _c;
	unsigned int i;

	for (i = 0; i < c->nfiles; i++) {
		storage_simple_close(c->file[i]);
	}

	shm_free(c);

	return 0;
}

static void
conv_cd_sector_to_msf(
	int32_t lba,
	uint8_t *min100p,
	uint8_t *minp,
	uint8_t *secp,
	uint8_t *framep
)
{
	lba += 150;
	*min100p = lba / (100 * 60 * 75);
	lba %= 100 * 60 * 75;
	*minp = lba / (60 * 75);
	lba %= 60 * 75;
	*secp = lba / 75;
	lba %= 75;
	*framep = lba;

	assert(*min100p <= 99);
	assert(*minp <= 99);
	assert(*secp <= 59);
	assert(*framep <= 74);
}

static uint8_t
conv_cd_bin_to_bcd(uint8_t bin)
{
	return ((bin / 10) << 4) | (bin % 10);
}

static int
_conv_cd_read(
	struct conv_cd *c,
	uint8_t *buf,
	int32_t blk
)
{
	blk -= 60 * 75; /* Skip PMA - FIXME */
	blk -= 60 * 75; /* Skip TOC - FIXME */

	if (blk < 0) {
		/*
		 * Read lead-in.
		 */
		 struct entry {
			uint8_t point;
			uint8_t pmin100;
			uint8_t pmin;
			uint8_t psec;
			uint8_t pframe;
		} entry[1024];
		unsigned int nentries;
		unsigned long lba;
		unsigned int area;
		uint8_t first_track;
		uint8_t last_track;

		blk = -blk - 1;
		assert(0 <= blk);

		/* Build TOC. */
		nentries = 0;
		lba = -150;
		first_track = 100;
		last_track = 0;
		for (area = 0; area < c->nareas; area++) {
			if (area == 0) {
				/* Lead-in */
				continue;

			} else if (area == c->nareas - 1) {
				/* Lead-out */
				entry[nentries].point = 0xa2;
				conv_cd_sector_to_msf(lba,
						&entry[nentries].pmin100,
						&entry[nentries].pmin,
						&entry[nentries].psec,
						&entry[nentries].pframe);
				nentries++;

			} else {
				/* User-area */
				if (c->area[area].indexnum == 1) {
					entry[nentries].point = c->area[area].tracknum;
					conv_cd_sector_to_msf(lba,
							&entry[nentries].pmin100,
							&entry[nentries].pmin,
							&entry[nentries].psec,
							&entry[nentries].pframe);
					nentries++;

					if (c->area[area].tracknum < first_track) {
						first_track = c->area[area].tracknum;
					}
					if (last_track < c->area[area].tracknum) {
						last_track = c->area[area].tracknum;
					}
				}
			}
			lba += c->area[area].size / 2048;
		}

		/* First track. */
		entry[nentries].point = 0xa0;
		entry[nentries].pmin100 = 0;
		entry[nentries].pmin = first_track;
		entry[nentries].psec = 0;
		entry[nentries].pframe = 0;
		nentries++;

		/* Last track. */
		entry[nentries].point = 0xa1;
		entry[nentries].pmin100 = 0;
		entry[nentries].pmin = last_track;
		entry[nentries].psec = 0;
		entry[nentries].pframe = 0;
		nentries++;

		/* All data is repeated (at least) three times. */
		blk /= 3;
		blk %= nentries;

		memset(buf, 0, 2352);
		buf += 2352;

		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */

		/* Q-sub-channel Mode-1 */
		*buf++ = (0x1 << 4) | 0x4; /* Or "... | 0x6" */
		*buf++ = 0x00; /* Track: Lead-in */
		*buf++ = entry[blk].point;
		*buf++ = conv_cd_bin_to_bcd(0); /* sec -- FIXME */
		*buf++ = conv_cd_bin_to_bcd(0); /* min -- FIXME */
		*buf++ = conv_cd_bin_to_bcd(0); /* frame -- FIXME */
		*buf++ = conv_cd_bin_to_bcd(entry[blk].pmin100); /* FIXME */
		*buf++ = conv_cd_bin_to_bcd(entry[blk].pmin);
		*buf++ = conv_cd_bin_to_bcd(entry[blk].psec);
		*buf++ = conv_cd_bin_to_bcd(entry[blk].pframe);

		return 1;

	} else {
		/*
		 * Read user-area/lead-out.
		 */
		unsigned long start;
		unsigned int area;

		start = 0;
		for (area = 0; ; area++) {
			if (area == c->nareas) {
				/* End of media reached. */
				return 0;
			}

			if (area == 0) {
				/* Skip LEADIN. */
				assert(c->area[area].type == CD_LEADIN);
				continue;
			} else if (area == c->nareas - 1) {
				/* Skip LEADOUT. */
				assert(c->area[area].type == CD_LEADOUT);
				continue;
			}

			if (start <= blk
			 && blk < start + c->area[area].size / 2048) {
				/* Area found. */
				break;
			}

			start += c->area[area].size / 2048;
		}
		blk -= (unsigned long long) start;

		switch (c->area[area].type) {
		case CD_LEADIN:
		case CD_LEADOUT:
			assert(0);
		case CD_AUDIO:
			fixme();
			break;
		case CD_DATA:
			storage_simple_read(c->file[c->area[area].filenum],
					buf,
					2048,
					c->area[area].fileoffset + (unsigned long long) blk * 2048);
				/* Blocksize FIXME */
			break;
		case CD_PREGAP:
		case CD_POSTGAP:
			memset(buf, 0, 2352);
			break;
		default:
			assert(0);
		}
		buf += 2352;

		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */

		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */
		*buf++ = 0; /* FIXME */

		return 1;
	}
}

static long long
conv_cd_read(
	void *_c,
	void *_buf,
	unsigned long long buflen,
	unsigned long long pos
)
{
	struct conv_cd *c = _c;
	uint8_t *buf = _buf;
	long long size;

	size = 0;
	while (0 < buflen) {
		unsigned long long count;
		uint8_t cache[BLOCKSIZE];
		int ret;

		count = buflen;
		if (BLOCKSIZE - pos % BLOCKSIZE < count) {
			count = BLOCKSIZE - pos % BLOCKSIZE;
		}

		if (count == BLOCKSIZE) {
			ret = _conv_cd_read(c, buf, pos / BLOCKSIZE);
		} else {
			ret = _conv_cd_read(c, cache, pos / BLOCKSIZE);
			memcpy(buf, &cache[pos % BLOCKSIZE], count);
		}
		if (ret < 0) {
			/* Read error. */
			if (size == 0) {
				size = -1;
			}
			break;
		} else if (ret == 0) {
			/* End of CD. */
			break;
		}

		pos += count;
		buf += count;
		buflen -= count;
		size += count;
	}

	return size;
}

void *
conv_cd_from_cue_open(const char *path)
{
	unsigned int size;
	int fd;
	char buffer[16 * 1024];
	int ret;

	/*
	 * Check for size.
	 */
	fd = open(path, O_RDONLY, 0);
	size = read(fd, buffer, sizeof(buffer));
	close(fd);

	if (sizeof(buffer) <= size) {
		/*
		 * This size constraint is basically an educated guess,
		 * but it should do.
		 */
		fprintf(stderr, "%s: WARNING: %s.\n", progname,
				"No CUE sheet");
		return NULL;
	}

	/*
	 * Read cue sheet.
	 */
	fd = open(path, O_RDONLY, 0);
	ret = read(fd, buffer, size);
	assert(ret == size);
	ret = close(fd);
	assert(0 <= ret);
	buffer[size + 1]='\0';

	/*
	 * Parse cue sheet.
	 */
	return conv_cd_open(buffer, path);
}

int
conv_cd_from_cue_close(void *_c)
{
	return conv_cd_close(_c);
}

long long
conv_cd_from_cue_read(
	void *_c,
	void *_buf,
	unsigned long long buflen,
	unsigned long long pos
)
{
	return conv_cd_read(_c, _buf, buflen, pos);
}

void *
conv_cd_from_iso_open(const char *path)
{
	char buffer[16 * 1024];

	sprintf(buffer,
			"FILE \"%s\" BINARY\n"
			"  TRACK 01 MODE1/2048\n"
			"    PREGAP 00:02:00\n"
			"    INDEX 01 00:00:00\n"
			/* "    POSTGAP 00:02:00\n" */,
			path);

	return conv_cd_open(buffer, ".");
}

int
conv_cd_from_iso_close(void *_c)
{
	return conv_cd_close(_c);
}

long long
conv_cd_from_iso_read(
	void *_c,
	void *_buf,
	unsigned long long buflen,
	unsigned long long pos
)
{
	return conv_cd_read(_c, _buf, buflen, pos);
}
