/*
 * $Id: fau-camera-adapter.c,v 1.18 2010-12-09 14:54:08 vrsieh Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG	1

#define DRIVER_MAJOR	241
#define DRIVER_MAX	8

#define WIDTH_MAX	1920
#define HEIGHT_MAX	1080

#define MOD_NAME	"fau-camera-adapter"

#define NPAGES		(((WIDTH_MAX * HEIGHT_MAX * 3) + 4095) / 4096)

#include <linux/version.h>
#include <linux/module.h>

#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/wait.h>

#include "fau-camera-adapter.h"

MODULE_LICENSE("GPL");

static struct dev {
	struct pci_dev *pci_dev;

	uint32_t base;

	spinlock_t lock;
	struct fasync_struct *async_queue;
	wait_queue_head_t wait;

	struct {
		unsigned int width;
		unsigned int height;
		uint64_t *pagelist;
	} buf[2];
	int read;
	int recv;
	int full;
} dev_table[DRIVER_MAX];

static int
fau_driver_wait(struct file *file)
{
	struct dev *dev = file->private_data;

	if (! dev->full
	 && (file->f_flags & O_NONBLOCK)) {
		return -EAGAIN;
	}

	return wait_event_interruptible(dev->wait, dev->full);
}

static ssize_t
fau_driver_read(struct file *file, char __user *buf, size_t count, loff_t *ppos)
{
	struct dev *dev = file->private_data;
	unsigned int to_read;
	unsigned int size;
	unsigned int n;
	unsigned int i;
	int ret;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": read called\n");
#endif

	ret = fau_driver_wait(file);
	if (ret != 0) {
		return ret;
	}

	dev->full = 0;

	to_read = dev->read;

	size = 3 * dev->buf[to_read].width * dev->buf[to_read].height;

	if (count < size) {
#if DEBUG
		printk(KERN_INFO MOD_NAME ": %d %d\n", (int) count, (int) size);
#endif
		return -EINVAL;
	}
	count = size;

	i = 0;
	for (n = count; 0 < n; n -= size) {
		if (n < 4096) {
			size = n;
		} else {
			size = 4096;
		}
		if (copy_to_user(buf,
				phys_to_virt(dev->buf[to_read].pagelist[i] & ~0xfffUL),
				size)) {
#if DEBUG
			printk(KERN_INFO MOD_NAME ": EFAULT\n");
#endif
			return -EFAULT;
		}
		buf += size;
		i++;
	}

#if DEBUG
	printk(KERN_INFO MOD_NAME ": read: %d\n", (int) count);
#endif
	return count;
}

static int
fau_driver_ioctl(
	struct inode *inode,
	struct file *file,
	unsigned int cmd,
	unsigned long _param
)
{
	struct dev *dev = file->private_data;
	int ret;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": ioctl called\n");
#endif

	switch (cmd) {
	case FAU_CAMERA_ADAPTER_GET_GEOMETRY: {
		struct fau_camera_adapter_geometry info;

		/* Copy parameter values. */
		/* None. */

		/* Do work. */
		ret = fau_driver_wait(file);
		if (ret != 0) {
			break;
		}

		info.width = dev->buf[dev->read].width;
		info.height = dev->buf[dev->read].height;

		/* Copy return values. */
		if (copy_to_user((void *) _param, &info, sizeof(info)) != 0) {
			ret = -EFAULT;
			break;
		}
		ret = 0;
		break;
	    }
	default:
		ret = -EINVAL;
		break;
	}

#if DEBUG
	printk(KERN_INFO MOD_NAME ": ioctl done\n");
#endif

	return ret;
}

static int
fau_driver_fasync(int fd, struct file *file, int on)
{
	struct dev *dev = file->private_data;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": fasync called\n");
#endif

        return fasync_helper(fd, file, on, &dev->async_queue);
}

static unsigned int
fau_driver_poll(struct file *file, poll_table *wait)
{
	struct dev *dev = file->private_data;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": poll called\n");
#endif

	poll_wait(file, &dev->wait, wait);

	return dev->full;
}

static int
fau_driver_open(struct inode *inode, struct file *file)
{
	unsigned int minor = iminor(inode);
	struct dev *dev;
	unsigned int i;
	unsigned int j;
	void *virt;
	uint64_t phys;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": open called\n");
#endif

	if (minor < 0
	 || DRIVER_MAX <= minor
	 || ! dev_table[minor].pci_dev) {
		return -ENODEV;
	}
	dev = &dev_table[minor];

	/* Initialize buffers. */
	for (j = 0; j < 2; j++) {
		dev->buf[j].pagelist
			= kmalloc(sizeof(*dev->buf[j].pagelist) * (NPAGES + 1),
				GFP_USER);
		for (i = 0; i < NPAGES; i++) {
			virt = kmalloc(4096, GFP_USER);
			phys = virt_to_phys(virt);
			printk(KERN_INFO MOD_NAME ": %p > 0x%llx\n",
					virt, phys);
			dev->buf[j].pagelist[i] = phys | 0x1;
		}
		dev->buf[j].pagelist[i] = 0;
	}

	dev->recv = 0;
	dev->read = 1;
	dev->full = 0;

	/* Start DMA. */
	virt = dev->buf[dev->recv].pagelist;
	phys = virt_to_phys(virt);
	printk(KERN_INFO MOD_NAME ": %p > 0x%llx (list)\n",
			virt, phys);
	outl(phys | 0x1, dev->base + 0x4); /* FIXME */

	file->private_data = dev;
	return 0;
}

static int
fau_driver_release(struct inode *inode, struct file *file)
{
	struct dev *dev = file->private_data;
	unsigned int i;
	unsigned int j;
	uint64_t phys;
	void *virt;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": release called\n");
#endif

	/* Remove ASYNC attributes. */
	if (file->f_flags & FASYNC)
		fau_driver_fasync(-1, file, 0);

	/* Stop DMA. */
	outl(0, dev->base + 0x4); /* FIXME */

	/* Free buffers. */
	for (j = 0; j < 2; j++) {
		for (i = 0; i < NPAGES; i++) {
			phys = dev->buf[j].pagelist[i] & ~0x1;
			virt = phys_to_virt(phys);
			kfree(virt);
		}
		kfree(dev->buf[j].pagelist);
	}

	return 0;
}

static irqreturn_t
fau_driver_isr(int irq, void *_dev)
{
	struct dev *dev = _dev;
	void *virt;
	uint64_t phys;

#if DEBUG
	printk(KERN_INFO "%s called\n", __FUNCTION__);
#endif

	if (inl(dev->base) & 1) {
		/*
		 * Interrupt bit is set.
		 */
		/* Now we have at least one filled buffer. */
		dev->full = 1;

		/* Remember geometry. */
		dev->buf[dev->recv].width = inl(dev->base + 0x8);
		dev->buf[dev->recv].height = inl(dev->base + 0xc);

		/* Switch to other buffer. */
		dev->read ^= 1;
		dev->recv ^= 1;

		virt = dev->buf[dev->recv].pagelist;
		phys = virt_to_phys(virt);
		printk(KERN_INFO MOD_NAME ": %p > 0x%llx (list)\n",
				virt, phys);
		outl(phys | 0x1, dev->base + 0x4); /* FIXME */

		/* Notify waiting processes. */
		wake_up_interruptible(&dev->wait);
		kill_fasync(&dev->async_queue, SIGIO, POLL_IN);

		return IRQ_HANDLED;

	} else {
		/*
		 * Interrupt not from this adapter.
		 */
		return IRQ_NONE;
	}
}

static int __devinit
fau_driver_probe(struct pci_dev *pci_dev, const struct pci_device_id *id)
{
	struct dev *dev;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": probe called\n");
#endif

	/*
	 * Lookup free state structure.
	 */
	for (dev = dev_table; ; dev++) {
		if (dev == &dev_table[DRIVER_MAX]) {
			return -ENOMEM;
		}
		if (! dev->pci_dev) {
			break;
		}
	}
	dev->pci_dev = pci_dev;

	/*
	 * Request hardware resources.
	 */
	if (pci_enable_device(pci_dev) < 0) {
		printk(KERN_ERR MOD_NAME ": unable to enable device!\n");
		dev->pci_dev = NULL;
		return -EIO;
	}

	if (pci_request_regions(pci_dev, MOD_NAME) < 0) {
		printk(KERN_ERR MOD_NAME ": unable to request regions!\n");
		pci_disable_device(pci_dev);
		dev->pci_dev = NULL;
		return -EBUSY;
	}
	if (request_irq(pci_dev->irq, fau_driver_isr, IRQF_SHARED,
			MOD_NAME, dev)) {
		printk(KERN_ERR MOD_NAME ": unable to request IRQ!\n");
		pci_release_regions(pci_dev);
		pci_disable_device(pci_dev);
		dev->pci_dev = NULL;
		return -EBUSY;
	}
	pci_set_master(pci_dev);

	/*
	 * Initialize state.
	 */
	spin_lock_init(&dev->lock);
	dev->async_queue = NULL;
	init_waitqueue_head(&dev->wait);

	dev->base = pci_resource_start(pci_dev, 0);
	if (pci_resource_len(pci_dev, 0) != 0x10) BUG();
	/* if (pci_resource_flags(pci_dev, 0) != ...) BUG(); */

	return 0;
}

static void __devexit
fau_driver_remove(struct pci_dev *pci_dev)
{
	struct dev *dev;

#if DEBUG
	printk(KERN_INFO MOD_NAME ": remove called\n");
#endif

	for (dev = dev_table; ; dev++) {
		if (dev == &dev_table[DRIVER_MAX]) {
			printk(KERN_ERR MOD_NAME ": removing not registered device!\n");
			return;
		}
		if (dev->pci_dev == pci_dev) {
			break;
		}
	}

	free_irq(dev->pci_dev->irq, dev);
	pci_release_regions(dev->pci_dev);
	pci_disable_device(dev->pci_dev);

	dev->pci_dev = NULL;
}

static const struct pci_device_id __devinitdata pci_device[] = {
	{
		0xaffe, /* Vendor */
		0xcafe, /* Device */
		PCI_ANY_ID, /* Sub-Vendor */
		PCI_ANY_ID, /* Sub-Device */
		0, 0, 0
	},
	{ 0, 0, 0, 0, 0, 0, 0 }
};

MODULE_DEVICE_TABLE(pci, pci_device);

static struct pci_driver fau_driver = {
	.name = MOD_NAME,
	.id_table = pci_device,
	.probe = fau_driver_probe,
	.remove = __devexit_p(fau_driver_remove),
};

static int __init
fau_driver_init(void)
{
	static const struct file_operations fops = {
		.owner = THIS_MODULE,

		.llseek = no_llseek,
		.read = fau_driver_read,
		.ioctl = fau_driver_ioctl,
		.fasync = fau_driver_fasync,
		.poll = fau_driver_poll,

		.open = fau_driver_open,
		.release = fau_driver_release,
	};

#if DEBUG
	printk(KERN_INFO MOD_NAME ": init called\n");
#endif

	if (register_chrdev(DRIVER_MAJOR, MOD_NAME, &fops) != 0) {
		return -EIO;
	}

	if (pci_register_driver(&fau_driver) < 0) {
		unregister_chrdev(DRIVER_MAJOR, MOD_NAME);
		return -EIO;
	}

	return 0;
}

static void __exit
fau_driver_exit(void)
{
#if DEBUG
	printk(KERN_INFO MOD_NAME ": exit called\n");
#endif

	pci_unregister_driver(&fau_driver);

	unregister_chrdev(DRIVER_MAJOR, MOD_NAME);
}

module_init(fau_driver_init);
module_exit(fau_driver_exit);
