/* render1.h
 *
 * By Byron C. Darrah
 * 5/13/94, Thursday
 *
 * render1 is a renderer which may be used with the generic vr simulator.
 * The characteristics of this renderer are:
 *
 * - Renders polygonal objects.
 * - 16 colors (this is very easily changed for the render1 module, but if you
 *   change it, don't forget to implement the new color scheme in the
 *   (system-dependent) display_list module).
 * - No shading or texture mapping - polygons use flat colors.
 *
 * Here is a brief description of each of the functions provided to
 * interface the renderer:
 *
 *
 * o    void rend_init(int *argc, char *argv[], float init_perspect);
 * 
 *      Fire up the renderer for use.  Call this once at the beginning of
 *      your simulation, and call rend_end() once at the end.  argc and argv
 *      are expected to be what they usually would be, as derived from the
 *      command line, except note that argc is a _pointer_ to an int.  This
 *      is because rend_init() may decide to destructively remove certain
 *      arguments from the list.  init_perspect is the focal length of the
 *      camera lense (note that the camera position and focus point are
 *      _not_ used to determine the focal length, this is).  Roughly, a small
 *      number gives you a wide-angle-style lense, and a big value gives you
 *      telephoto.
 *
 * o    void rend_end(void);
 *
 *      Call this routine before exiting a simulation, to shut down the
 *      renderer and free up any resources necessary that have been allocated.
 *
 * o    void rend_setsize(void);
 *
 *      Synchronize the scale of drawing operations with the size of the
 *      actual output window.
 *
 * o    void rend_set_camera(real_point_t *position,
 *                           real_point_t *focus,
 *                           real_point_t *up);
 *
 *      This function alters the camera matrix.  Note that each time the
 *      camera matrix is updated, all of the polygons currently submitted
 *      to render1 have to be retransformed and resorted, so use this function
 *      with care.  You should never need to use this function more than
 *      once per frame.  The focus represents a point at which the camera is
 *      aimed.  The position is the location of the camera. up represents the
 *      "top" of the camera -- a vector from the postion point to the up point
 *      tells you which way the camera thinks is up.
 *
 * o    void rend_register_obj(int obj_id, int npolygons);
 *
 *      Objects call this function to register themselves with the renderer.
 *      Only registered objects may submit polygons. obj_id is the object's
 *      unique id, and npolygons is the number of polygons the object
 *      will use to display itself (ie: for a cube, npolygons is 6).
 *
 * o    int  rend_submit_plist(int obj_id, polygon_t *polygons);
 *
 *      This is the function objects use to submit their polygons to render1
 *      for displaying.  obj_id is the id of a registered object, and polygons
 *      is an array of polygons.  It returns R1_NORMAL when successful, or
 *      R1_OBJ_UNKNOWN if an unregistered object id is given.
 *
 * o    void rend_rm_plist(int obj_id);
 *
 *      This is the function objects use to remove their polygons from the
 *      display.  Note: this does not unregister an object, it merely
 *      "unsubmits" it's polygons.
 *
 * o    void rend_transform_update_all(void);
 *
 *      This functions performs the camera matrix transformation on all
 *      currently submitted polygons.  It is used after rend_set_camera(),
 *      to update the polygons with respect to the new camera position.
 *      Note that polygons are automatically transformed whenever they are
 *      submitted.  Therefore, if you can guarantee that all submitted
 *      polygons will be either removed or resubmitted before the next frame is
 *      rendered, there is no need to call this function after
 *      rend_set_camera().
 *
 * o    void rend_frame(void);
 *
 *      Actually output all currently submitted polygons to the display.
 *
 * o    void rend_freeze(void);
 *
 *      Pause the renderer, leaving the most recently rendered frame visible.
 *      Currently, this function is not guaranteed to ever return.
 */

#ifndef render1_h
#define render1_h

#define DEFAULT_PERSPECTIVE 200.0

/* Here are status codes that render1 can return */
#define R1_NORMAL 0
#define R1_OBJ_UNKNOWN 300
#define TIMEOUT        301
#define RESIZE         302
#define SELECTION      303

/*---------------------------------------------------------------------------*/
/* Here are the data structures used to interface the renderer, render1.     */
/*---------------------------------------------------------------------------*/

/* Here are the colors currently supported by the renderer.  It is up to
 * The display_list module to actually provide these.
 */
#define BLACK        0
#define DARKRED      1
#define DARKGREEN    2
#define DARKBLUE     3
#define DARKBROWN    4
#define DARKMAGENTA  5
#define DARKCYAN     6
#define GREY1        7
#define GREY2        8
#define RED          9
#define GREEN       10
#define BLUE        11
#define BROWN       12
#define MAGENTA     13
#define CYAN        14
#define WHITE       15


typedef struct { float  x, y, z;
               } real_point_t;

typedef struct { float  i, j, k;
               } vector_t;

typedef struct { short x, y;
              } int_point_t;

typedef struct plygon_struct
   {
     int                   color;        /* A color table entry */
     real_point_t         *sim_points;
     int                   npoints;
     int_point_t          *int_points;
   } polygon_t;

void rend_init(int *argc, char *argv[], float init_perspect);
void rend_end(void);
void rend_setsize(void);
void rend_set_camera(real_point_t *position,
                     real_point_t *focus,
                     real_point_t *up);
void rend_register_obj(int obj_id, int npolygons);
int  rend_submit_plist(int obj_id, polygon_t *polygons);
int  rend_rm_plist(int obj_id);
void rend_transform_update_all(void);
void rend_frame(void);
int  rend_freeze(unsigned long interval);
void rend_bell(void);

#endif  /* render1_h */
