/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "cdw_regex.h"
#include "cdw_debug.h"
#include "gettext.h"



/**
   \file cdw_regex.c

   \brief Generic functions for extended regex type.

   The file defines functions that operate on cdw_regex_t type
   variables. The type combines several fields related to processing
   regular expressions, making dealing with regular expressions easier.
*/




/**
   \brief Allocate memory for regular expression data, compile regular expressions

   Allocate memory for all variables of type regex_t and regmatch_t that
   are part of cdw_regex_t type variables located in given \p table.

   Function calls cdw_regex_regcomp_error_handler() on errors.
   Call cdw_regex_clean_up_regexes_table() to deallocate all memory
   allocated by this function.

   \param table - table of variables of type cdw_regex_t to initialize.
                  The table needs to have guard element with debug_id == -1
		  as a guard value.

   \return CDW_GEN_ERROR on errors
   \return CDW_OK on success
*/
cdw_rv_t cdw_regex_prepare_regexes_table(cdw_regex_t table[])
{
	int i = 0;
	while (table[i].debug_id != -1) {
		cdw_assert (table[i].handler != (cdw_regex_handler_t) NULL,
			    "ERROR: regex handler #%d is NULL (id = %lld / \"%s\")\n",
			    i, table[i].debug_id, table[i].debug_label);
		table[i].regex = (regex_t *) malloc(sizeof(regex_t));
		if (table[i].regex == (regex_t *) NULL) {
			cdw_vdm ("ERROR: failed to allocate memory for regex #%d\n", i);
			return CDW_ERROR;
		}
		int rv = regcomp(table[i].regex, table[i].pattern, table[i].regcomp_opts);
		table[i].matches = (regmatch_t *) calloc(1, (table[i].regex->re_nsub + 1) * sizeof(regmatch_t));
		if (rv) {
			cdw_regex_regcomp_error_handler(__func__, rv, table[i].regex, table[i].debug_id);
		}
		i++;
	}

	return CDW_OK;
}





/**
   \brief Compare data in buffer against regular expressions, call handler function

   This function calls regexec() to compare data from \p buffer against
   every regular expression from \p table. If there is a match, the
   function will call proper handler function associated with matched
   regular expression.

   \param table  - table of variables of type cdw_regex_t to initialize.
                   The table needs to have guard element with debug_id == -1
		   as a guard value.
   \param buffer - buffer with data to be compared against regular
                   expressions from \p table. Data in buffer must have
		   proper ending '\0' char.
*/
void cdw_regex_execute_regexes_table(cdw_regex_t table[], const char *buffer)
{
	int i = 0;
	while (table[i].debug_id != -1) {
		int rv = regexec(table[i].regex, buffer, table[i].regex->re_nsub + 1, table[i].matches, 0);
		if (rv == 0) {
			table[i].handler(table[i].regex, table[i].matches);
			cdw_sdm ("    MATCHED LINE \"%s\"\n", buffer);
			return;
		}
		i++;
	}

	cdw_sdm ("NOT MATCHED LINE \"%s\"\n", buffer);
	return;
}





/**
   \brief Free all memory used by regular expression variables in the table

   Function deallocates all memory associated with variables of type
   cdw_regex_t, stored in \p table. This function should be used to free
   memory that was allocated with cdw_regex_prepare_regexes_table().

   \param table - table of variables of type cdw_regex_t to initialize.
                  The table needs to have guard element with debug_id == -1
		  as a guard value.
*/
void cdw_regex_clean_up_regexes_table(cdw_regex_t table[])
{
	int i = 0;
	while (table[i].debug_id != -1) {
		if (table[i].regex != (regex_t *) NULL) {
			/* regfree() only frees memory allocated by
			   regcomp(), you still have to free malloced()
			   pointer, i.e. [i].regex */
			regfree(table[i].regex);
			free(table[i].regex);
			table[i].regex = (regex_t *) NULL;
		}
		if (table[i].matches != (regmatch_t *) NULL) {
			free(table[i].matches);
			table[i].matches = (regmatch_t *) NULL;
		}
		i++;
	}

	return;
}





/**
   \brief Display information about error in regcomp() call and call exit(1)

   \param caller_name - name of function calling this handler
   \param errcode - error value returned by regcomp()
   \param regex - regex string for which regcomp() returned error
   \param id - unique int value, specific for given regex string
*/
void cdw_regex_regcomp_error_handler(const char *caller_name, int errcode, const regex_t *regex, cdw_id_t id)
{
	/* buffer for explanation of regcomp failure; WARNING: arbitrary buffer size! */
	const size_t e_size = 100;
	char errbuf[e_size];

	regerror(errcode, regex, errbuf, e_size);
	/* 2TRANS: this is debug message displayed in console, regcomp is a
	   function compiling regular expressions, used in cdw code */
	fprintf(stderr, _("ERROR: REGCOMP ERROR in %s(): regular expression = \"%s\", id = %lld\n"), caller_name, errbuf, id);

	exit(EXIT_FAILURE);
}





/**
   \brief Get a substring (regular expression token) from given data buffer

   Function copies a token (a substring, subexpression) from \p buffer to
   \p submatch. Subexpression to copy is specified by \p sub index. Start
   and end of given subexpression in \p buffer is taken from \p matches.

   Data in \p buffer is the data that was successfully compared against
   a regular expression. The matching comparison results in \p matches
   with valid data.

   Size of \p submatch must be no smaller than size of \p buffer.

   TODO: this function probably should be inline function (or not).

   \param matches - result of successful call to regexec()
   \param sub - index of subexpression to extract from \p buffer
   \param buffer - buffer with data that was compared against regular expression
   \param submatch - buffer in which the function will put a submatch

   \return -1 on errors
   \return non-negative length of subexpression copied to \p submatch
*/
int cdw_regex_get_submatch(regmatch_t *matches, unsigned int sub, char *buffer, char *submatch)
{
	cdw_assert (buffer != (char *) NULL, "ERROR: data buffer pointer is NULL\n");
	cdw_assert (submatch != (char *) NULL, "ERROR: submatch buffer pointer is NULL\n");

	int len = matches[sub].rm_eo - matches[sub].rm_so;
	if (len < 0) {
		cdw_vdm ("ERROR: len of subexpr #%d is negative: %d\n", sub, len);
		return -1;
	} else {
		strncpy(submatch, (char *) &buffer[matches[sub].rm_so], (size_t) len);
		submatch[len] = '\0';
		return len;
	}
}
