/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools;

import htsjdk.samtools.BamIndexValidator;
import htsjdk.samtools.CoordinateSortedPairInfoMap;
import htsjdk.samtools.FileTruncatedException;
import htsjdk.samtools.ReservedTagConstants;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFormatException;
import htsjdk.samtools.SAMProgramRecord;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMRecordIterator;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSortOrderChecker;
import htsjdk.samtools.SAMTag;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.SAMValidationError;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.ValidationStringency;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.reference.ReferenceSequenceFileWalker;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.CloseableIterator;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.FastqQualityFormat;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.QualityEncodingDetector;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class SamFileValidator {
    private static final Log log = Log.getInstance(SamFileValidator.class);
    private final PrintWriter out;
    private Histogram<SAMValidationError.Type> errorsByType;
    private PairEndInfoMap pairEndInfoByName;
    private ReferenceSequenceFileWalker refFileWalker;
    private SAMSequenceDictionary samSequenceDictionary;
    private boolean verbose;
    private int maxVerboseOutput;
    private SAMSortOrderChecker orderChecker;
    private Set<SAMValidationError.Type> errorsToIgnore;
    private boolean ignoreWarnings;
    private boolean bisulfiteSequenced;
    private BamIndexValidator.IndexValidationStringency indexValidationStringency;
    private boolean sequenceDictionaryEmptyAndNoWarningEmitted;
    private int numWarnings;
    private int numErrors;
    private final int maxTempFiles;
    private int qualityNotStoredErrorCount = 0;
    public static final int MAX_QUALITY_NOT_STORED_ERRORS = 100;

    public SamFileValidator(PrintWriter out, int maxTempFiles) {
        this.out = out;
        this.maxTempFiles = maxTempFiles;
        this.errorsByType = new Histogram();
        this.refFileWalker = null;
        this.maxVerboseOutput = 100;
        this.indexValidationStringency = BamIndexValidator.IndexValidationStringency.NONE;
        this.errorsToIgnore = EnumSet.noneOf(SAMValidationError.Type.class);
        this.verbose = false;
        this.ignoreWarnings = false;
        this.bisulfiteSequenced = false;
        this.sequenceDictionaryEmptyAndNoWarningEmitted = false;
        this.numWarnings = 0;
        this.numErrors = 0;
    }

    Histogram<SAMValidationError.Type> getErrorsByType() {
        return this.errorsByType;
    }

    public void setErrorsToIgnore(Collection<SAMValidationError.Type> types) {
        if (!types.isEmpty()) {
            this.errorsToIgnore = EnumSet.copyOf(types);
        }
    }

    public void setIgnoreWarnings(boolean ignoreWarnings) {
        this.ignoreWarnings = ignoreWarnings;
    }

    public boolean validateSamFileSummary(SamReader samReader, ReferenceSequenceFile reference) {
        this.init(reference, samReader.getFileHeader());
        this.validateSamFile(samReader, this.out);
        boolean result = this.errorsByType.isEmpty();
        if (this.errorsByType.getCount() > 0.0) {
            Histogram<String> errorsAndWarningsByType = new Histogram<String>("Error Type", "Count");
            for (Histogram.Bin<SAMValidationError.Type> bin : this.errorsByType.values()) {
                errorsAndWarningsByType.increment(bin.getId().getHistogramString(), bin.getValue());
            }
            MetricsFile metricsFile = new MetricsFile();
            this.errorsByType.setBinLabel("Error Type");
            this.errorsByType.setValueLabel("Count");
            metricsFile.setHistogram(errorsAndWarningsByType);
            metricsFile.write(this.out);
        }
        this.cleanup();
        return result;
    }

    public boolean validateSamFileVerbose(SamReader samReader, ReferenceSequenceFile reference) {
        this.init(reference, samReader.getFileHeader());
        try {
            this.validateSamFile(samReader, this.out);
        }
        catch (MaxOutputExceededException e) {
            this.out.println("Maximum output of [" + this.maxVerboseOutput + "] errors reached.");
        }
        boolean result = this.errorsByType.isEmpty();
        this.cleanup();
        return result;
    }

    public void validateBamFileTermination(File inputFile) {
        BufferedInputStream inputStream = null;
        try {
            inputStream = IOUtil.toBufferedStream(new FileInputStream(inputFile));
            if (!BlockCompressedInputStream.isValidFile(inputStream)) {
                return;
            }
            BlockCompressedInputStream.FileTermination terminationState = BlockCompressedInputStream.checkTermination(inputFile);
            if (terminationState.equals((Object)BlockCompressedInputStream.FileTermination.DEFECTIVE)) {
                this.addError(new SAMValidationError(SAMValidationError.Type.TRUNCATED_FILE, "BAM file has defective last gzip block", inputFile.getPath()));
            } else if (terminationState.equals((Object)BlockCompressedInputStream.FileTermination.HAS_HEALTHY_LAST_BLOCK)) {
                this.addError(new SAMValidationError(SAMValidationError.Type.BAM_FILE_MISSING_TERMINATOR_BLOCK, "Older BAM file -- does not have terminator block", inputFile.getPath()));
            }
        }
        catch (IOException e) {
            throw new SAMException("IOException", e);
        }
        finally {
            if (inputStream != null) {
                CloserUtil.close(inputStream);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void validateSamFile(SamReader samReader, PrintWriter out) {
        try {
            this.validateHeader(samReader.getFileHeader());
            this.orderChecker = new SAMSortOrderChecker(samReader.getFileHeader().getSortOrder());
            this.validateSamRecordsAndQualityFormat(samReader, samReader.getFileHeader());
            this.validateUnmatchedPairs();
            if (this.indexValidationStringency != BamIndexValidator.IndexValidationStringency.NONE) {
                try {
                    if (this.indexValidationStringency == BamIndexValidator.IndexValidationStringency.LESS_EXHAUSTIVE) {
                        BamIndexValidator.lessExhaustivelyTestIndex(samReader);
                    } else {
                        BamIndexValidator.exhaustivelyTestIndex(samReader);
                    }
                }
                catch (Exception e) {
                    this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_INDEX_FILE_POINTER, e.getMessage(), null));
                }
            }
            if (this.errorsByType.isEmpty()) {
                out.println("No errors found");
            }
        }
        finally {
            out.flush();
        }
    }

    private void validateUnmatchedPairs() {
        InMemoryPairEndInfoMap inMemoryPairMap;
        if (this.pairEndInfoByName instanceof CoordinateSortedPairEndInfoMap) {
            inMemoryPairMap = new InMemoryPairEndInfoMap();
            Iterator it = this.pairEndInfoByName.iterator();
            while (it.hasNext()) {
                Map.Entry entry = (Map.Entry)it.next();
                PairEndInfo pei = inMemoryPairMap.remove(((PairEndInfo)entry.getValue()).readReferenceIndex, (String)entry.getKey());
                if (pei != null) {
                    List<SAMValidationError> errors = pei.validateMates((PairEndInfo)entry.getValue(), (String)entry.getKey());
                    for (SAMValidationError error : errors) {
                        this.addError(error);
                    }
                    continue;
                }
                inMemoryPairMap.put(((PairEndInfo)entry.getValue()).mateReferenceIndex, (String)entry.getKey(), (PairEndInfo)entry.getValue());
            }
            it.close();
        } else {
            inMemoryPairMap = (InMemoryPairEndInfoMap)this.pairEndInfoByName;
        }
        for (Map.Entry entry : inMemoryPairMap) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MATE_NOT_FOUND, "Mate not found for paired read", (String)entry.getKey()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void validateSamRecordsAndQualityFormat(Iterable<SAMRecord> samRecords, SAMFileHeader header) {
        SAMRecordIterator iter = (SAMRecordIterator)samRecords.iterator();
        ProgressLogger progress = new ProgressLogger(log, 10000000, "Validated Read");
        QualityEncodingDetector qualityDetector = new QualityEncodingDetector();
        try {
            while (iter.hasNext()) {
                long recordNumber;
                SAMRecord record;
                block17: {
                    record = (SAMRecord)iter.next();
                    qualityDetector.add(record);
                    recordNumber = progress.getCount() + 1L;
                    List<SAMValidationError> errors = record.isValid();
                    if (errors != null) {
                        for (SAMValidationError error : errors) {
                            error.setRecordNumber(recordNumber);
                            this.addError(error);
                        }
                    }
                    this.validateMateFields(record, recordNumber);
                    boolean hasValidSortOrder = this.validateSortOrder(record, recordNumber);
                    this.validateReadGroup(record, header);
                    boolean cigarIsValid = this.validateCigar(record, recordNumber);
                    if (cigarIsValid) {
                        try {
                            this.validateNmTag(record, recordNumber);
                        }
                        catch (SAMException e) {
                            if (!hasValidSortOrder) break block17;
                            throw e;
                        }
                    }
                }
                this.validateSecondaryBaseCalls(record, recordNumber);
                this.validateTags(record, recordNumber);
                if (this.sequenceDictionaryEmptyAndNoWarningEmitted && !record.getReadUnmappedFlag()) {
                    this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_SEQUENCE_DICTIONARY, "Sequence dictionary is empty", null));
                    this.sequenceDictionaryEmptyAndNoWarningEmitted = false;
                }
                if (this.qualityNotStoredErrorCount++ < 100 && record.getBaseQualityString().equals("*")) {
                    this.addError(new SAMValidationError(SAMValidationError.Type.QUALITY_NOT_STORED, "QUAL field is set to * (unspecified quality scores), this is allowed by the SAM specification but many tools expect reads to include qualities ", record.getReadName(), recordNumber));
                }
                progress.record(record);
            }
            try {
                FastqQualityFormat format;
                if (progress.getCount() > 0L && (format = qualityDetector.generateBestGuess(QualityEncodingDetector.FileContext.SAM, FastqQualityFormat.Standard)) != FastqQualityFormat.Standard) {
                    this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_QUALITY_FORMAT, String.format("Detected %s quality score encoding, but expected %s.", new Object[]{format, FastqQualityFormat.Standard}), null));
                }
            }
            catch (SAMException e) {
                this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_QUALITY_FORMAT, e.getMessage(), null));
            }
        }
        catch (SAMFormatException e) {
            String msg = "SAMFormatException on record " + progress.getCount() + 1;
            this.out.println(msg);
            throw new SAMException(msg, e);
        }
        catch (FileTruncatedException e) {
            this.addError(new SAMValidationError(SAMValidationError.Type.TRUNCATED_FILE, "File is truncated", null));
        }
        finally {
            iter.close();
        }
    }

    private void validateReadGroup(SAMRecord record, SAMFileHeader header) {
        SAMReadGroupRecord rg = record.getReadGroup();
        if (rg == null) {
            this.addError(new SAMValidationError(SAMValidationError.Type.RECORD_MISSING_READ_GROUP, "A record is missing a read group", record.getReadName()));
        } else if (header.getReadGroup(rg.getId()) == null) {
            this.addError(new SAMValidationError(SAMValidationError.Type.READ_GROUP_NOT_FOUND, "A record has a read group not found in the header: ", record.getReadName() + ", " + rg.getReadGroupId()));
        }
    }

    private void validateTags(SAMRecord record, long recordNumber) {
        for (SAMRecord.SAMTagAndValue tagAndValue : record.getAttributes()) {
            if (!(tagAndValue.value instanceof Long)) continue;
            this.addError(new SAMValidationError(SAMValidationError.Type.TAG_VALUE_TOO_LARGE, "Numeric value too large for tag " + tagAndValue.tag, record.getReadName(), recordNumber));
        }
    }

    private void validateSecondaryBaseCalls(SAMRecord record, long recordNumber) {
        String u2;
        String e2 = (String)record.getAttribute(SAMTag.E2.name());
        if (e2 != null) {
            if (e2.length() != record.getReadLength()) {
                this.addError(new SAMValidationError(SAMValidationError.Type.MISMATCH_READ_LENGTH_AND_E2_LENGTH, String.format("E2 tag length (%d) != read length (%d)", e2.length(), record.getReadLength()), record.getReadName(), recordNumber));
            }
            byte[] bases = record.getReadBases();
            byte[] secondaryBases = StringUtil.stringToBytes(e2);
            for (int i = 0; i < Math.min(bases.length, secondaryBases.length); ++i) {
                if (SequenceUtil.isNoCall(bases[i]) || SequenceUtil.isNoCall(secondaryBases[i]) || !SequenceUtil.basesEqual(bases[i], secondaryBases[i])) continue;
                this.addError(new SAMValidationError(SAMValidationError.Type.E2_BASE_EQUALS_PRIMARY_BASE, String.format("Secondary base call  (%c) == primary base call (%c)", Character.valueOf((char)secondaryBases[i]), Character.valueOf((char)bases[i])), record.getReadName(), recordNumber));
                break;
            }
        }
        if ((u2 = (String)record.getAttribute(SAMTag.U2.name())) != null && u2.length() != record.getReadLength()) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MISMATCH_READ_LENGTH_AND_U2_LENGTH, String.format("U2 tag length (%d) != read length (%d)", u2.length(), record.getReadLength()), record.getReadName(), recordNumber));
        }
    }

    private boolean validateCigar(SAMRecord record, long recordNumber) {
        return record.getReadUnmappedFlag() || this.validateCigar(record, recordNumber, true);
    }

    private boolean validateMateCigar(SAMRecord record, long recordNumber) {
        return this.validateCigar(record, recordNumber, false);
    }

    private boolean validateCigar(SAMRecord record, long recordNumber, boolean isReadCigar) {
        ValidationStringency savedStringency = record.getValidationStringency();
        record.setValidationStringency(ValidationStringency.LENIENT);
        List<SAMValidationError> errors = isReadCigar ? record.validateCigar(recordNumber) : SAMUtils.validateMateCigar(record, recordNumber);
        record.setValidationStringency(savedStringency);
        if (errors == null) {
            return true;
        }
        boolean valid = true;
        for (SAMValidationError error : errors) {
            this.addError(error);
            valid = false;
        }
        return valid;
    }

    private boolean validateSortOrder(SAMRecord record, long recordNumber) {
        SAMRecord prev = this.orderChecker.getPreviousRecord();
        boolean isValidSortOrder = this.orderChecker.isSorted(record);
        if (!isValidSortOrder) {
            this.addError(new SAMValidationError(SAMValidationError.Type.RECORD_OUT_OF_ORDER, String.format("The record is out of [%s] order, prior read name [%s], prior coodinates [%d:%d]", record.getHeader().getSortOrder().name(), prev.getReadName(), prev.getReferenceIndex(), prev.getAlignmentStart()), record.getReadName(), recordNumber));
        }
        return isValidSortOrder;
    }

    private void init(ReferenceSequenceFile reference, SAMFileHeader header) {
        this.pairEndInfoByName = header.getSortOrder() == SAMFileHeader.SortOrder.coordinate ? new CoordinateSortedPairEndInfoMap() : new InMemoryPairEndInfoMap();
        if (reference != null) {
            this.refFileWalker = new ReferenceSequenceFileWalker(reference);
            this.samSequenceDictionary = reference.getSequenceDictionary();
        }
    }

    private void cleanup() {
        this.errorsByType = null;
        this.pairEndInfoByName = null;
        this.refFileWalker = null;
    }

    private void validateNmTag(SAMRecord record, long recordNumber) {
        if (!record.getReadUnmappedFlag()) {
            ReferenceSequence refSequence;
            int actualNucleotideDiffs;
            Integer tagNucleotideDiffs = record.getIntegerAttribute(ReservedTagConstants.NM);
            if (tagNucleotideDiffs == null) {
                this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_TAG_NM, "NM tag (nucleotide differences) is missing", record.getReadName(), recordNumber));
            } else if (this.refFileWalker != null && !tagNucleotideDiffs.equals(actualNucleotideDiffs = SequenceUtil.calculateSamNmTag(record, (refSequence = this.refFileWalker.get(record.getReferenceIndex())).getBases(), 0, this.isBisulfiteSequenced()))) {
                this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_TAG_NM, "NM tag (nucleotide differences) in file [" + tagNucleotideDiffs + "] does not match reality [" + actualNucleotideDiffs + "]", record.getReadName(), recordNumber));
            }
        }
    }

    private void validateMateFields(SAMRecord record, long recordNumber) {
        if (!record.getReadPairedFlag() || record.isSecondaryOrSupplementary()) {
            return;
        }
        this.validateMateCigar(record, recordNumber);
        PairEndInfo pairEndInfo = this.pairEndInfoByName.remove(record.getReferenceIndex(), record.getReadName());
        if (pairEndInfo == null) {
            this.pairEndInfoByName.put(record.getMateReferenceIndex(), record.getReadName(), new PairEndInfo(record, recordNumber));
        } else {
            List<SAMValidationError> errors = pairEndInfo.validateMates(new PairEndInfo(record, recordNumber), record.getReadName());
            for (SAMValidationError error : errors) {
                this.addError(error);
            }
        }
    }

    private void validateHeader(SAMFileHeader fileHeader) {
        for (SAMValidationError error : fileHeader.getValidationErrors()) {
            this.addError(error);
        }
        if (fileHeader.getVersion() == null) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_VERSION_NUMBER, "Header has no version number", null));
        } else if (!SAMFileHeader.ACCEPTABLE_VERSIONS.contains(fileHeader.getVersion())) {
            this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_VERSION_NUMBER, "Header version: " + fileHeader.getVersion() + " does not match any of the acceptable versions: " + StringUtil.join(", ", SAMFileHeader.ACCEPTABLE_VERSIONS.toArray(new String[0])), null));
        }
        if (fileHeader.getSequenceDictionary().isEmpty()) {
            this.sequenceDictionaryEmptyAndNoWarningEmitted = true;
        } else if (this.samSequenceDictionary != null && !fileHeader.getSequenceDictionary().isSameDictionary(this.samSequenceDictionary)) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MISMATCH_FILE_SEQ_DICT, "Mismatch between file and sequence dictionary", null));
        }
        if (fileHeader.getReadGroups().isEmpty()) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_READ_GROUP, "Read groups is empty", null));
        }
        List<SAMProgramRecord> pgs = fileHeader.getProgramRecords();
        for (int i = 0; i < pgs.size() - 1; ++i) {
            for (int j = i + 1; j < pgs.size(); ++j) {
                if (!pgs.get(i).getProgramGroupId().equals(pgs.get(j).getProgramGroupId())) continue;
                this.addError(new SAMValidationError(SAMValidationError.Type.DUPLICATE_PROGRAM_GROUP_ID, "Duplicate program group id: " + pgs.get(i).getProgramGroupId(), null));
            }
        }
        List<SAMReadGroupRecord> rgs = fileHeader.getReadGroups();
        HashSet<String> readGroupIDs = new HashSet<String>();
        for (SAMReadGroupRecord record : rgs) {
            String readGroupID = record.getReadGroupId();
            if (readGroupIDs.contains(readGroupID)) {
                this.addError(new SAMValidationError(SAMValidationError.Type.DUPLICATE_READ_GROUP_ID, "Duplicate read group id: " + readGroupID, null));
            } else {
                readGroupIDs.add(readGroupID);
            }
            String platformValue = record.getPlatform();
            if (platformValue == null || "".equals(platformValue)) {
                this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_PLATFORM_VALUE, "A platform (PL) attribute was not found for read group ", readGroupID));
                continue;
            }
            try {
                SAMReadGroupRecord.PlatformValue.valueOf(platformValue.toUpperCase());
            }
            catch (IllegalArgumentException e) {
                this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_PLATFORM_VALUE, "The platform (PL) attribute (" + platformValue + ") + was not one of the valid values for read group ", readGroupID));
            }
        }
    }

    public int getNumWarnings() {
        return this.numWarnings;
    }

    public int getNumErrors() {
        return this.numErrors;
    }

    private void addError(SAMValidationError error) {
        if (this.errorsToIgnore.contains((Object)error.getType())) {
            return;
        }
        switch (error.getType().severity) {
            case WARNING: {
                if (this.ignoreWarnings) {
                    return;
                }
                ++this.numWarnings;
                break;
            }
            case ERROR: {
                ++this.numErrors;
                break;
            }
            default: {
                throw new SAMException("Unknown SAM validation error severity: " + (Object)((Object)error.getType().severity));
            }
        }
        this.errorsByType.increment(error.getType());
        if (this.verbose) {
            this.out.println(error);
            this.out.flush();
            if (this.errorsByType.getCount() >= (double)this.maxVerboseOutput) {
                throw new MaxOutputExceededException();
            }
        }
    }

    public void setVerbose(boolean verbose, int maxVerboseOutput) {
        this.verbose = verbose;
        this.maxVerboseOutput = maxVerboseOutput;
    }

    public boolean isBisulfiteSequenced() {
        return this.bisulfiteSequenced;
    }

    public void setBisulfiteSequenced(boolean bisulfiteSequenced) {
        this.bisulfiteSequenced = bisulfiteSequenced;
    }

    @Deprecated
    public SamFileValidator setValidateIndex(boolean validateIndex) {
        return this.setIndexValidationStringency(validateIndex ? BamIndexValidator.IndexValidationStringency.EXHAUSTIVE : BamIndexValidator.IndexValidationStringency.NONE);
    }

    public SamFileValidator setIndexValidationStringency(BamIndexValidator.IndexValidationStringency stringency) {
        this.indexValidationStringency = stringency;
        return this;
    }

    private static class InMemoryPairEndInfoMap
    implements PairEndInfoMap {
        private final Map<String, PairEndInfo> map = new HashMap<String, PairEndInfo>();

        private InMemoryPairEndInfoMap() {
        }

        @Override
        public void put(int mateReferenceIndex, String key, PairEndInfo value) {
            if (mateReferenceIndex != value.mateReferenceIndex) {
                throw new IllegalArgumentException("mateReferenceIndex does not agree with PairEndInfo");
            }
            this.map.put(key, value);
        }

        @Override
        public PairEndInfo remove(int mateReferenceIndex, String key) {
            return this.map.remove(key);
        }

        @Override
        public CloseableIterator<Map.Entry<String, PairEndInfo>> iterator() {
            final Iterator<Map.Entry<String, PairEndInfo>> it = this.map.entrySet().iterator();
            return new CloseableIterator<Map.Entry<String, PairEndInfo>>(){

                @Override
                public void close() {
                }

                @Override
                public boolean hasNext() {
                    return it.hasNext();
                }

                @Override
                public Map.Entry<String, PairEndInfo> next() {
                    return (Map.Entry)it.next();
                }

                @Override
                public void remove() {
                    it.remove();
                }
            };
        }
    }

    private class CoordinateSortedPairEndInfoMap
    implements PairEndInfoMap {
        private final CoordinateSortedPairInfoMap<String, PairEndInfo> onDiskMap;

        private CoordinateSortedPairEndInfoMap() {
            this.onDiskMap = new CoordinateSortedPairInfoMap<String, PairEndInfo>(SamFileValidator.this.maxTempFiles, new Codec());
        }

        @Override
        public void put(int mateReferenceIndex, String key, PairEndInfo value) {
            this.onDiskMap.put(mateReferenceIndex, key, value);
        }

        @Override
        public PairEndInfo remove(int mateReferenceIndex, String key) {
            return this.onDiskMap.remove(mateReferenceIndex, key);
        }

        @Override
        public CloseableIterator<Map.Entry<String, PairEndInfo>> iterator() {
            return this.onDiskMap.iterator();
        }

        private class Codec
        implements CoordinateSortedPairInfoMap.Codec<String, PairEndInfo> {
            private DataInputStream in;
            private DataOutputStream out;

            private Codec() {
            }

            @Override
            public void setOutputStream(OutputStream os) {
                this.out = new DataOutputStream(os);
            }

            @Override
            public void setInputStream(InputStream is) {
                this.in = new DataInputStream(is);
            }

            @Override
            public void encode(String key, PairEndInfo record) {
                try {
                    this.out.writeUTF(key);
                    this.out.writeInt(record.readAlignmentStart);
                    this.out.writeInt(record.readReferenceIndex);
                    this.out.writeBoolean(record.readNegStrandFlag);
                    this.out.writeBoolean(record.readUnmappedFlag);
                    this.out.writeUTF(record.readCigarString);
                    this.out.writeInt(record.mateAlignmentStart);
                    this.out.writeInt(record.mateReferenceIndex);
                    this.out.writeBoolean(record.mateNegStrandFlag);
                    this.out.writeBoolean(record.mateUnmappedFlag);
                    this.out.writeUTF(record.mateCigarString != null ? record.mateCigarString : "");
                    this.out.writeBoolean(record.firstOfPairFlag);
                    this.out.writeLong(record.recordNumber);
                }
                catch (IOException e) {
                    throw new SAMException("Error spilling PairInfo to disk", e);
                }
            }

            @Override
            public Map.Entry<String, PairEndInfo> decode() {
                try {
                    String key = this.in.readUTF();
                    int readAlignmentStart = this.in.readInt();
                    int readReferenceIndex = this.in.readInt();
                    boolean readNegStrandFlag = this.in.readBoolean();
                    boolean readUnmappedFlag = this.in.readBoolean();
                    String readCigarString = this.in.readUTF();
                    int mateAlignmentStart = this.in.readInt();
                    int mateReferenceIndex = this.in.readInt();
                    boolean mateNegStrandFlag = this.in.readBoolean();
                    boolean mateUnmappedFlag = this.in.readBoolean();
                    String mcs = this.in.readUTF();
                    String mateCigarString = !mcs.isEmpty() ? mcs : null;
                    boolean firstOfPairFlag = this.in.readBoolean();
                    long recordNumber = this.in.readLong();
                    PairEndInfo rec = new PairEndInfo(readAlignmentStart, readReferenceIndex, readNegStrandFlag, readUnmappedFlag, readCigarString, mateAlignmentStart, mateReferenceIndex, mateNegStrandFlag, mateUnmappedFlag, mateCigarString, firstOfPairFlag, recordNumber);
                    return new AbstractMap.SimpleEntry<String, PairEndInfo>(key, rec);
                }
                catch (IOException e) {
                    throw new SAMException("Error reading PairInfo from disk", e);
                }
            }
        }
    }

    static interface PairEndInfoMap
    extends Iterable<Map.Entry<String, PairEndInfo>> {
        public void put(int var1, String var2, PairEndInfo var3);

        public PairEndInfo remove(int var1, String var2);

        @Override
        public CloseableIterator<Map.Entry<String, PairEndInfo>> iterator();
    }

    private static class MaxOutputExceededException
    extends SAMException {
        MaxOutputExceededException() {
            super("maxVerboseOutput exceeded.");
        }
    }

    private static class PairEndInfo {
        private final int readAlignmentStart;
        private final int readReferenceIndex;
        private final boolean readNegStrandFlag;
        private final boolean readUnmappedFlag;
        private final String readCigarString;
        private final int mateAlignmentStart;
        private final int mateReferenceIndex;
        private final boolean mateNegStrandFlag;
        private final boolean mateUnmappedFlag;
        private final String mateCigarString;
        private final boolean firstOfPairFlag;
        private final long recordNumber;

        public PairEndInfo(SAMRecord record, long recordNumber) {
            this.recordNumber = recordNumber;
            this.readAlignmentStart = record.getAlignmentStart();
            this.readNegStrandFlag = record.getReadNegativeStrandFlag();
            this.readReferenceIndex = record.getReferenceIndex();
            this.readUnmappedFlag = record.getReadUnmappedFlag();
            this.readCigarString = record.getCigarString();
            this.mateAlignmentStart = record.getMateAlignmentStart();
            this.mateNegStrandFlag = record.getMateNegativeStrandFlag();
            this.mateReferenceIndex = record.getMateReferenceIndex();
            this.mateUnmappedFlag = record.getMateUnmappedFlag();
            Object mcs = record.getAttribute(SAMTag.MC.name());
            this.mateCigarString = mcs != null ? (String)mcs : null;
            this.firstOfPairFlag = record.getFirstOfPairFlag();
        }

        private PairEndInfo(int readAlignmentStart, int readReferenceIndex, boolean readNegStrandFlag, boolean readUnmappedFlag, String readCigarString, int mateAlignmentStart, int mateReferenceIndex, boolean mateNegStrandFlag, boolean mateUnmappedFlag, String mateCigarString, boolean firstOfPairFlag, long recordNumber) {
            this.readAlignmentStart = readAlignmentStart;
            this.readReferenceIndex = readReferenceIndex;
            this.readNegStrandFlag = readNegStrandFlag;
            this.readUnmappedFlag = readUnmappedFlag;
            this.readCigarString = readCigarString;
            this.mateAlignmentStart = mateAlignmentStart;
            this.mateReferenceIndex = mateReferenceIndex;
            this.mateNegStrandFlag = mateNegStrandFlag;
            this.mateUnmappedFlag = mateUnmappedFlag;
            this.mateCigarString = mateCigarString;
            this.firstOfPairFlag = firstOfPairFlag;
            this.recordNumber = recordNumber;
        }

        public List<SAMValidationError> validateMates(PairEndInfo mate, String readName) {
            ArrayList<SAMValidationError> errors = new ArrayList<SAMValidationError>();
            this.validateMateFields(this, mate, readName, errors);
            this.validateMateFields(mate, this, readName, errors);
            if (this.firstOfPairFlag == mate.firstOfPairFlag) {
                String whichEnd = this.firstOfPairFlag ? "first" : "second";
                errors.add(new SAMValidationError(SAMValidationError.Type.MATES_ARE_SAME_END, "Both mates are marked as " + whichEnd + " of pair", readName, this.recordNumber));
            }
            return errors;
        }

        private void validateMateFields(PairEndInfo end1, PairEndInfo end2, String readName, List<SAMValidationError> errors) {
            if (end1.mateAlignmentStart != end2.readAlignmentStart) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_MATE_ALIGNMENT_START, "Mate alignment does not match alignment start of mate", readName, end1.recordNumber));
            }
            if (end1.mateNegStrandFlag != end2.readNegStrandFlag) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_FLAG_MATE_NEG_STRAND, "Mate negative strand flag does not match read negative strand flag of mate", readName, end1.recordNumber));
            }
            if (end1.mateReferenceIndex != end2.readReferenceIndex) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_MATE_REF_INDEX, "Mate reference index (MRNM) does not match reference index of mate", readName, end1.recordNumber));
            }
            if (end1.mateUnmappedFlag != end2.readUnmappedFlag) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_FLAG_MATE_UNMAPPED, "Mate unmapped flag does not match read unmapped flag of mate", readName, end1.recordNumber));
            }
            if (end1.mateCigarString != null && !end1.mateCigarString.equals(end2.readCigarString)) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_MATE_CIGAR_STRING, "Mate CIGAR string does not match CIGAR string of mate", readName, end1.recordNumber));
            }
        }
    }

    public static class ValidationMetrics
    extends MetricBase {
    }
}

