# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2012 Canonical
# Author: Thomi Richards
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#

"""Functional tests for pyruntest.py script."""

from os import remove
import os.path
from shutil import rmtree
import subprocess
from tempfile import mkdtemp
from testtools import TestCase
from testtools.matchers import Equals, NotEquals, Contains, Not
from textwrap import dedent


class FunctionalRunTests(TestCase):

    """Functional tests for running the pyruntest script."""

    def create_empty_test_file(self):
        open('test_empty.py', 'w').write(dedent('''\

            from testtools import TestCase

            class SimpleTests(TestCase):

                def test_simple(self):
                    self.assertEqual(1,1)

            '''))
        self.addCleanup(remove, 'test_empty.py')

    def create_test_file_with_logging(self):
        open('test_logging.py', 'w').write(dedent('''\

            from testtools import TestCase
            import logging

            class SimpleTests(TestCase):

                def test_simple(self):
                    logging.error("logging.error")
                    self.assertEqual(1,1)

            '''))
        self.addCleanup(remove, 'test_logging.py')

    def create_test_package(self):
        location = mkdtemp()
        self.addCleanup(rmtree, location)
        open(os.path.join(location, '__init__.py'), 'w').write('#')
        os.mkdir(os.path.join(location, 'tests'))
        open(os.path.join(location, 'tests', '__init__.py'), 'w').write('#')
        open(os.path.join(location, 'tests', 'test_empty.py'), 'w').write(
            dedent('''\

            from testtools import TestCase


            class SimpleTests(TestCase):

                def test_simple(self):
                    self.assertEqual(1,1)

            '''))
        return location

    def run_script(self, *args, **kwargs):
        arguments = ['./pyruntest']
        arguments.extend(args)

        if 'cwd' in kwargs:
            arguments[0] = os.path.abspath('./pyruntest')

        process = subprocess.Popen(arguments,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            **kwargs
            )
        stdout, stderr = process.communicate()
        return (stdout, stderr, process.poll())

    def test_can_run_script(self):
        self.run_script("-h")

    def test_must_specify_suite(self):
        out, err, code = self.run_script()
        self.assertThat(err, NotEquals(0))

    def test_get_output(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('test_empty')

        self.assertThat(out, Contains('Ran 1 test in'))
        self.assertThat(code, Equals(0))

    def test_output_can_be_saved_to_file(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-fo', 'test_output', 'test_empty')
        self.addCleanup(remove, 'test_output')

        self.assertThat(code, Equals(0))
        self.assertThat(out, Equals(''))
        self.assertThat(err, Equals(''))
        self.assertThat(open('test_output').read(), Contains('Ran 1 test in'))

    def test_output_can_be_saved_to_xml_file(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-fo', 'test_output', '-f', 'xml', 'test_empty')
        self.addCleanup(remove, 'test_output')

        self.assertThat(code, Equals(0))
        self.assertThat(out, Equals(''))
        self.assertThat(err, Equals(''))
        self.assertThat(open('test_output').read(), Contains('<testcase classname="test_empty.SimpleTests" name="test_simple"'))

    def test_no_logging_in_xml_file(self):
        self.create_test_file_with_logging()
        out, err, code = self.run_script('-fo', 'test_output', '-f', 'xml',
            'test_logging')
        self.addCleanup(remove, 'test_output')

        self.assertThat(code, Equals(0))
        self.assertThat(out, Contains('logging.error'))
        self.assertThat(err, Equals(''))
        self.assertThat(open('test_output').read(), Contains('<testcase classname="test_logging.SimpleTests" name="test_simple"'))
        self.assertThat(open('test_output').read(), Not(Contains('logging.error')))


    def test_generates_text_coverage_by_default(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', 'test_empty')
        self.addCleanup(remove, 'coverage.txt')

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertTrue(os.path.exists('coverage.txt'))

    def test_can_rename_text_coverage(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', '-co', 'my_coverage.txt', 'test_empty')
        self.addCleanup(remove, 'my_coverage.txt')

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertTrue(os.path.exists('my_coverage.txt'))

    def test_text_coverage_appends_missing_extension(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', '-co', 'my_coverage', 'test_empty')
        self.addCleanup(remove, 'my_coverage.txt')

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertTrue(os.path.exists('my_coverage.txt'))

    def test_can_generate_xml_coverage(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', '-cf', 'xml', 'test_empty')
        self.addCleanup(remove, 'coverage.xml')

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertTrue(os.path.exists('coverage.xml'))
        self.assertThat(open('coverage.xml').read(), Contains("""<!DOCTYPE coverage
  SYSTEM 'http://cobertura.sourceforge.net/xml/coverage-03.dtd'>"""))

    def test_can_rename_xml_coverage(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', '-cf', 'xml', '-co', 'my_coverage.xml', 'test_empty')
        self.addCleanup(remove, 'my_coverage.xml')

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertTrue(os.path.exists('my_coverage.xml'))

    def test_xml_coverage_appends_missing_extension(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', '-cf', 'xml', '-co', 'my_coverage', 'test_empty')
        self.addCleanup(remove, 'my_coverage.xml')

        self.assertThat(err, Equals(''))
        self.assertThat(code, Equals(0))
        self.assertTrue(os.path.exists('my_coverage.xml'))

    def test_can_generate_html_coverage(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', '-cf', 'html', 'test_empty')
        self.addCleanup(rmtree, 'coverage')

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertTrue(os.path.exists('coverage'))
        self.assertTrue(os.path.isdir('coverage'))
        self.assertTrue(os.path.exists('coverage/index.html'))

    def test_can_exclude_file_pattern(self):
        self.create_empty_test_file()
        out, err, code = self.run_script('-c', '-cf', 'xml', 'test_empty')

        self.assertThat(open('coverage.xml').read(), Contains('/usr/'))
        remove('coverage.xml')
        out, err, code = self.run_script('-c', '-cf', 'xml', '-ce', '/usr*', 'test_empty')
        self.assertThat(open('coverage.xml').read(), Not(Contains('/usr/')))

    def test_can_run_from_inside_package(self):
        location = self.create_test_package()
        out, err, code = self.run_script('tests', cwd=location)

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertThat(out, Contains('Ran 1 test in'))

    def test_can_run_from_outside_package(self):
        location = self.create_test_package()
        pkg_name = location.split(os.path.sep)[-1]
        parent_dir = os.path.sep.join(location.split(os.path.sep)[:-1])
        out, err, code = self.run_script(pkg_name, cwd=parent_dir)

        self.assertThat(code, Equals(0))
        self.assertThat(err, Equals(''))
        self.assertThat(out, Contains('Ran 1 test in'))
