/* This file is part of KDevelop
    Copyright (C) 2002,2003 Roberto Raggi <roberto@kdevelop.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef DRIVER_H
#define DRIVER_H

#include "ast.h"

#include <tqpair.h>
#include <tqvaluestack.h>
#include <tqstringlist.h>
#include <tqmap.h>

class Lexer;
class Parser;

class Problem
{
public:
    enum
    {
	Level_Error = 0,
	Level_Warning,
	Level_Todo,
	Level_Fixme
    };

public:
    Problem() {}
    Problem( const Problem& source )
	: m_text( source.m_text ), m_line( source.m_line ),
	  m_column( source.m_column ), m_level( source.m_level ) {}
    Problem( const TQString& text, int line, int column, int level=Level_Error )
	: m_text( text ), m_line( line ), m_column( column ), m_level(level) {}

    Problem& operator = ( const Problem& source )
    {
	m_text = source.m_text;
	m_line = source.m_line;
	m_column = source.m_column;
	m_level = source.m_level;
	return( *this );
    }

    bool operator == ( const Problem& p ) const
    {
	return m_text == p.m_text && m_line == p.m_line && m_column == p.m_column && m_level == p.m_level;
    }

    TQString text() const { return m_text; }
    int line() const { return m_line; }
    int column() const { return m_column; }
    int level() const { return m_level; }

private:
    TQString m_text;
    int m_line;
    int m_column;
    int m_level;
};

enum
{
    Dep_Global,
    Dep_Local
};

typedef TQPair<TQString, int> Dependence;

class Macro
{
public:
    typedef TQString Argument;

public:
    Macro( bool hasArguments = false ): m_hasArguments( hasArguments ) {}
    Macro( const TQString &n, const TQString &b ) : m_name( n ), m_body( b ), m_hasArguments( false ) {}

    Macro( const Macro& source )
	: m_name( source.m_name),
          m_fileName( source.m_fileName ),
	  m_body( source.m_body ),
	  m_hasArguments( source.m_hasArguments ),
	  m_argumentList( source.m_argumentList ) {}

    Macro& operator = ( const Macro& source )
    {
	m_name = source.m_name;
	m_body = source.m_body;
        m_fileName = source.m_fileName;
	m_hasArguments = source.m_hasArguments;
	m_argumentList = source.m_argumentList;
	return *this;
    }

    bool operator == ( const Macro& source ) const
    {
	return
	    m_name == source.m_name &&
            m_fileName == source.m_fileName &&
	    m_body == source.m_body &&
	    m_hasArguments == source.m_hasArguments &&
	    m_argumentList == source.m_argumentList;
    }

    TQString name() const { return m_name; }
    void setName( const TQString& name ) { m_name = name; }

    TQString fileName() const { return m_fileName; }
    void setFileName( const TQString& fileName ) { m_fileName = fileName; }

    TQString body() const { return m_body; }
    void setBody( const TQString& body ) { m_body = body; }

    bool hasArguments() const { return m_hasArguments; }
    void setHasArguments( bool hasArguments ) { m_hasArguments = hasArguments; }
    TQValueList<Argument> argumentList() const { return m_argumentList; }

    void clearArgumentList() { m_argumentList.clear(); m_hasArguments = false; }
    void addArgument( const Argument& argument ) { m_argumentList << argument; }
    void addArgumentList( const TQValueList<Argument>& arguments ) { m_argumentList += arguments; }

private:
    TQString m_name;
    TQString m_fileName;
    TQString m_body;
    bool m_hasArguments;
    TQValueList<Argument> m_argumentList;
};

class SourceProvider
{
public:
    SourceProvider() {}
    virtual ~SourceProvider() {}

    virtual TQString contents( const TQString& fileName ) = 0;
    virtual bool isModified( const TQString& fileName ) = 0;

private:
    SourceProvider( const SourceProvider& source );
    void operator = ( const SourceProvider& source );
};

class Driver
{
public:
    Driver();
    virtual ~Driver();

    SourceProvider* sourceProvider();
    void setSourceProvider( SourceProvider* sourceProvider );

    virtual void reset();

    virtual void parseFile( const TQString& fileName, bool onlyPreProcesss=false, bool force=false );
    virtual void fileParsed( const TQString& fileName );
    virtual void remove( const TQString& fileName );

    virtual void addDependence( const TQString& fileName, const Dependence& dep );
    virtual void addMacro( const Macro& macro );
    virtual void addProblem( const TQString& fileName, const Problem& problem );


    TQString currentFileName() const { return m_currentFileName; }
    TranslationUnitAST::Node takeTranslationUnit( const TQString& fileName );
    TranslationUnitAST* translationUnit( const TQString& fileName ) const;
    TQMap<TQString, Dependence> dependences( const TQString& fileName ) const;
    TQMap<TQString, Macro> macros() const;
    TQValueList<Problem> problems( const TQString& fileName ) const;

    bool hasMacro( const TQString& name ) const { return m_macros.contains( name ); }
    const Macro& macro( const TQString& name ) const { return m_macros[ name ]; }
    Macro& macro( const TQString& name ) { return m_macros[ name ]; }

    virtual void removeMacro( const TQString& macroName );
    virtual void removeAllMacrosInFile( const TQString& fileName );

    TQStringList includePaths() const { return m_includePaths; }
    virtual void addIncludePath( const TQString &path );

    /// @todo remove
    const TQMap<TQString, TranslationUnitAST*> &parsedUnits() const { return m_parsedUnits; }

    virtual void setResolveDependencesEnabled( bool enabled );
    bool isResolveDependencesEnabled() const { return depresolv; }

protected:
    virtual void setupLexer( Lexer* lexer );
    virtual void setupParser( Parser* parser );
    virtual void setupPreProcessor();

private:
    TQMap<TQString, Dependence>& findOrInsertDependenceList( const TQString& fileName );
    TQValueList<Problem>& findOrInsertProblemList( const TQString& fileName );
    TQString findIncludeFile( const Dependence& dep ) const;

private:
    TQString m_currentFileName;
    TQMap< TQString, TQMap<TQString, Dependence> > m_dependences;
    TQMap<TQString, Macro> m_macros;
    TQMap< TQString, TQValueList<Problem> > m_problems;
    TQMap<TQString, TranslationUnitAST*> m_parsedUnits;
    TQStringList m_includePaths;
    uint depresolv : 1;
    Lexer *lexer;
    SourceProvider* m_sourceProvider;

private:
    Driver( const Driver& source );
    void operator = ( const Driver& source );
};

#endif
