/***************************************************************************
 *   Copyright (C) 2003 by Sylvain Joyeux                                  *
 *   sylvain.joyeux@m4x.org                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "parsers.h"
#include "../apt.h"

#include <tdelocale.h>
#include <tqregexp.h>

static const TQString
  html_attribute_begin("<tr><td class=\"attname\">%1</td>\n\t<td>\n\t\t"),
  html_attribute_classed("<tr class=\"%1\"><td class=\"attname\">%2</td>\n\t<td>\n\t\t"),
  html_attribute_end("\n\t</td>\n</tr>\n");

// Converts the special chars in orig into its HTML equivalents
static TQString text2html(const TQString& orig)
{ TQString ret(orig);
  ret = ret.replace("<(?!a href)", "&lt;");
  //ret = ret.replace( TQRegExp("\n"), "<br />\n");
  return ret;
}

static void close_indent(int indent, TQString& buffer)
{
  if (buffer.isEmpty()) return;
  if (indent)
    buffer += "\n\t\t</div>";
}
static void close_item(int indent, TQString& buffer)
{
  if (buffer.isEmpty()) return;
  close_indent(indent, buffer);
  buffer += html_attribute_end;
}

static void close_table(const TQString& version, int indent, TQString& buffer)
{
  if (buffer.isEmpty()) return;
  close_item(indent, buffer);
  buffer = version + "<table class=\"version\">\n" + buffer + "</table>\n";
}

static TQString version_header_link(const KURL& url, const TQString& name)
{ return TQString("\t<a class=\"links\" href=\"" + url.htmlURL() + "\">" + name + "</a>\n"); }

namespace Parsers
{

Show::Show(const TQString& package, const TQString& installed_version, bool act)
  : m_package(package), m_installed(installed_version), m_act(false)
{
  m_act = act;
}

void Show::operator() (AptProtocol* slave, const TQString& tag, const TQString& value)
{
  // Since we want to show the version first, we should wait for it
  // before sending anything
  static TQString version;
  static TQString buffer;
  static TQString attribute;
  static int indent;
  static bool multiline = false;
  static bool first_line = false, new_paragraph = true;

  if (tag == "begin" || tag == "package" || tag == "end")
  {
    if (multiline && !new_paragraph)
      buffer += "</p>";

    if (tag == "begin")
      m_result_count = 0;
    else
    {
      m_result_count += (tag == "package");

      close_table(version, indent, buffer);
      if (!buffer.isEmpty()) *slave << buffer;
    }

    // init the state variables for the next package
    buffer = "";
    indent = 0;
    first_line = false;
    new_paragraph = true;
  }
  else if (tag == "field")
  {
    if (multiline && !new_paragraph)
      buffer += "</p>";

    attribute = value;
    if (value == "Depends" || value == "Description")
    {
      close_item(indent, buffer);
      buffer +=
      	html_attribute_begin.arg(value)
        + "&nbsp;";

      close_item(indent, buffer);
      buffer += html_attribute_begin.arg("&nbsp;");
    }
    else if (value != "Version" && value != "Package")
    {
      close_item(indent, buffer);
      buffer += html_attribute_begin.arg(value);
    }

    if (value == "Description")
    {
      multiline = true;
      new_paragraph = true;
      first_line = true;
    }
    else
    {
      multiline = false;
    }
  }
  else if (tag == "indent")
  {
    close_indent(indent, buffer);

    int new_indent = value.toInt();
    if (new_indent)
      buffer += "\n\t<div style=\"margin-left: " + value + "em\">";
    indent = new_indent;
  }
  else if (tag == "data" && attribute == "Version")
  {
    KURL action ("apt:/get");

    TQString item_id = mangle_version(value);
    if (value == m_installed)
    {
      action.setQuery("remove");
      version = TQString("<div class=\"version-header-installed\" id=\"%1\">").arg(item_id)
                + i18n("Installed version %1").arg(value);
    }
    else
    {
      action.setQuery("install");
      version = TQString("<div class=\"version-header\" id=\"%1\">").arg(item_id)
                + i18n("Version %1").arg(value);
    }

    action.addQueryItem("package", m_package);
    action.addQueryItem("version", value);

    if (m_act)
    {
      if (value == m_installed)
        version += version_header_link(action, i18n("Remove"));
      else
        version += version_header_link(action, i18n("Install"));
    }
    version += "</div>\n";

  }
  else if (tag == "data")
  {
    if (multiline)
    {
      static const TQRegExp rx_empty("^\\s*$");
      if (rx_empty.exactMatch(value))
      {
        buffer += "</p>";
        new_paragraph = true;
      }
      else if (first_line)
      {
        new_paragraph = true;
        first_line = false;
      }
      else if (new_paragraph)
      {
        buffer += "<p>";
        new_paragraph = false;
      }
    }
    buffer += text2html(TQString(value).replace(TQRegExp("(http://\\S+)"),TQString("<a href=\"\\1\">\\1</a>")));
  }
  else if (tag == "package_link")
    buffer += "<a href=\"" + slave->buildURL("show", value).htmlURL() + "\">" + value + "</a>";
}

}
