/***************************************************************************
 *   Copyright (C) 2003,2005 by David Saxton                               *
 *   david@bluehaze.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef ITEMSELECTOR_H
#define ITEMSELECTOR_H

#include <tdelistview.h>

#include <tqpixmap.h>
#include <tqstring.h>

class ProjectItem;
class TQStoredDrag;
namespace KateMDI { class ToolView; }

/**
@short Contains info about item for ItemSelector
@author David Saxton
*/
class ILVItem : public TQObject, public TDEListViewItem
{
	public:
		ILVItem( TQListView *parent, const TQString &id );
		ILVItem( TQListViewItem *parent, const TQString &id );
		
		void setProjectItem( ProjectItem * projectItem ) { m_pProjectItem = projectItem; }
		ProjectItem * projectItem() const { return m_pProjectItem; }
		
		TQString id() const { return m_id; }
	
		TQString key( int, bool ) const { return m_id; }
		/**
		 * Set whether the item can be removed from the listview by the user
		 */
		void setRemovable( bool isRemovable ) { b_isRemovable = isRemovable; }
		/**
		 * Whether the item can be removed from the listview by the user
		 */
		bool isRemovable() const { return b_isRemovable; }
	
	protected:
		TQString m_id;
		bool b_isRemovable;
		ProjectItem * m_pProjectItem;
};

/**
@short Allows selection of generic items for dragging / clicking
@author David Saxton
*/
class ItemSelector : public TDEListView
{
	TQ_OBJECT
  
	public:
		ItemSelector( TQWidget *parent, const char *name );
		~ItemSelector();
		/**
		 * Adds a listview item to the ListView
		 * @param caption The displayed text
		 * @param id A unique identification for when it is dragged or activated
		 * @param category The category it is in, eg "Integrated Circuits
		 * @param icon The icon to be displayed to the left of the text
		 * @param removable Whether the user can right-click on the item and select Remove
		 */
		void addItem( const TQString & caption, const TQString & id, const TQString & category, const TQPixmap & icon = TQPixmap(), bool removable = false );
	
	public slots:
		virtual void slotContextMenuRequested( TQListViewItem *item, const TQPoint &pos, int col );
		virtual void clear();
		void slotRemoveSelectedItem();
	
	signals:
		/**
		 * Emitted when a user selects an item and removes it
		 */
		void itemRemoved( const TQString &id );
		void itemDoubleClicked( const TQString &id );
		void itemClicked( const TQString &id );
	
	protected:
		/**
		 * Sets the caption of the ListView (eg 'Components' or 'Files')
		 */
		void setListCaption( const TQString &caption );
		/**
		 * Writes the open status (folded or unfolded) of "parent" items in the view
		 * to the config file.
		 */
		void writeOpenStates();
		/**
		 * Reads the open status (folded or unfolded) of the given item. The default
		 * status for non-existant items is true.
		 */
		bool readOpenState( const TQString &id );

	private slots:
		void slotItemClicked( TQListViewItem *item );
		void slotItemDoubleClicked( TQListViewItem *item );

	private:
		/**
		 * @return a dragobject encoding the currently selected component item.
		 */
		TQDragObject * dragObject();
	
		TQStringList m_categories;
};


/**
@short Allows selection of electrical components
@author David Saxton
 */
class ComponentSelector : public ItemSelector
{
	TQ_OBJECT
  
	public:
		static ComponentSelector * self( KateMDI::ToolView * parent = 0l );
		static TQString toolViewIdentifier() { return "ComponentSelector"; }
	
	private:
		ComponentSelector( KateMDI::ToolView * parent );
		static ComponentSelector * m_pSelf;
};


/**
@short Allows selection of PIC parts (eg 'Pause')
@author David Saxton
 */
class FlowPartSelector : public ItemSelector
{
	TQ_OBJECT
  
	public:
		static FlowPartSelector * self( KateMDI::ToolView * parent = 0l );
		static TQString toolViewIdentifier() { return "FlowPartSelector"; }
	
	private:
		FlowPartSelector( KateMDI::ToolView * parent );
		static FlowPartSelector * m_pSelf;
};


/**
@author David Saxton
 */
class MechanicsSelector : public ItemSelector
{
	TQ_OBJECT
  
	public:
		static MechanicsSelector * self( KateMDI::ToolView * parent = 0l );
		static TQString toolViewIdentifier() { return "MechanicsSelector"; }
	
	private:
		MechanicsSelector( TQWidget *parent = 0L );
		static MechanicsSelector * m_pSelf;
};


#endif
