/*
  Copyright (C) 2002 Rik Hemsley (rikkus) <rik@kde.org>
  Copyright (C) 2002 Benjamin Meyer <ben-devel@meyerhome.net>
  Copyright (C) 2002 Nadeem Hasan <nhasan@kde.org>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; see the file COPYING.LIB.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
  Boston, MA 02110-1301, USA.
*/

#include <kdebug.h>
#include <kstandarddirs.h>

#include <tqfile.h>
#include <tqdir.h>

#include "cache.h"
#include "config.h"

namespace KCDDB
{
    TQString
  Cache::fileName( const TQString &category, const TQString &discid, const TQString &cacheDir )
  {
    TQDir dir( cacheDir );
    if ( !dir.exists( category ) )
      dir.mkdir( category );

    return cacheDir + "/" + category + "/" + discid;
  }

    CDInfoList
  Cache::lookup( const TQString &cddbId )
  {
    kdDebug(60010) << "Looking up " << cddbId << " in CDDB cache" << endl;

    CDInfoList infoList;
    Config c;
    c.readConfig();
    TQStringList cddbCacheDirs = c.cacheLocations();

    for (TQStringList::Iterator cddbCacheDir = cddbCacheDirs.begin();
        cddbCacheDir != cddbCacheDirs.end(); ++cddbCacheDir)
    {
      TQDir dir( *cddbCacheDir );
      TQStringList dirList = dir.entryList( TQDir::Dirs );

      TQStringList::ConstIterator it = dirList.begin();

      while ( it != dirList.end() )
      {
        TQString category( *it );
        if ( category[ 0 ] != '.' )
        {
          TQFile f( *cddbCacheDir + "/" + category + "/" + cddbId );
          if ( f.exists() && f.open(IO_ReadOnly) )
          {
              TQTextStream ts(&f);
              ts.setEncoding(TQTextStream::UnicodeUTF8);
              TQString cddbData = ts.read();
              f.close();
              CDInfo info;
              info.load(cddbData);
              info.category = category;

              infoList.append( info );
          }
        }
        ++it;
      }
    }

    return infoList;
  }

    void
  Cache::store(const CDInfoList& list)
  {
    CDInfoList::ConstIterator it=list.begin(); 
    while (it!=list.end())
    {
      CDInfo info( *it );
      store(info);
      ++it;
    }
  }

    void
  Cache::store(const CDInfo& info)
  {
    Config c;
    c.readConfig();

    TQString cacheDir = c.cacheLocations().first();
    TQDir d(cacheDir);
    if (!d.exists())
      d.mkdir(cacheDir);

    // The same entry can contain several discids (separated by a ','),
    // so we save the entry to all of them
    TQStringList discids = TQStringList::split(',', info.id);
    for (TQStringList::Iterator it = discids.begin(); it != discids.end(); ++it)
    {
      TQString cacheFile = fileName(info.category, *it, cacheDir);

      kdDebug(60010) << "Storing " << cacheFile << " in CDDB cache" << endl;

      TQFile f(cacheFile);
      if ( f.open(IO_WriteOnly) )
      {
        TQTextStream ts(&f);
        ts.setEncoding(TQTextStream::UnicodeUTF8);
        ts << info.toString();
        f.close();
      }
    }
  }
}
