/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "kftpqueueconverter.h"
#include "kftpqueue.h"

#include <tqfile.h>
#include <tqobjectlist.h>

#include <kfilterdev.h>

KFTPQueueConverter::KFTPQueueConverter(TQObject *parent, const char *name)
 : TQObject(parent, name)
{
}

void KFTPQueueConverter::importQueue(const TQString &filename)
{
  m_xml = TQDomDocument("KFTPGrabberQueue");
  KFTPQueue::Manager::self()->clearQueue();
  
  // Load from file
  TQIODevice *file = KFilterDev::deviceForFile(filename);
  m_xml.setContent(file);
  file->close();
  delete file;
  
  // Parse XML and create KFTPQueueTransfers
  TQDomNode n = m_xml.documentElement().firstChild();
  while (!n.isNull()) {
    importNode(n);
    
    n = n.nextSibling();
  }
  
  KFTPQueue::Manager::self()->doEmitUpdate();
}

void KFTPQueueConverter::exportQueue(const TQString &filename)
{
  m_xml = TQDomDocument("KFTPGrabberQueue");
  m_xml.setContent(TQString("<queue></queue>"));
  
  // Go trough all KFTPQueueTransfers and generate XML
  KFTPQueue::QueueObject *i;
  TQPtrList<KFTPQueue::QueueObject> sites = KFTPQueue::Manager::self()->topLevelObject()->getChildrenList();
  
  for (i = sites.first(); i; i = sites.next()) {
    KFTPQueue::QueueObject *t;
    TQPtrList<KFTPQueue::QueueObject> list = i->getChildrenList();
    
    for (t = list.first(); t; t = list.next())
      generateXML(static_cast<KFTPQueue::Transfer*>(t), m_xml.documentElement());
  }
  
  // Save to file
  TQIODevice *file = KFilterDev::deviceForFile(filename, "application/x-gzip");
  if (!file->open(IO_WriteOnly)) {
    tqDebug("WARNING: Unable to open xml for writing!");
    return;
  }
  
  TQTextStream fileStream(file);
  m_xml.save(fileStream, 2);
  file->flush();
  file->close();
  delete file;
}

void KFTPQueueConverter::generateXML(KFTPQueue::Transfer *transfer, TQDomNode parent)
{
  // Create the item
  TQDomElement item = m_xml.createElement("item");
  parent.appendChild(item);

  // Create text nodes
  createTextNode("source", transfer->getSourceUrl().url(), item);
  createTextNode("dest", transfer->getDestUrl().url(), item);
  createTextNode("size", TQString::number(transfer->getSize()), item);
  createTextNode("type", transfer->isDir() ? "directory" : "file", item);
  
  if (transfer->isDir() && transfer->children()) {
    // Transfer has children, add them as well
    TQDomElement tag = m_xml.createElement("children");
    item.appendChild(tag);
    
    KFTPQueue::QueueObject *i;
    TQPtrList<KFTPQueue::QueueObject> list = transfer->getChildrenList();
    for (i = list.first(); i; i = list.next()) {
      generateXML(static_cast<KFTPQueue::Transfer*>(i), tag);
    }
  }
}

void KFTPQueueConverter::importNode(TQDomNode node, TQObject *parent)
{
  // Get node data
  KURL srcUrl = KURL(getTextNode("source", node));
  KURL dstUrl = KURL(getTextNode("dest", node));
  filesize_t size = getTextNode("size", node).toULongLong();
  bool dir = getTextNode("type", node) == "directory";
  
  KFTPQueue::TransferType transType = KFTPQueue::Download;
  
  if (srcUrl.isLocalFile() && !dstUrl.isLocalFile()) {
    transType = KFTPQueue::Upload;
  } else if (!srcUrl.isLocalFile() && dstUrl.isLocalFile()) {
    transType = KFTPQueue::Download;
  } else if (!srcUrl.isLocalFile() && !dstUrl.isLocalFile()) {
    transType = KFTPQueue::FXP;
  }
  
  // Create new transfer
  if (!parent)
    parent = KFTPQueue::Manager::self()->topLevelObject();
    
  KFTPQueue::Transfer *transfer = 0L;
  if (dir)
    transfer = new KFTPQueue::TransferDir(parent);
  else
    transfer = new KFTPQueue::TransferFile(parent);
  
  transfer->setSourceUrl(srcUrl);
  transfer->setDestUrl(dstUrl);
  transfer->addSize(dir ? 0 : size);
  transfer->setTransferType(transType);
  
  if (parent == KFTPQueue::Manager::self()->topLevelObject()) {
    KFTPQueue::Manager::self()->insertTransfer(transfer);
  } else {
    transfer->setId(KFTPQueue::Manager::self()->m_lastTQID++);
    emit KFTPQueue::Manager::self()->newTransfer(transfer);
  }
  
  TQDomNodeList tagNodes = node.toElement().elementsByTagName("children");
  if (dir && tagNodes.length() > 0) {
    // Import all child nodes
    TQDomNode n = node.firstChild();
    while (!n.isNull()) {
      if (n.toElement().tagName() == "children") {
        n = n.firstChild();
        break;
      }
      
      n = n.nextSibling();
    }
    
    while (!n.isNull()) {
      importNode(n, transfer);
      
      n = n.nextSibling();
    }
  }
}

void KFTPQueueConverter::createTextNode(const TQString &name, const TQString &value, TQDomNode parent)
{
  TQDomElement tag = m_xml.createElement(name);
  parent.appendChild(tag);
  
  TQDomText textNode = m_xml.createTextNode(value);
  tag.appendChild(textNode);
}

TQString KFTPQueueConverter::getTextNode(const TQString &name, TQDomNode parent)
{
  TQDomNodeList tagNodes = parent.toElement().elementsByTagName(name);
  
  if (tagNodes.length() > 0) {
    TQString prop = tagNodes.item(0).toElement().text();
    prop.stripWhiteSpace();

    return prop;
  } else {
    return TQString::null;
  }
}

#include "kftpqueueconverter.moc"
