/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; fill-column: 100 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#pragma once

#include <curl/curl.h>

#if defined(LINUX) && !defined(SYSTEM_CURL)
#include <com/sun/star/uno/RuntimeException.hpp>

#include <unistd.h>

static char const* GetCABundleFile()
{
    // try system ones first; inspired by:
    // https://www.happyassassin.net/posts/2015/01/12/a-note-about-ssltls-trusted-certificate-stores-and-platforms/
    auto const candidates = {
        "/etc/pki/tls/certs/ca-bundle.crt",
        "/etc/pki/tls/certs/ca-bundle.trust.crt",
        "/etc/ssl/certs/ca-certificates.crt",
        "/var/lib/ca-certificates/ca-bundle.pem",
    };
    for (char const* const candidate : candidates)
    {
        if (access(candidate, R_OK) == 0)
        {
            return candidate;
        }
    }

    throw css::uno::RuntimeException("no OpenSSL CA certificate bundle found");
}

static void InitCurl_easy(CURL* const pCURL)
{
    char const* const path = GetCABundleFile();
    auto rc = curl_easy_setopt(pCURL, CURLOPT_CAINFO, path);
    if (rc != CURLE_OK) // only if OOM?
    {
        throw css::uno::RuntimeException("CURLOPT_CAINFO failed");
    }
}

#else

static void InitCurl_easy(CURL* const)
{
    // these don't use OpenSSL so CAs work out of the box
}

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab cinoptions=b1,g0,N-s cinkeys+=0=break: */
