// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_FAVICON_CORE_FAVICON_DRIVER_OBSERVER_H_
#define COMPONENTS_FAVICON_CORE_FAVICON_DRIVER_OBSERVER_H_

#include "base/macros.h"

namespace gfx {
class Image;
}

class GURL;

namespace favicon {

class FaviconDriver;

// An observer implemented by classes which are interested in event from
// FaviconDriver.
class FaviconDriverObserver {
 public:
  // The type of the icon in the OnFaviconUpdated() notification.
  enum NotificationIconType {
    // Multi-resolution 16x16 (gfx::kFaviconSize) device independant pixel
    // favicon of type favicon_base::IconType::kFavicon. If the page does not
    // provide a 16x16 DIP icon, the icon is generated by resizing another icon.
    NON_TOUCH_16_DIP,
    // Largest icon specified by the page which is of type
    // favicon_base::IconType::kFavicon.
    NON_TOUCH_LARGEST,
    // Largest icon specified by the page which is of type
    // favicon_base::IconType::kTouchIcon or of type
    // favicon_base::IconType::kTouchPrecomposedIcon.
    TOUCH_LARGEST
  };

  FaviconDriverObserver() {}
  virtual ~FaviconDriverObserver() {}

  // Called when either:
  // 1) Chrome determines the best icon for the page for
  //    |notification_icon_type|.
  //    Not called if the site does not provide a custom icon and the best icon
  //    for the page is the default one provided by Chrome.
  // 2) The site changes its icon via Javascript.
  // |icon_url_changed| is false if OnFaviconAvailable() was already called for
  // |notification_icon_type| for the current page URL and |icon_url| is the
  // same as for the previous notification for |notification_icon_type|.
  // Example:
  // Page: www.google.com
  // Icon: www.google.com/favicon.ico
  // Data for www.google.com/favicon.ico in the database has expired.
  // i) OnFaviconUpdated() is called with |icon_url_changed| == true to notify
  //    that a favicon was found in the history database.
  // ii) As the history data has expired, the icon at www.google.com/favicon.ico
  //     is redownloaded and stored into the database. OnFaviconUpdated() is
  //     called with |icon_url_changed| == false to notify that the icon in the
  //     history database MAY have changed visually.
  //
  // |icon_url| is empty for the cases where there is no actual icon (e.g. the
  // page removed the favicon via Javascript).
  virtual void OnFaviconUpdated(FaviconDriver* favicon_driver,
                                NotificationIconType notification_icon_type,
                                const GURL& icon_url,
                                bool icon_url_changed,
                                const gfx::Image& image) = 0;

 private:
  DISALLOW_COPY_AND_ASSIGN(FaviconDriverObserver);
};

}  // namespace favicon

#endif  // COMPONENTS_FAVICON_CORE_FAVICON_DRIVER_OBSERVER_H_
