##! Events and records generated by the Redis analyzer.

module Redis;

export {
	## The Redis SET command.
	type SetCommand: record {
		## The key the SET command is setting.
		key: string &log;
		## The value the SET command is setting key to.
		value: string &log;
		## If NX is set -- only set the key if it does not exist.
		nx: bool;
		## If XX is set -- only set the key if it already exists.
		xx: bool;
		## If GET is set -- return the old string stored at key.
		get: bool;
		## EX option -- set the specified expire time, in seconds.
		ex: count &optional;
		## PX option -- set the specified expire time, in milliseconds.
		px: count &optional;
		## EXAT option-- set the specified Unix time at which the key will
		## expire, in seconds.
		exat: count &optional;
		## PXAT option -- set the specified Unix time at which the key will
		## expire, in milliseconds.
		pxat: count &optional;
		## If KEEPTTL is set -- retain the time to live associated with the key.
		keep_ttl: bool;
	};

	## The Redis AUTH command.
	type AuthCommand: record {
		## The username getting authenticated.
		username: string &optional;
		## The password authenticated with.
		password: string;
	};

	## The Redis HELLO command (handshake).
	type HelloCommand: record {
		## The sent requested RESP version, such as "2" or "3"
		requested_resp_version: string &optional;
	};

	## A generic Redis command from the client.
	type Command: record {
		## The raw command, exactly as parsed
		raw: vector of string;
		## The first element of the command. Some commands are two strings, meaning
		## this is inaccurate for those cases.
		name: string &log;
		## The key, if this command is known to have a key
		key: string &log &optional;
		## The value, if this command is known to have a value
		value: string &log &optional;
		## The command in an enum if it was known
		known: RedisCommand &optional;
	};

	## A generic Redis reply from the client.
	type ReplyData: record {
		## The RESP3 attributes applied to this, if any
		attributes: string &optional;
		## The string version of the reply data
		value: string &log;
		## The minimum RESP version that supports this reply type
		min_protocol_version: count;
	};
}

## Generated for Redis SET commands sent to the Redis server.
##
## c: The connection.
##
## command: The SET command sent to the server and its data.
global set_command: event(c: connection, command: SetCommand);

## Generated for Redis GET commands sent to the Redis server.
##
## c: The connection.
##
## command: The GET command sent to the server and its data.
global get_command: event(c: connection, key: string);

## Generated for Redis AUTH commands sent to the Redis server.
##
## c: The connection.
##
## command: The AUTH command sent to the server and its data.
global auth_command: event(c: connection, command: AuthCommand);

## Generated for Redis HELLO commands sent to the Redis server.
##
## c: The connection.
##
## command: The HELLO command sent to the server and its data.
global hello_command: event(c: connection, command: HelloCommand);

## Generated for every command sent by the client to the Redis server.
##
## c: The connection.
##
## cmd: The command sent to the server.
global command: event(c: connection, cmd: Command);

## Generated for every successful response sent by the Redis server to the
## client. For RESP2, this includes "push" messages, which are out of band.
## These will also raise a server_push event. RESP3 push messages will only
## raise a server_push event.
##
## c: The connection.
##
## data: The server data sent to the client.
##
## .. zeek:see:: Redis::server_push
global reply: event(c: connection, data: ReplyData);

## Generated for every error response sent by the Redis server to the
## client.
##
## c: The connection.
##
## data: The server data sent to the client.
global error: event(c: connection, data: ReplyData);

## Generated for out-of-band data, outside of the request-response
## model.
##
## c: The connection.
##
## data: The server data sent to the client.
global server_push: event(c: connection, data: ReplyData);
