/*
 * Decompiled with CFR 0.152.
 */
package com.google.openlocationcode;

import java.util.Objects;

public final class OpenLocationCode {
    public static final int CODE_PRECISION_NORMAL = 10;
    public static final String CODE_ALPHABET = "23456789CFGHJMPQRVWX";
    public static final char SEPARATOR = '+';
    public static final char PADDING_CHARACTER = '0';
    private static final int SEPARATOR_POSITION = 8;
    public static final int MAX_DIGIT_COUNT = 15;
    private static final int PAIR_CODE_LENGTH = 10;
    private static final int GRID_CODE_LENGTH = 5;
    private static final int ENCODING_BASE = "23456789CFGHJMPQRVWX".length();
    private static final long LATITUDE_MAX = 90L;
    private static final long LONGITUDE_MAX = 180L;
    private static final int GRID_COLUMNS = 4;
    private static final int GRID_ROWS = 5;
    private static final long LAT_INTEGER_MULTIPLIER = 25000000L;
    private static final long LNG_INTEGER_MULTIPLIER = 0x7D0000L;
    private static final long LAT_MSP_VALUE = 25000000L * (long)ENCODING_BASE * (long)ENCODING_BASE;
    private static final long LNG_MSP_VALUE = 0x7D0000L * (long)ENCODING_BASE * (long)ENCODING_BASE;
    private final String code;

    public OpenLocationCode(String code) {
        if (!OpenLocationCode.isValidCode(code.toUpperCase())) {
            throw new IllegalArgumentException("The provided code '" + code + "' is not a valid Open Location Code.");
        }
        this.code = code.toUpperCase();
    }

    public OpenLocationCode(double latitude, double longitude, int codeLength) {
        int i;
        codeLength = Math.min(codeLength, 15);
        if (codeLength < 10 && codeLength % 2 == 1 || codeLength < 4) {
            throw new IllegalArgumentException("Illegal code length " + codeLength);
        }
        latitude = OpenLocationCode.clipLatitude(latitude);
        longitude = OpenLocationCode.normalizeLongitude(longitude);
        if (latitude == 90.0) {
            latitude -= 0.9 * OpenLocationCode.computeLatitudePrecision(codeLength);
        }
        StringBuilder revCodeBuilder = new StringBuilder();
        long latVal = (long)((double)Math.round((latitude + 90.0) * 2.5E7 * 1000000.0) / 1000000.0);
        long lngVal = (long)((double)Math.round((longitude + 180.0) * 8192000.0 * 1000000.0) / 1000000.0);
        if (codeLength > 10) {
            for (i = 0; i < 5; ++i) {
                long latDigit = latVal % 5L;
                long lngDigit = lngVal % 4L;
                int ndx = (int)(latDigit * 4L + lngDigit);
                revCodeBuilder.append(CODE_ALPHABET.charAt(ndx));
                latVal /= 5L;
                lngVal /= 4L;
            }
        } else {
            latVal = (long)((double)latVal / Math.pow(5.0, 5.0));
            lngVal = (long)((double)lngVal / Math.pow(4.0, 5.0));
        }
        for (i = 0; i < 5; ++i) {
            revCodeBuilder.append(CODE_ALPHABET.charAt((int)(lngVal % (long)ENCODING_BASE)));
            revCodeBuilder.append(CODE_ALPHABET.charAt((int)(latVal % (long)ENCODING_BASE)));
            latVal /= (long)ENCODING_BASE;
            lngVal /= (long)ENCODING_BASE;
            if (i != 0) continue;
            revCodeBuilder.append('+');
        }
        StringBuilder codeBuilder = revCodeBuilder.reverse();
        if (codeLength < 8) {
            for (int i2 = codeLength; i2 < 8; ++i2) {
                codeBuilder.setCharAt(i2, '0');
            }
        }
        this.code = codeBuilder.subSequence(0, Math.max(9, codeLength + 1)).toString();
    }

    public OpenLocationCode(double latitude, double longitude) {
        this(latitude, longitude, 10);
    }

    public String getCode() {
        return this.code;
    }

    public static String encode(double latitude, double longitude) {
        return new OpenLocationCode(latitude, longitude).getCode();
    }

    public static String encode(double latitude, double longitude, int codeLength) {
        return new OpenLocationCode(latitude, longitude, codeLength).getCode();
    }

    public CodeArea decode() {
        int i;
        if (!OpenLocationCode.isFullCode(this.code)) {
            throw new IllegalStateException("Method decode() could only be called on valid full codes, code was " + this.code + ".");
        }
        String clean = this.code.replace(String.valueOf('+'), "").replace(String.valueOf('0'), "");
        long latVal = -2250000000L;
        long lngVal = -1474560000L;
        long latPlaceVal = LAT_MSP_VALUE;
        long lngPlaceVal = LNG_MSP_VALUE;
        for (i = 0; i < Math.min(clean.length(), 10); i += 2) {
            latVal += (long)CODE_ALPHABET.indexOf(clean.charAt(i)) * (latPlaceVal /= (long)ENCODING_BASE);
            lngVal += (long)CODE_ALPHABET.indexOf(clean.charAt(i + 1)) * (lngPlaceVal /= (long)ENCODING_BASE);
        }
        for (i = 10; i < Math.min(clean.length(), 15); ++i) {
            int digit = CODE_ALPHABET.indexOf(clean.charAt(i));
            int row = digit / 4;
            int col = digit % 4;
            latVal += (long)row * (latPlaceVal /= 5L);
            lngVal += (long)col * (lngPlaceVal /= 4L);
        }
        double latitudeLo = (double)latVal / 2.5E7;
        double longitudeLo = (double)lngVal / 8192000.0;
        double latitudeHi = (double)(latVal + latPlaceVal) / 2.5E7;
        double longitudeHi = (double)(lngVal + lngPlaceVal) / 8192000.0;
        return new CodeArea(latitudeLo, longitudeLo, latitudeHi, longitudeHi, Math.min(clean.length(), 15));
    }

    public static CodeArea decode(String code) throws IllegalArgumentException {
        return new OpenLocationCode(code).decode();
    }

    public boolean isFull() {
        return this.code.indexOf(43) == 8;
    }

    public static boolean isFull(String code) throws IllegalArgumentException {
        return new OpenLocationCode(code).isFull();
    }

    public boolean isShort() {
        return this.code.indexOf(43) >= 0 && this.code.indexOf(43) < 8;
    }

    public static boolean isShort(String code) throws IllegalArgumentException {
        return new OpenLocationCode(code).isShort();
    }

    private boolean isPadded() {
        return this.code.indexOf(48) >= 0;
    }

    public static boolean isPadded(String code) throws IllegalArgumentException {
        return new OpenLocationCode(code).isPadded();
    }

    public OpenLocationCode shorten(double referenceLatitude, double referenceLongitude) {
        if (!this.isFull()) {
            throw new IllegalStateException("shorten() method could only be called on a full code.");
        }
        if (this.isPadded()) {
            throw new IllegalStateException("shorten() method can not be called on a padded code.");
        }
        CodeArea codeArea = this.decode();
        double range = Math.max(Math.abs(referenceLatitude - codeArea.getCenterLatitude()), Math.abs(referenceLongitude - codeArea.getCenterLongitude()));
        for (int i = 4; i >= 1; --i) {
            if (!(range < OpenLocationCode.computeLatitudePrecision(i * 2) * 0.3)) continue;
            return new OpenLocationCode(this.code.substring(i * 2));
        }
        throw new IllegalArgumentException("Reference location is too far from the Open Location Code center.");
    }

    public OpenLocationCode recover(double referenceLatitude, double referenceLongitude) {
        if (this.isFull()) {
            return this;
        }
        referenceLatitude = OpenLocationCode.clipLatitude(referenceLatitude);
        referenceLongitude = OpenLocationCode.normalizeLongitude(referenceLongitude);
        int digitsToRecover = 8 - this.code.indexOf(43);
        double prefixPrecision = Math.pow(ENCODING_BASE, 2 - digitsToRecover / 2);
        String recoveredPrefix = new OpenLocationCode(referenceLatitude, referenceLongitude).getCode().substring(0, digitsToRecover);
        OpenLocationCode recovered = new OpenLocationCode(recoveredPrefix + this.code);
        CodeArea recoveredCodeArea = recovered.decode();
        double recoveredLatitude = recoveredCodeArea.getCenterLatitude();
        double recoveredLongitude = recoveredCodeArea.getCenterLongitude();
        double latitudeDiff = recoveredLatitude - referenceLatitude;
        if (latitudeDiff > prefixPrecision / 2.0 && recoveredLatitude - prefixPrecision > -90.0) {
            recoveredLatitude -= prefixPrecision;
        } else if (latitudeDiff < -prefixPrecision / 2.0 && recoveredLatitude + prefixPrecision < 90.0) {
            recoveredLatitude += prefixPrecision;
        }
        double longitudeDiff = recoveredCodeArea.getCenterLongitude() - referenceLongitude;
        if (longitudeDiff > prefixPrecision / 2.0) {
            recoveredLongitude -= prefixPrecision;
        } else if (longitudeDiff < -prefixPrecision / 2.0) {
            recoveredLongitude += prefixPrecision;
        }
        return new OpenLocationCode(recoveredLatitude, recoveredLongitude, recovered.getCode().length() - 1);
    }

    public boolean contains(double latitude, double longitude) {
        CodeArea codeArea = this.decode();
        return codeArea.getSouthLatitude() <= latitude && latitude < codeArea.getNorthLatitude() && codeArea.getWestLongitude() <= longitude && longitude < codeArea.getEastLongitude();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        OpenLocationCode that = (OpenLocationCode)o;
        return Objects.equals(this.code, that.code);
    }

    public int hashCode() {
        return this.code != null ? this.code.hashCode() : 0;
    }

    public String toString() {
        return this.getCode();
    }

    public static boolean isValidCode(String code) {
        int i;
        if (code == null || code.length() < 2) {
            return false;
        }
        int separatorPosition = (code = code.toUpperCase()).indexOf(43);
        if (separatorPosition == -1) {
            return false;
        }
        if (separatorPosition != code.lastIndexOf(43)) {
            return false;
        }
        if (separatorPosition % 2 != 0 || separatorPosition > 8) {
            return false;
        }
        if (separatorPosition == 8) {
            if (CODE_ALPHABET.indexOf(code.charAt(0)) > 8) {
                return false;
            }
            if (CODE_ALPHABET.indexOf(code.charAt(1)) > 17) {
                return false;
            }
        }
        boolean paddingStarted = false;
        for (i = 0; i < separatorPosition; ++i) {
            if (CODE_ALPHABET.indexOf(code.charAt(i)) == -1 && code.charAt(i) != '0') {
                return false;
            }
            if (paddingStarted) {
                if (code.charAt(i) == '0') continue;
                return false;
            }
            if (code.charAt(i) != '0') continue;
            paddingStarted = true;
            if (separatorPosition < 8) {
                return false;
            }
            if (i == 2 || i == 4 || i == 6) continue;
            return false;
        }
        if (code.length() > separatorPosition + 1) {
            if (paddingStarted) {
                return false;
            }
            if (code.length() == separatorPosition + 2) {
                return false;
            }
            for (i = separatorPosition + 1; i < code.length(); ++i) {
                if (CODE_ALPHABET.indexOf(code.charAt(i)) != -1) continue;
                return false;
            }
        }
        return true;
    }

    public static boolean isFullCode(String code) {
        try {
            return new OpenLocationCode(code).isFull();
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean isShortCode(String code) {
        try {
            return new OpenLocationCode(code).isShort();
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    private static double clipLatitude(double latitude) {
        return Math.min(Math.max(latitude, -90.0), 90.0);
    }

    private static double normalizeLongitude(double longitude) {
        while (longitude < -180.0) {
            longitude += 360.0;
        }
        while (longitude >= 180.0) {
            longitude -= 360.0;
        }
        return longitude;
    }

    private static double computeLatitudePrecision(int codeLength) {
        if (codeLength <= 10) {
            return Math.pow(ENCODING_BASE, Math.floor(codeLength / -2 + 2));
        }
        return Math.pow(ENCODING_BASE, -3.0) / Math.pow(5.0, codeLength - 10);
    }

    public static class CodeArea {
        private final double southLatitude;
        private final double westLongitude;
        private final double northLatitude;
        private final double eastLongitude;
        private final int length;

        public CodeArea(double southLatitude, double westLongitude, double northLatitude, double eastLongitude, int length) {
            this.southLatitude = southLatitude;
            this.westLongitude = westLongitude;
            this.northLatitude = northLatitude;
            this.eastLongitude = eastLongitude;
            this.length = length;
        }

        public double getSouthLatitude() {
            return this.southLatitude;
        }

        public double getWestLongitude() {
            return this.westLongitude;
        }

        public double getLatitudeHeight() {
            return this.northLatitude - this.southLatitude;
        }

        public double getLongitudeWidth() {
            return this.eastLongitude - this.westLongitude;
        }

        public double getCenterLatitude() {
            return (this.southLatitude + this.northLatitude) / 2.0;
        }

        public double getCenterLongitude() {
            return (this.westLongitude + this.eastLongitude) / 2.0;
        }

        public double getNorthLatitude() {
            return this.northLatitude;
        }

        public double getEastLongitude() {
            return this.eastLongitude;
        }

        public int getLength() {
            return this.length;
        }
    }
}

