/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.auth;

import com.google.api.client.auth.openidconnect.IdToken;
import com.google.api.client.auth.openidconnect.IdTokenVerifier;
import com.google.api.client.googleapis.auth.oauth2.GooglePublicKeysManager;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.webtoken.JsonWebSignature;
import com.google.api.client.util.ArrayMap;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.firebase.ErrorCode;
import com.google.firebase.auth.AuthErrorCode;
import com.google.firebase.auth.FirebaseAuthException;
import com.google.firebase.auth.FirebaseToken;
import com.google.firebase.auth.FirebaseTokenVerifier;
import com.google.firebase.auth.internal.Utils;
import com.google.firebase.internal.Nullable;
import java.io.IOException;
import java.math.BigDecimal;
import java.security.GeneralSecurityException;
import java.security.PublicKey;
import java.util.List;
import java.util.Map;

final class FirebaseTokenVerifierImpl
implements FirebaseTokenVerifier {
    private static final String RS256 = "RS256";
    private static final String FIREBASE_AUDIENCE = "https://identitytoolkit.googleapis.com/google.identity.identitytoolkit.v1.IdentityToolkit";
    private final JsonFactory jsonFactory;
    private final GooglePublicKeysManager publicKeysManager;
    private final IdTokenVerifier idTokenVerifier;
    private final String method;
    private final String shortName;
    private final String articledShortName;
    private final String docUrl;
    private final AuthErrorCode invalidTokenErrorCode;
    private final AuthErrorCode expiredTokenErrorCode;
    private final String tenantId;

    private FirebaseTokenVerifierImpl(Builder builder) {
        this.jsonFactory = (JsonFactory)Preconditions.checkNotNull((Object)builder.jsonFactory);
        this.publicKeysManager = (GooglePublicKeysManager)Preconditions.checkNotNull((Object)builder.publicKeysManager);
        this.idTokenVerifier = (IdTokenVerifier)Preconditions.checkNotNull((Object)builder.idTokenVerifier);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.method) ? 1 : 0) != 0, (Object)"method name must be specified");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.shortName) ? 1 : 0) != 0, (Object)"shortName must be specified");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.docUrl) ? 1 : 0) != 0, (Object)"docUrl must be specified");
        this.method = builder.method;
        this.shortName = builder.shortName;
        this.articledShortName = this.prefixWithIndefiniteArticle(this.shortName);
        this.docUrl = builder.docUrl;
        this.invalidTokenErrorCode = (AuthErrorCode)((Object)Preconditions.checkNotNull((Object)((Object)builder.invalidTokenErrorCode)));
        this.expiredTokenErrorCode = (AuthErrorCode)((Object)Preconditions.checkNotNull((Object)((Object)builder.expiredTokenErrorCode)));
        this.tenantId = builder.tenantId;
    }

    @Override
    public FirebaseToken verifyToken(String token) throws FirebaseAuthException {
        boolean isEmulatorMode = Utils.isEmulatorMode();
        IdToken idToken = this.parse(token);
        this.checkContents(idToken, isEmulatorMode);
        if (!isEmulatorMode) {
            this.checkSignature(idToken);
        }
        FirebaseToken firebaseToken = new FirebaseToken((Map<String, Object>)idToken.getPayload());
        this.checkTenantId(firebaseToken);
        return firebaseToken;
    }

    GooglePublicKeysManager getPublicKeysManager() {
        return this.publicKeysManager;
    }

    IdTokenVerifier getIdTokenVerifier() {
        return this.idTokenVerifier;
    }

    String getMethod() {
        return this.method;
    }

    String getShortName() {
        return this.shortName;
    }

    String getArticledShortName() {
        return this.articledShortName;
    }

    String getDocUrl() {
        return this.docUrl;
    }

    private String prefixWithIndefiniteArticle(String word) {
        if ("aeiouAEIOU".indexOf(word.charAt(0)) < 0) {
            return "a " + word;
        }
        return "an " + word;
    }

    private IdToken parse(String token) throws FirebaseAuthException {
        try {
            return IdToken.parse((JsonFactory)this.jsonFactory, (String)token);
        }
        catch (IOException | IllegalArgumentException e) {
            String detailedError = String.format("Failed to parse Firebase %s. Make sure you passed a string that represents a complete and valid JWT. See %s for details on how to retrieve %s.", this.shortName, this.docUrl, this.articledShortName);
            throw this.newException(detailedError, this.invalidTokenErrorCode, e);
        }
    }

    private void checkSignature(IdToken token) throws FirebaseAuthException {
        if (!this.isSignatureValid(token)) {
            String message = String.format("Failed to verify the signature of Firebase %s. %s", this.shortName, this.getVerifyTokenMessage());
            throw this.newException(message, this.invalidTokenErrorCode);
        }
    }

    private void checkContents(IdToken idToken, boolean isEmulatorMode) throws FirebaseAuthException {
        JsonWebSignature.Header header = idToken.getHeader();
        IdToken.Payload payload = idToken.getPayload();
        long currentTimeMillis = this.idTokenVerifier.getClock().currentTimeMillis();
        String errorMessage = null;
        AuthErrorCode errorCode = this.invalidTokenErrorCode;
        if (!isEmulatorMode && header.getKeyId() == null) {
            errorMessage = this.getErrorForTokenWithoutKid(header, payload);
        } else if (!isEmulatorMode && !RS256.equals(header.getAlgorithm())) {
            errorMessage = String.format("Firebase %s has incorrect algorithm. Expected \"%s\" but got \"%s\".", this.shortName, RS256, header.getAlgorithm());
        } else if (!idToken.verifyAudience(this.idTokenVerifier.getAudience())) {
            errorMessage = String.format("Firebase %s has incorrect \"aud\" (audience) claim. Expected \"%s\" but got \"%s\". %s", this.shortName, this.joinWithComma(this.idTokenVerifier.getAudience()), this.joinWithComma(payload.getAudienceAsList()), this.getProjectIdMatchMessage());
        } else if (!idToken.verifyIssuer(this.idTokenVerifier.getIssuers())) {
            errorMessage = String.format("Firebase %s has incorrect \"iss\" (issuer) claim. Expected \"%s\" but got \"%s\". %s", this.shortName, this.joinWithComma(this.idTokenVerifier.getIssuers()), payload.getIssuer(), this.getProjectIdMatchMessage());
        } else if (payload.getSubject() == null) {
            errorMessage = String.format("Firebase %s has no \"sub\" (subject) claim.", this.shortName);
        } else if (payload.getSubject().isEmpty()) {
            errorMessage = String.format("Firebase %s has an empty string \"sub\" (subject) claim.", this.shortName);
        } else if (payload.getSubject().length() > 128) {
            errorMessage = String.format("Firebase %s has \"sub\" (subject) claim longer than 128 characters.", this.shortName);
        } else if (!idToken.verifyExpirationTime(currentTimeMillis, this.idTokenVerifier.getAcceptableTimeSkewSeconds())) {
            errorMessage = String.format("Firebase %s has expired. Get a fresh %s and try again.", this.shortName, this.shortName);
            errorCode = this.expiredTokenErrorCode;
        } else if (!idToken.verifyIssuedAtTime(currentTimeMillis, this.idTokenVerifier.getAcceptableTimeSkewSeconds())) {
            errorMessage = String.format("Firebase %s is not yet valid.", this.shortName);
        }
        if (errorMessage != null) {
            String detailedError = String.format("%s %s", errorMessage, this.getVerifyTokenMessage());
            throw this.newException(detailedError, errorCode);
        }
    }

    private FirebaseAuthException newException(String message, AuthErrorCode errorCode) {
        return this.newException(message, errorCode, null);
    }

    private FirebaseAuthException newException(String message, AuthErrorCode errorCode, Throwable cause) {
        return new FirebaseAuthException(ErrorCode.INVALID_ARGUMENT, message, cause, null, errorCode);
    }

    private String getVerifyTokenMessage() {
        return String.format("See %s for details on how to retrieve %s.", this.docUrl, this.articledShortName);
    }

    private boolean isSignatureValid(IdToken token) throws FirebaseAuthException {
        for (PublicKey key : this.fetchPublicKeys()) {
            if (!this.isSignatureValid(token, key)) continue;
            return true;
        }
        return false;
    }

    private boolean isSignatureValid(IdToken token, PublicKey key) throws FirebaseAuthException {
        try {
            return token.verifySignature(key);
        }
        catch (GeneralSecurityException e) {
            throw new FirebaseAuthException(ErrorCode.UNKNOWN, String.format("Unexpected error while verifying %s: %s", this.shortName, e.getMessage()), e, null, this.invalidTokenErrorCode);
        }
    }

    private List<PublicKey> fetchPublicKeys() throws FirebaseAuthException {
        try {
            return this.publicKeysManager.getPublicKeys();
        }
        catch (IOException | GeneralSecurityException e) {
            throw new FirebaseAuthException(ErrorCode.UNKNOWN, "Error while fetching public key certificates: " + e.getMessage(), e, null, AuthErrorCode.CERTIFICATE_FETCH_FAILED);
        }
    }

    private String getErrorForTokenWithoutKid(JsonWebSignature.Header header, IdToken.Payload payload) {
        if (this.isCustomToken(payload)) {
            return String.format("%s expects %s, but was given a custom token.", this.method, this.articledShortName);
        }
        if (this.isLegacyCustomToken(header, payload)) {
            return String.format("%s expects %s, but was given a legacy custom token.", this.method, this.articledShortName);
        }
        return String.format("Firebase %s has no \"kid\" claim.", this.shortName);
    }

    private String joinWithComma(Iterable<String> strings) {
        return Joiner.on((char)',').join(strings);
    }

    private String getProjectIdMatchMessage() {
        return String.format("Make sure the %s comes from the same Firebase project as the service account used to authenticate this SDK.", this.shortName);
    }

    private boolean isCustomToken(IdToken.Payload payload) {
        return FIREBASE_AUDIENCE.equals(payload.getAudience());
    }

    private boolean isLegacyCustomToken(JsonWebSignature.Header header, IdToken.Payload payload) {
        return "HS256".equals(header.getAlgorithm()) && new BigDecimal(0).equals(payload.get((Object)"v")) && this.containsLegacyUidField(payload);
    }

    private boolean containsLegacyUidField(IdToken.Payload payload) {
        Object dataField = payload.get((Object)"d");
        if (dataField instanceof ArrayMap) {
            return ((ArrayMap)dataField).get((Object)"uid") != null;
        }
        return false;
    }

    private void checkTenantId(FirebaseToken firebaseToken) throws FirebaseAuthException {
        String tokenTenantId = firebaseToken.getTenantId();
        if (this.tenantId != null && !this.tenantId.equals(tokenTenantId)) {
            String message = String.format("The tenant ID ('%s') of the token did not match the expected value ('%s')", Strings.nullToEmpty((String)tokenTenantId), this.tenantId);
            throw this.newException(message, AuthErrorCode.TENANT_ID_MISMATCH);
        }
    }

    static Builder builder() {
        return new Builder();
    }

    static final class Builder {
        private JsonFactory jsonFactory;
        private GooglePublicKeysManager publicKeysManager;
        private String method;
        private String shortName;
        private IdTokenVerifier idTokenVerifier;
        private String docUrl;
        private AuthErrorCode invalidTokenErrorCode;
        private AuthErrorCode expiredTokenErrorCode;
        private String tenantId;

        private Builder() {
        }

        Builder setJsonFactory(JsonFactory jsonFactory) {
            this.jsonFactory = jsonFactory;
            return this;
        }

        Builder setPublicKeysManager(GooglePublicKeysManager publicKeysManager) {
            this.publicKeysManager = publicKeysManager;
            return this;
        }

        Builder setMethod(String method) {
            this.method = method;
            return this;
        }

        Builder setShortName(String shortName) {
            this.shortName = shortName;
            return this;
        }

        Builder setIdTokenVerifier(IdTokenVerifier idTokenVerifier) {
            this.idTokenVerifier = idTokenVerifier;
            return this;
        }

        Builder setDocUrl(String docUrl) {
            this.docUrl = docUrl;
            return this;
        }

        Builder setInvalidTokenErrorCode(AuthErrorCode invalidTokenErrorCode) {
            this.invalidTokenErrorCode = invalidTokenErrorCode;
            return this;
        }

        Builder setExpiredTokenErrorCode(AuthErrorCode expiredTokenErrorCode) {
            this.expiredTokenErrorCode = expiredTokenErrorCode;
            return this;
        }

        Builder setTenantId(@Nullable String tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        FirebaseTokenVerifierImpl build() {
            return new FirebaseTokenVerifierImpl(this);
        }
    }
}

